"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.resource = props.resource;
        this.restApi = props.resource.restApi;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = (_a = authorizer) === null || _a === void 0 ? void 0 : _a.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = ((_b = authorizer) === null || _b === void 0 ? void 0 : _b.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if (((_c = authorizer) === null || _c === void 0 ? void 0 : _c.authorizationType) && authorizationTypeOption && ((_d = authorizer) === null || _d === void 0 ? void 0 : _d.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer instanceof authorizer_1.Authorizer) {
            authorizer._attachToApi(this.restApi);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.restApi.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: options.requestValidator ? options.requestValidator.requestValidatorId : undefined,
            authorizationScopes: (_e = options.authorizationScopes, (_e !== null && _e !== void 0 ? _e : defaultMethodOptions.authorizationScopes)),
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        props.resource.restApi._attachMethod(this);
        const deployment = props.resource.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, this.resource.path, 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error(`'credentialsPassthrough' and 'credentialsRole' are mutually exclusive`);
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error(`'connectionType' of VPC_LINK requires 'vpcLink' prop to be set`);
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error(`cannot set 'vpcLink' where 'connectionType' is INTERNET`);
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
//# sourceMappingURL=data:application/json;base64,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