"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
class ResourceBase extends core_1.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = `'Origin'`;
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = `'true'`;
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error(`The options "maxAge" and "disableCache" are mutually exclusive`);
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() }
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams }
            ]
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push(`#set($origin = $input.params("Origin"))`);
            template.push(`#if($origin == "") #set($origin = $input.params("origin")) #end`);
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push(`  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)`);
            template.push('#end');
            return {
                'application/json': template.join('\n')
            };
        }
    }
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error(`resourceForPath cannot be called with an empty path`);
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
class Resource extends ResourceBase {
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.restApi.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.restApi = props.parent.restApi;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.restApi.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 */
class ProxyResource extends Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZXNvdXJjZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBLHdDQUFxRztBQUNyRyxpRUFBdUU7QUFDdkUsaUNBQTJDO0FBRTNDLGlEQUFpRDtBQUNqRCxxQ0FBaUQ7QUFxSmpELE1BQXNCLFlBQWEsU0FBUSxlQUFpQjtJQVcxRCxZQUFZLEtBQWdCLEVBQUUsRUFBVTtRQUN0QyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBSEYsYUFBUSxHQUFxQyxFQUFHLENBQUM7SUFJbEUsQ0FBQztJQUVNLFdBQVcsQ0FBQyxRQUFnQixFQUFFLE9BQXlCO1FBQzVELE9BQU8sSUFBSSxRQUFRLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLEdBQUcsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUM5RSxDQUFDO0lBRU0sU0FBUyxDQUFDLFVBQWtCLEVBQUUsV0FBeUIsRUFBRSxPQUF1QjtRQUNyRixPQUFPLElBQUksZUFBTSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUUsRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztJQUM1RixDQUFDO0lBRU0sUUFBUSxDQUFDLE9BQThCO1FBQzVDLE9BQU8sSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsR0FBRyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQzNFLENBQUM7SUFFTSxnQkFBZ0IsQ0FBQyxPQUFvQjtRQUMxQyxNQUFNLE9BQU8sR0FBK0IsRUFBRyxDQUFDO1FBRWhELEVBQUU7UUFDRiwrQkFBK0I7UUFFL0IsTUFBTSxZQUFZLEdBQUcsT0FBTyxDQUFDLFlBQVksSUFBSSxXQUFJLENBQUMsZUFBZSxDQUFDO1FBQ2xFLE9BQU8sQ0FBQyw4QkFBOEIsQ0FBQyxHQUFHLElBQUksWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDO1FBRXhFLEVBQUU7UUFDRiw4QkFBOEI7UUFFOUIsSUFBSSxPQUFPLENBQUMsWUFBWSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDckMsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBRUQsSUFBSSxPQUFPLENBQUMsWUFBWSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsSUFBSSxPQUFPLENBQUMsWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDekUsTUFBTSxJQUFJLEtBQUssQ0FBQyxrRUFBa0UsT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ3JIO1FBRUQsNkVBQTZFO1FBQzdFLDREQUE0RDtRQUM1RCxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzlDLE9BQU8sQ0FBQyw2QkFBNkIsQ0FBQyxHQUFHLElBQUksYUFBYSxHQUFHLENBQUM7UUFFOUQsOERBQThEO1FBQzlELHlHQUF5RztRQUN6RyxJQUFJLGFBQWEsS0FBSyxHQUFHLEVBQUU7WUFDekIsT0FBTyxDQUFDLElBQUksR0FBRyxVQUFVLENBQUM7U0FDM0I7UUFFRCxFQUFFO1FBQ0YsK0JBQStCO1FBRS9CLElBQUksWUFBWSxHQUFHLE9BQU8sQ0FBQyxZQUFZLElBQUksV0FBSSxDQUFDLFdBQVcsQ0FBQztRQUU1RCxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDaEMsSUFBSSxZQUFZLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtnQkFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyx1REFBdUQsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDbEc7WUFFRCxZQUFZLEdBQUcsV0FBSSxDQUFDLFdBQVcsQ0FBQztTQUNqQztRQUVELE9BQU8sQ0FBQyw4QkFBOEIsQ0FBQyxHQUFHLElBQUksWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDO1FBRXhFLEVBQUU7UUFDRixtQ0FBbUM7UUFFbkMsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLEVBQUU7WUFDNUIsT0FBTyxDQUFDLGtDQUFrQyxDQUFDLEdBQUcsUUFBUSxDQUFDO1NBQ3hEO1FBRUQsRUFBRTtRQUNGLHlCQUF5QjtRQUV6QixJQUFJLGFBQWEsQ0FBQztRQUVsQixJQUFJLE9BQU8sQ0FBQyxNQUFNLElBQUksT0FBTyxDQUFDLFlBQVksRUFBRTtZQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLGdFQUFnRSxDQUFDLENBQUM7U0FDbkY7UUFFRCxJQUFJLE9BQU8sQ0FBQyxNQUFNLEVBQUU7WUFDbEIsYUFBYSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLENBQUM7U0FDNUM7UUFFRCxJQUFJLE9BQU8sQ0FBQyxZQUFZLEVBQUU7WUFDeEIsYUFBYSxHQUFHLENBQUMsQ0FBQyxDQUFDO1NBQ3BCO1FBRUQsSUFBSSxhQUFhLEVBQUU7WUFDakIsT0FBTyxDQUFDLHdCQUF3QixDQUFDLEdBQUcsSUFBSSxhQUFhLEdBQUcsQ0FBQztTQUMxRDtRQUVELEVBQUU7UUFDRixnQ0FBZ0M7UUFDaEMsRUFBRTtRQUVGLElBQUksT0FBTyxDQUFDLGFBQWEsRUFBRTtZQUN6QixPQUFPLENBQUMsK0JBQStCLENBQUMsR0FBRyxJQUFJLE9BQU8sQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUM7U0FDbkY7UUFFRCxFQUFFO1FBQ0YsYUFBYTtRQUViLE1BQU0sVUFBVSxHQUFHLE9BQU8sQ0FBQyxVQUFVLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUM7UUFFL0UsRUFBRTtRQUNGLHlCQUF5QjtRQUV6QixNQUFNLHlCQUF5QixHQUE0QixFQUFHLENBQUM7UUFDL0QsTUFBTSxtQkFBbUIsR0FBNkIsRUFBRyxDQUFDO1FBRTFELEtBQUssTUFBTSxDQUFFLElBQUksRUFBRSxLQUFLLENBQUUsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFO1lBQ3JELE1BQU0sR0FBRyxHQUFHLDBCQUEwQixJQUFJLEVBQUUsQ0FBQztZQUM3Qyx5QkFBeUIsQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7WUFDdkMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLEdBQUcsSUFBSSxDQUFDO1NBQ2pDO1FBRUQsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRSxJQUFJLDhCQUFlLENBQUM7WUFDbkQsZ0JBQWdCLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxxQkFBcUIsRUFBRTtZQUMvRCxvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxVQUFVLEVBQUUsR0FBRyxVQUFVLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSx5QkFBeUIsRUFBRSxpQkFBaUIsRUFBRSxzQkFBc0IsRUFBRSxFQUFFO2FBQzVIO1NBQ0YsQ0FBQyxFQUFFO1lBQ0YsZUFBZSxFQUFFO2dCQUNmLEVBQUUsVUFBVSxFQUFFLEdBQUcsVUFBVSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsbUJBQW1CLEVBQUU7YUFDekU7U0FDRixDQUFDLENBQUM7UUFFSCx5RkFBeUY7UUFDekYsU0FBUyxzQkFBc0I7WUFDN0IsTUFBTSxPQUFPLEdBQUcsT0FBTyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFFOUMsSUFBSSxPQUFPLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFDeEIsT0FBTyxTQUFTLENBQUM7YUFDbEI7WUFFRCxNQUFNLFFBQVEsR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1lBRXJDLFFBQVEsQ0FBQyxJQUFJLENBQUMseUNBQXlDLENBQUMsQ0FBQztZQUN6RCxRQUFRLENBQUMsSUFBSSxDQUFDLGlFQUFpRSxDQUFDLENBQUM7WUFFakYsTUFBTSxTQUFTLEdBQUcsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUUzRSxRQUFRLENBQUMsSUFBSSxDQUFDLE9BQU8sU0FBUyxHQUFHLENBQUMsQ0FBQztZQUNuQyxRQUFRLENBQUMsSUFBSSxDQUFDLGdGQUFnRixDQUFDLENBQUM7WUFDaEcsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUV0QixPQUFPO2dCQUNMLGtCQUFrQixFQUFFLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO2FBQ3hDLENBQUM7UUFDSixDQUFDO0lBQ0gsQ0FBQztJQUVNLFdBQVcsQ0FBQyxRQUFnQjtRQUNqQyxPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVyxDQUFDLFFBQWdCLEVBQUUsUUFBa0I7UUFDckQsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsR0FBRyxRQUFRLENBQUM7SUFDckMsQ0FBQztJQUVNLGVBQWUsQ0FBQyxJQUFZO1FBQ2pDLElBQUksQ0FBQyxJQUFJLEVBQUU7WUFDVCxPQUFPLElBQUksQ0FBQztTQUNiO1FBRUQsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ3hCLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxHQUFHLEVBQUU7Z0JBQ3JCLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQy9GO1lBRUQsb0JBQW9CO1lBQ3BCLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDN0M7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzlCLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUMzQixJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksS0FBSyxFQUFFLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1NBQ3hFO1FBRUQsSUFBSSxRQUFRLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ2IsUUFBUSxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDbkM7UUFFRCxPQUFPLFFBQVEsQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRCxJQUFXLEdBQUc7UUFDWixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM1QyxDQUFDO0NBQ0Y7QUE1TUQsb0NBNE1DO0FBRUQsTUFBYSxRQUFTLFNBQVEsWUFBWTtJQVV4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsd0JBQXdCLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQztRQUVuQyxJQUFJLEtBQUssQ0FBQyxNQUFNLFlBQVksWUFBWSxFQUFFO1lBQ3hDLEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxNQUFNLGFBQWEsR0FBcUI7WUFDdEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVM7WUFDekMsUUFBUSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtZQUNqQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7U0FDekIsQ0FBQztRQUNGLE1BQU0sUUFBUSxHQUFHLElBQUksa0NBQVcsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRWxFLElBQUksQ0FBQyxVQUFVLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztRQUMvQixJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDO1FBRXBDLCtDQUErQztRQUMvQyxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO1FBQzlCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUFFLElBQUksQ0FBQyxJQUFJLElBQUksR0FBRyxDQUFDO1NBQUU7UUFDbkQsSUFBSSxDQUFDLElBQUksSUFBSSxLQUFLLENBQUMsUUFBUSxDQUFDO1FBRTVCLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDO1FBQ3pELElBQUksVUFBVSxFQUFFO1lBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDeEMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsOEVBQThFO1FBQzlFLGtFQUFrRTtRQUNsRSxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUM7UUFDdEYsSUFBSSxDQUFDLG9CQUFvQixHQUFHO1lBQzFCLEdBQUcsS0FBSyxDQUFDLE1BQU0sQ0FBQyxvQkFBb0I7WUFDcEMsR0FBRyxLQUFLLENBQUMsb0JBQW9CO1NBQzlCLENBQUM7UUFDRixJQUFJLENBQUMsMkJBQTJCLEdBQUcsS0FBSyxDQUFDLDJCQUEyQixJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsMkJBQTJCLENBQUM7UUFFakgsSUFBSSxJQUFJLENBQUMsMkJBQTJCLEVBQUU7WUFDcEMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1NBQ3pEO0lBQ0gsQ0FBQztDQUNGO0FBdkRELDRCQXVEQztBQW9CRDs7O0dBR0c7QUFDSCxNQUFhLGFBQWMsU0FBUSxRQUFRO0lBT3pDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDcEIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsa0JBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQjtZQUM1QyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsb0JBQW9CO1NBQ2pELENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDekUsSUFBSSxTQUFTLEVBQUU7WUFDYixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7U0FDeEM7SUFDSCxDQUFDO0lBRU0sU0FBUyxDQUFDLFVBQWtCLEVBQUUsV0FBeUIsRUFBRSxPQUF1QjtRQUNyRix3RUFBd0U7UUFDeEUsb0RBQW9EO1FBQ3BELElBQUksSUFBSSxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksS0FBSyxHQUFHLEVBQUU7WUFDM0QsNERBQTREO1lBQzVELElBQUksQ0FBQyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsWUFBWSxlQUFNLENBQUMsRUFBRTtnQkFDMUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsVUFBVSxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQzthQUNqRTtTQUNGO1FBQ0QsT0FBTyxLQUFLLENBQUMsU0FBUyxDQUFDLFVBQVUsRUFBRSxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDM0QsQ0FBQztDQUNGO0FBaENELHNDQWdDQztBQUVELFNBQVMsd0JBQXdCLENBQUMsSUFBWTtJQUM1Qyw4Q0FBOEM7SUFDOUMsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLEVBQUU7UUFDOUMsSUFBSSxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFFdkMsZ0RBQWdEO1FBQ2hELElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUN0QixJQUFJLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztTQUN4QztLQUNGO0lBRUQsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRTtRQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDO3VEQUNtQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO0tBQzVEO0FBQ0gsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbnN0cnVjdCwgSVJlc291cmNlIGFzIElSZXNvdXJjZUJhc2UsIFJlc291cmNlIGFzIFJlc291cmNlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDZm5SZXNvdXJjZSwgQ2ZuUmVzb3VyY2VQcm9wcyB9IGZyb20gJy4vYXBpZ2F0ZXdheS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQ29ycywgQ29yc09wdGlvbnMgfSBmcm9tICcuL2NvcnMnO1xuaW1wb3J0IHsgSW50ZWdyYXRpb24gfSBmcm9tICcuL2ludGVncmF0aW9uJztcbmltcG9ydCB7IE1vY2tJbnRlZ3JhdGlvbiB9IGZyb20gJy4vaW50ZWdyYXRpb25zJztcbmltcG9ydCB7IE1ldGhvZCwgTWV0aG9kT3B0aW9ucyB9IGZyb20gJy4vbWV0aG9kJztcbmltcG9ydCB7IFJlc3RBcGkgfSBmcm9tICcuL3Jlc3RhcGknO1xuXG5leHBvcnQgaW50ZXJmYWNlIElSZXNvdXJjZSBleHRlbmRzIElSZXNvdXJjZUJhc2Uge1xuICAvKipcbiAgICogVGhlIHBhcmVudCBvZiB0aGlzIHJlc291cmNlIG9yIHVuZGVmaW5lZCBmb3IgdGhlIHJvb3QgcmVzb3VyY2UuXG4gICAqL1xuICByZWFkb25seSBwYXJlbnRSZXNvdXJjZT86IElSZXNvdXJjZTtcblxuICAvKipcbiAgICogVGhlIHJlc3QgQVBJIHRoYXQgdGhpcyByZXNvdXJjZSBpcyBwYXJ0IG9mLlxuICAgKlxuICAgKiBUaGUgcmVhc29uIHdlIG5lZWQgdGhlIFJlc3RBcGkgb2JqZWN0IGl0c2VsZiBhbmQgbm90IGp1c3QgdGhlIElEIGlzIGJlY2F1c2UgdGhlIG1vZGVsXG4gICAqIGlzIGJlaW5nIHRyYWNrZWQgYnkgdGhlIHRvcC1sZXZlbCBSZXN0QXBpIG9iamVjdCBmb3IgdGhlIHB1cnBvc2Ugb2YgY2FsY3VsYXRpbmcgaXQnc1xuICAgKiBoYXNoIHRvIGRldGVybWluZSB0aGUgSUQgb2YgdGhlIGRlcGxveW1lbnQuIFRoaXMgYWxsb3dzIHVzIHRvIGF1dG9tYXRpY2FsbHkgdXBkYXRlXG4gICAqIHRoZSBkZXBsb3ltZW50IHdoZW4gdGhlIG1vZGVsIG9mIHRoZSBSRVNUIEFQSSBjaGFuZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzdEFwaTogUmVzdEFwaTtcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSByZXNvdXJjZS5cbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZnVsbCBwYXRoIG9mIHRoaXMgcmVzdW9yY2UuXG4gICAqL1xuICByZWFkb25seSBwYXRoOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIGludGVncmF0aW9uIHRvIHVzZSBhcyBhIGRlZmF1bHQgZm9yIGFsbCBtZXRob2RzIGNyZWF0ZWQgd2l0aGluIHRoaXNcbiAgICogQVBJIHVubGVzcyBhbiBpbnRlZ3JhdGlvbiBpcyBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcblxuICAvKipcbiAgICogTWV0aG9kIG9wdGlvbnMgdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGN1c3RvbSBvcHRpb25zIGFyZSBzcGVjaWZpZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0TWV0aG9kT3B0aW9ucz86IE1ldGhvZE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIERlZmF1bHQgb3B0aW9ucyBmb3IgQ09SUyBwcmVmbGlnaHQgT1BUSU9OUyBtZXRob2QuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnM/OiBDb3JzT3B0aW9ucztcblxuICAvKipcbiAgICogR2V0cyBvciBjcmVhdGUgYWxsIHJlc291cmNlcyBsZWFkaW5nIHVwIHRvIHRoZSBzcGVjaWZpZWQgcGF0aC5cbiAgICpcbiAgICogLSBQYXRoIG1heSBvbmx5IHN0YXJ0IHdpdGggXCIvXCIgaWYgdGhpcyBtZXRob2QgaXMgY2FsbGVkIG9uIHRoZSByb290IHJlc291cmNlLlxuICAgKiAtIEFsbCByZXNvdXJjZXMgYXJlIGNyZWF0ZWQgdXNpbmcgZGVmYXVsdCBvcHRpb25zLlxuICAgKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcmVsYXRpdmUgcGF0aFxuICAgKiBAcmV0dXJucyBhIG5ldyBvciBleGlzdGluZyByZXNvdXJjZS5cbiAgICovXG4gIHJlc291cmNlRm9yUGF0aChwYXRoOiBzdHJpbmcpOiBSZXNvdXJjZTtcblxuICAvKipcbiAgICogRGVmaW5lcyBhIG5ldyBjaGlsZCByZXNvdXJjZSB3aGVyZSB0aGlzIHJlc291cmNlIGlzIHRoZSBwYXJlbnQuXG4gICAqIEBwYXJhbSBwYXRoUGFydCBUaGUgcGF0aCBwYXJ0IGZvciB0aGUgY2hpbGQgcmVzb3VyY2VcbiAgICogQHBhcmFtIG9wdGlvbnMgUmVzb3VyY2Ugb3B0aW9uc1xuICAgKiBAcmV0dXJucyBBIFJlc291cmNlIG9iamVjdFxuICAgKi9cbiAgYWRkUmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZywgb3B0aW9ucz86IFJlc291cmNlT3B0aW9ucyk6IFJlc291cmNlO1xuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYSBjaGlsZCByZXNvdXJjZSBieSBwYXRoIHBhcnQuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoUGFydCBUaGUgcGF0aCBwYXJ0IG9mIHRoZSBjaGlsZCByZXNvdXJjZVxuICAgKiBAcmV0dXJucyB0aGUgY2hpbGQgcmVzb3VyY2Ugb3IgdW5kZWZpbmVkIGlmIG5vdCBmb3VuZFxuICAgKi9cbiAgZ2V0UmVzb3VyY2UocGF0aFBhcnQ6IHN0cmluZyk6IElSZXNvdXJjZSB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogQWRkcyBhIGdyZWVkeSBwcm94eSByZXNvdXJjZSAoXCJ7cHJveHkrfVwiKSBhbmQgYW4gQU5ZIG1ldGhvZCB0byB0aGlzIHJvdXRlLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBEZWZhdWx0IGludGVncmF0aW9uIGFuZCBtZXRob2Qgb3B0aW9ucy5cbiAgICovXG4gIGFkZFByb3h5KG9wdGlvbnM/OiBQcm94eVJlc291cmNlT3B0aW9ucyk6IFByb3h5UmVzb3VyY2U7XG5cbiAgLyoqXG4gICAqIERlZmluZXMgYSBuZXcgbWV0aG9kIGZvciB0aGlzIHJlc291cmNlLlxuICAgKiBAcGFyYW0gaHR0cE1ldGhvZCBUaGUgSFRUUCBtZXRob2RcbiAgICogQHBhcmFtIHRhcmdldCBUaGUgdGFyZ2V0IGJhY2tlbmQgaW50ZWdyYXRpb24gZm9yIHRoaXMgbWV0aG9kXG4gICAqIEBwYXJhbSBvcHRpb25zIE1ldGhvZCBvcHRpb25zLCBzdWNoIGFzIGF1dGhlbnRpY2F0aW9uLlxuICAgKlxuICAgKiBAcmV0dXJucyBUaGUgbmV3bHkgY3JlYXRlZCBgTWV0aG9kYCBvYmplY3QuXG4gICAqL1xuICBhZGRNZXRob2QoaHR0cE1ldGhvZDogc3RyaW5nLCB0YXJnZXQ/OiBJbnRlZ3JhdGlvbiwgb3B0aW9ucz86IE1ldGhvZE9wdGlvbnMpOiBNZXRob2Q7XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gT1BUSU9OUyBtZXRob2QgdG8gdGhpcyByZXNvdXJjZSB3aGljaCByZXNwb25kcyB0byBDcm9zcy1PcmlnaW5cbiAgICogUmVzb3VyY2UgU2hhcmluZyAoQ09SUykgcHJlZmxpZ2h0IHJlcXVlc3RzLlxuICAgKlxuICAgKiBDcm9zcy1PcmlnaW4gUmVzb3VyY2UgU2hhcmluZyAoQ09SUykgaXMgYSBtZWNoYW5pc20gdGhhdCB1c2VzIGFkZGl0aW9uYWxcbiAgICogSFRUUCBoZWFkZXJzIHRvIHRlbGwgYnJvd3NlcnMgdG8gZ2l2ZSBhIHdlYiBhcHBsaWNhdGlvbiBydW5uaW5nIGF0IG9uZVxuICAgKiBvcmlnaW4sIGFjY2VzcyB0byBzZWxlY3RlZCByZXNvdXJjZXMgZnJvbSBhIGRpZmZlcmVudCBvcmlnaW4uIEEgd2ViXG4gICAqIGFwcGxpY2F0aW9uIGV4ZWN1dGVzIGEgY3Jvc3Mtb3JpZ2luIEhUVFAgcmVxdWVzdCB3aGVuIGl0IHJlcXVlc3RzIGFcbiAgICogcmVzb3VyY2UgdGhhdCBoYXMgYSBkaWZmZXJlbnQgb3JpZ2luIChkb21haW4sIHByb3RvY29sLCBvciBwb3J0KSBmcm9tIGl0c1xuICAgKiBvd24uXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kZXZlbG9wZXIubW96aWxsYS5vcmcvZW4tVVMvZG9jcy9XZWIvSFRUUC9DT1JTXG4gICAqIEBwYXJhbSBvcHRpb25zIENPUlMgb3B0aW9uc1xuICAgKiBAcmV0dXJucyBhIGBNZXRob2RgIG9iamVjdFxuICAgKi9cbiAgYWRkQ29yc1ByZWZsaWdodChvcHRpb25zOiBDb3JzT3B0aW9ucyk6IE1ldGhvZDtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogQW4gaW50ZWdyYXRpb24gdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGFuIGludGVncmF0aW9uIGlzIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJbmhlcml0ZWQgZnJvbSBwYXJlbnQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcblxuICAvKipcbiAgICogTWV0aG9kIG9wdGlvbnMgdG8gdXNlIGFzIGEgZGVmYXVsdCBmb3IgYWxsIG1ldGhvZHMgY3JlYXRlZCB3aXRoaW4gdGhpc1xuICAgKiBBUEkgdW5sZXNzIGN1c3RvbSBvcHRpb25zIGFyZSBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gSW5oZXJpdGVkIGZyb20gcGFyZW50LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVmYXVsdE1ldGhvZE9wdGlvbnM/OiBNZXRob2RPcHRpb25zO1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgQ09SUyBwcmVmbGlnaHQgT1BUSU9OUyBtZXRob2QgdG8gdGhpcyByZXNvdXJjZSBhbmQgYWxsIGNoaWxkXG4gICAqIHJlc291cmNlcy5cbiAgICpcbiAgICogWW91IGNhbiBhZGQgQ09SUyBhdCB0aGUgcmVzb3VyY2UtbGV2ZWwgdXNpbmcgYGFkZENvcnNQcmVmbGlnaHRgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIENPUlMgaXMgZGlzYWJsZWRcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucz86IENvcnNPcHRpb25zO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFJlc291cmNlUHJvcHMgZXh0ZW5kcyBSZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHBhcmVudCByZXNvdXJjZSBvZiB0aGlzIHJlc291cmNlLiBZb3UgY2FuIGVpdGhlciBwYXNzIGFub3RoZXJcbiAgICogYFJlc291cmNlYCBvYmplY3Qgb3IgYSBgUmVzdEFwaWAgb2JqZWN0IGhlcmUuXG4gICAqL1xuICByZWFkb25seSBwYXJlbnQ6IElSZXNvdXJjZTtcblxuICAvKipcbiAgICogQSBwYXRoIG5hbWUgZm9yIHRoZSByZXNvdXJjZS5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGhQYXJ0OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBSZXNvdXJjZUJhc2UgZXh0ZW5kcyBSZXNvdXJjZUNvbnN0cnVjdCBpbXBsZW1lbnRzIElSZXNvdXJjZSB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwYXJlbnRSZXNvdXJjZT86IElSZXNvdXJjZTtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJlc3RBcGk6IFJlc3RBcGk7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByZXNvdXJjZUlkOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwYXRoOiBzdHJpbmc7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlZmF1bHRNZXRob2RPcHRpb25zPzogTWV0aG9kT3B0aW9ucztcbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucz86IENvcnNPcHRpb25zO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgY2hpbGRyZW46IHsgW3BhdGhQYXJ0OiBzdHJpbmddOiBSZXNvdXJjZSB9ID0geyB9O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIGFkZFJlc291cmNlKHBhdGhQYXJ0OiBzdHJpbmcsIG9wdGlvbnM/OiBSZXNvdXJjZU9wdGlvbnMpOiBSZXNvdXJjZSB7XG4gICAgcmV0dXJuIG5ldyBSZXNvdXJjZSh0aGlzLCBwYXRoUGFydCwgeyBwYXJlbnQ6IHRoaXMsIHBhdGhQYXJ0LCAuLi5vcHRpb25zIH0pO1xuICB9XG5cbiAgcHVibGljIGFkZE1ldGhvZChodHRwTWV0aG9kOiBzdHJpbmcsIGludGVncmF0aW9uPzogSW50ZWdyYXRpb24sIG9wdGlvbnM/OiBNZXRob2RPcHRpb25zKTogTWV0aG9kIHtcbiAgICByZXR1cm4gbmV3IE1ldGhvZCh0aGlzLCBodHRwTWV0aG9kLCB7IHJlc291cmNlOiB0aGlzLCBodHRwTWV0aG9kLCBpbnRlZ3JhdGlvbiwgb3B0aW9ucyB9KTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRQcm94eShvcHRpb25zPzogUHJveHlSZXNvdXJjZU9wdGlvbnMpOiBQcm94eVJlc291cmNlIHtcbiAgICByZXR1cm4gbmV3IFByb3h5UmVzb3VyY2UodGhpcywgJ3twcm94eSt9JywgeyBwYXJlbnQ6IHRoaXMsIC4uLm9wdGlvbnMgfSk7XG4gIH1cblxuICBwdWJsaWMgYWRkQ29yc1ByZWZsaWdodChvcHRpb25zOiBDb3JzT3B0aW9ucykge1xuICAgIGNvbnN0IGhlYWRlcnM6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9ID0geyB9O1xuXG4gICAgLy9cbiAgICAvLyBBY2Nlc3MtQ29udHJvbC1BbGxvdy1IZWFkZXJzXG5cbiAgICBjb25zdCBhbGxvd0hlYWRlcnMgPSBvcHRpb25zLmFsbG93SGVhZGVycyB8fCBDb3JzLkRFRkFVTFRfSEVBREVSUztcbiAgICBoZWFkZXJzWydBY2Nlc3MtQ29udHJvbC1BbGxvdy1IZWFkZXJzJ10gPSBgJyR7YWxsb3dIZWFkZXJzLmpvaW4oJywnKX0nYDtcblxuICAgIC8vXG4gICAgLy8gQWNjZXNzLUNvbnRyb2wtQWxsb3ctT3JpZ2luXG5cbiAgICBpZiAob3B0aW9ucy5hbGxvd09yaWdpbnMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2FsbG93T3JpZ2lucyBtdXN0IGNvbnRhaW4gYXQgbGVhc3Qgb25lIG9yaWdpbicpO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLmFsbG93T3JpZ2lucy5pbmNsdWRlcygnKicpICYmIG9wdGlvbnMuYWxsb3dPcmlnaW5zLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBcImFsbG93T3JpZ2luc1wiIC0gY2Fubm90IG1peCBcIipcIiB3aXRoIHNwZWNpZmljIG9yaWdpbnM6ICR7b3B0aW9ucy5hbGxvd09yaWdpbnMuam9pbignLCcpfWApO1xuICAgIH1cblxuICAgIC8vIHdlIHVzZSB0aGUgZmlyc3Qgb3JpZ2luIGhlcmUgYW5kIGlmIHRoZXJlIGFyZSBtb3JlIG9yaWdpbnMgaW4gdGhlIGxpc3QsIHdlXG4gICAgLy8gd2lsbCBtYXRjaCBhZ2FpbnN0IHRoZW0gaW4gdGhlIHJlc3BvbnNlIHZlbG9jaXR5IHRlbXBsYXRlXG4gICAgY29uc3QgaW5pdGlhbE9yaWdpbiA9IG9wdGlvbnMuYWxsb3dPcmlnaW5zWzBdO1xuICAgIGhlYWRlcnNbJ0FjY2Vzcy1Db250cm9sLUFsbG93LU9yaWdpbiddID0gYCcke2luaXRpYWxPcmlnaW59J2A7XG5cbiAgICAvLyB0aGUgXCJWYXJ5XCIgaGVhZGVyIGlzIHJlcXVpcmVkIGlmIHdlIGFsbG93IGEgc3BlY2lmaWMgb3JpZ2luXG4gICAgLy8gaHR0cHM6Ly9kZXZlbG9wZXIubW96aWxsYS5vcmcvZW4tVVMvZG9jcy9XZWIvSFRUUC9IZWFkZXJzL0FjY2Vzcy1Db250cm9sLUFsbG93LU9yaWdpbiNDT1JTX2FuZF9jYWNoaW5nXG4gICAgaWYgKGluaXRpYWxPcmlnaW4gIT09ICcqJykge1xuICAgICAgaGVhZGVycy5WYXJ5ID0gYCdPcmlnaW4nYDtcbiAgICB9XG5cbiAgICAvL1xuICAgIC8vIEFjY2Vzcy1Db250cm9sLUFsbG93LU1ldGhvZHNcblxuICAgIGxldCBhbGxvd01ldGhvZHMgPSBvcHRpb25zLmFsbG93TWV0aG9kcyB8fCBDb3JzLkFMTF9NRVRIT0RTO1xuXG4gICAgaWYgKGFsbG93TWV0aG9kcy5pbmNsdWRlcygnQU5ZJykpIHtcbiAgICAgIGlmIChhbGxvd01ldGhvZHMubGVuZ3RoID4gMSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFOWSBjYW5ub3QgYmUgdXNlZCB3aXRoIGFueSBvdGhlciBtZXRob2QuIFJlY2VpdmVkOiAke2FsbG93TWV0aG9kcy5qb2luKCcsJyl9YCk7XG4gICAgICB9XG5cbiAgICAgIGFsbG93TWV0aG9kcyA9IENvcnMuQUxMX01FVEhPRFM7XG4gICAgfVxuXG4gICAgaGVhZGVyc1snQWNjZXNzLUNvbnRyb2wtQWxsb3ctTWV0aG9kcyddID0gYCcke2FsbG93TWV0aG9kcy5qb2luKCcsJyl9J2A7XG5cbiAgICAvL1xuICAgIC8vIEFjY2Vzcy1Db250cm9sLUFsbG93LUNyZWRlbnRpYWxzXG5cbiAgICBpZiAob3B0aW9ucy5hbGxvd0NyZWRlbnRpYWxzKSB7XG4gICAgICBoZWFkZXJzWydBY2Nlc3MtQ29udHJvbC1BbGxvdy1DcmVkZW50aWFscyddID0gYCd0cnVlJ2A7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBBY2Nlc3MtQ29udHJvbC1NYXgtQWdlXG5cbiAgICBsZXQgbWF4QWdlU2Vjb25kcztcblxuICAgIGlmIChvcHRpb25zLm1heEFnZSAmJiBvcHRpb25zLmRpc2FibGVDYWNoZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgb3B0aW9ucyBcIm1heEFnZVwiIGFuZCBcImRpc2FibGVDYWNoZVwiIGFyZSBtdXR1YWxseSBleGNsdXNpdmVgKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5tYXhBZ2UpIHtcbiAgICAgIG1heEFnZVNlY29uZHMgPSBvcHRpb25zLm1heEFnZS50b1NlY29uZHMoKTtcbiAgICB9XG5cbiAgICBpZiAob3B0aW9ucy5kaXNhYmxlQ2FjaGUpIHtcbiAgICAgIG1heEFnZVNlY29uZHMgPSAtMTtcbiAgICB9XG5cbiAgICBpZiAobWF4QWdlU2Vjb25kcykge1xuICAgICAgaGVhZGVyc1snQWNjZXNzLUNvbnRyb2wtTWF4LUFnZSddID0gYCcke21heEFnZVNlY29uZHN9J2A7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBBY2Nlc3MtQ29udHJvbC1FeHBvc2UtSGVhZGVyc1xuICAgIC8vXG5cbiAgICBpZiAob3B0aW9ucy5leHBvc2VIZWFkZXJzKSB7XG4gICAgICBoZWFkZXJzWydBY2Nlc3MtQ29udHJvbC1FeHBvc2UtSGVhZGVycyddID0gYCcke29wdGlvbnMuZXhwb3NlSGVhZGVycy5qb2luKCcsJyl9J2A7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBzdGF0dXNDb2RlXG5cbiAgICBjb25zdCBzdGF0dXNDb2RlID0gb3B0aW9ucy5zdGF0dXNDb2RlICE9PSB1bmRlZmluZWQgPyBvcHRpb25zLnN0YXR1c0NvZGUgOiAyMDQ7XG5cbiAgICAvL1xuICAgIC8vIHByZXBhcmUgcmVzcG9uc2VQYXJhbXNcblxuICAgIGNvbnN0IGludGVncmF0aW9uUmVzcG9uc2VQYXJhbXM6IHsgW3A6IHN0cmluZ106IHN0cmluZyB9ID0geyB9O1xuICAgIGNvbnN0IG1ldGhvZFJlcG9uc2VQYXJhbXM6IHsgW3A6IHN0cmluZ106IGJvb2xlYW4gfSA9IHsgfTtcblxuICAgIGZvciAoY29uc3QgWyBuYW1lLCB2YWx1ZSBdIG9mIE9iamVjdC5lbnRyaWVzKGhlYWRlcnMpKSB7XG4gICAgICBjb25zdCBrZXkgPSBgbWV0aG9kLnJlc3BvbnNlLmhlYWRlci4ke25hbWV9YDtcbiAgICAgIGludGVncmF0aW9uUmVzcG9uc2VQYXJhbXNba2V5XSA9IHZhbHVlO1xuICAgICAgbWV0aG9kUmVwb25zZVBhcmFtc1trZXldID0gdHJ1ZTtcbiAgICB9XG5cbiAgICByZXR1cm4gdGhpcy5hZGRNZXRob2QoJ09QVElPTlMnLCBuZXcgTW9ja0ludGVncmF0aW9uKHtcbiAgICAgIHJlcXVlc3RUZW1wbGF0ZXM6IHsgJ2FwcGxpY2F0aW9uL2pzb24nOiAneyBzdGF0dXNDb2RlOiAyMDAgfScgfSxcbiAgICAgIGludGVncmF0aW9uUmVzcG9uc2VzOiBbXG4gICAgICAgIHsgc3RhdHVzQ29kZTogYCR7c3RhdHVzQ29kZX1gLCByZXNwb25zZVBhcmFtZXRlcnM6IGludGVncmF0aW9uUmVzcG9uc2VQYXJhbXMsIHJlc3BvbnNlVGVtcGxhdGVzOiByZW5kZXJSZXNwb25zZVRlbXBsYXRlKCkgfVxuICAgICAgXSxcbiAgICB9KSwge1xuICAgICAgbWV0aG9kUmVzcG9uc2VzOiBbXG4gICAgICAgIHsgc3RhdHVzQ29kZTogYCR7c3RhdHVzQ29kZX1gLCByZXNwb25zZVBhcmFtZXRlcnM6IG1ldGhvZFJlcG9uc2VQYXJhbXMgfVxuICAgICAgXVxuICAgIH0pO1xuXG4gICAgLy8gcmVuZGVycyB0aGUgcmVzcG9uc2UgdGVtcGxhdGUgdG8gbWF0Y2ggYWxsIHBvc3NpYmxlIG9yaWdpbnMgKGlmIHdlIGhhdmUgbW9yZSB0aGFuIG9uZSlcbiAgICBmdW5jdGlvbiByZW5kZXJSZXNwb25zZVRlbXBsYXRlKCkge1xuICAgICAgY29uc3Qgb3JpZ2lucyA9IG9wdGlvbnMuYWxsb3dPcmlnaW5zLnNsaWNlKDEpO1xuXG4gICAgICBpZiAob3JpZ2lucy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cblxuICAgICAgY29uc3QgdGVtcGxhdGUgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuXG4gICAgICB0ZW1wbGF0ZS5wdXNoKGAjc2V0KCRvcmlnaW4gPSAkaW5wdXQucGFyYW1zKFwiT3JpZ2luXCIpKWApO1xuICAgICAgdGVtcGxhdGUucHVzaChgI2lmKCRvcmlnaW4gPT0gXCJcIikgI3NldCgkb3JpZ2luID0gJGlucHV0LnBhcmFtcyhcIm9yaWdpblwiKSkgI2VuZGApO1xuXG4gICAgICBjb25zdCBjb25kaXRpb24gPSBvcmlnaW5zLm1hcChvID0+IGAkb3JpZ2luLm1hdGNoZXMoXCIke299XCIpYCkuam9pbignIHx8ICcpO1xuXG4gICAgICB0ZW1wbGF0ZS5wdXNoKGAjaWYoJHtjb25kaXRpb259KWApO1xuICAgICAgdGVtcGxhdGUucHVzaChgICAjc2V0KCRjb250ZXh0LnJlc3BvbnNlT3ZlcnJpZGUuaGVhZGVyLkFjY2Vzcy1Db250cm9sLUFsbG93LU9yaWdpbiA9ICRvcmlnaW4pYCk7XG4gICAgICB0ZW1wbGF0ZS5wdXNoKCcjZW5kJyk7XG5cbiAgICAgIHJldHVybiB7XG4gICAgICAgICdhcHBsaWNhdGlvbi9qc29uJzogdGVtcGxhdGUuam9pbignXFxuJylcbiAgICAgIH07XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGdldFJlc291cmNlKHBhdGhQYXJ0OiBzdHJpbmcpOiBJUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLmNoaWxkcmVuW3BhdGhQYXJ0XTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdHJhY2tDaGlsZChwYXRoUGFydDogc3RyaW5nLCByZXNvdXJjZTogUmVzb3VyY2UpIHtcbiAgICB0aGlzLmNoaWxkcmVuW3BhdGhQYXJ0XSA9IHJlc291cmNlO1xuICB9XG5cbiAgcHVibGljIHJlc291cmNlRm9yUGF0aChwYXRoOiBzdHJpbmcpOiBSZXNvdXJjZSB7XG4gICAgaWYgKCFwYXRoKSB7XG4gICAgICByZXR1cm4gdGhpcztcbiAgICB9XG5cbiAgICBpZiAocGF0aC5zdGFydHNXaXRoKCcvJykpIHtcbiAgICAgIGlmICh0aGlzLnBhdGggIT09ICcvJykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFBhdGggbWF5IHN0YXJ0IHdpdGggXCIvXCIgb25seSBmb3IgdGhlIHJlc291cmNlLCBidXQgd2UgYXJlIGF0OiAke3RoaXMucGF0aH1gKTtcbiAgICAgIH1cblxuICAgICAgLy8gdHJpbSB0cmFpbGluZyBcIi9cIlxuICAgICAgcmV0dXJuIHRoaXMucmVzb3VyY2VGb3JQYXRoKHBhdGguc3Vic3RyKDEpKTtcbiAgICB9XG5cbiAgICBjb25zdCBwYXJ0cyA9IHBhdGguc3BsaXQoJy8nKTtcbiAgICBjb25zdCBuZXh0ID0gcGFydHMuc2hpZnQoKTtcbiAgICBpZiAoIW5leHQgfHwgbmV4dCA9PT0gJycpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgcmVzb3VyY2VGb3JQYXRoIGNhbm5vdCBiZSBjYWxsZWQgd2l0aCBhbiBlbXB0eSBwYXRoYCk7XG4gICAgfVxuXG4gICAgbGV0IHJlc291cmNlID0gdGhpcy5nZXRSZXNvdXJjZShuZXh0KTtcbiAgICBpZiAoIXJlc291cmNlKSB7XG4gICAgICByZXNvdXJjZSA9IHRoaXMuYWRkUmVzb3VyY2UobmV4dCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc291cmNlLnJlc291cmNlRm9yUGF0aChwYXJ0cy5qb2luKCcvJykpO1xuICB9XG5cbiAgcHVibGljIGdldCB1cmwoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5yZXN0QXBpLnVybEZvclBhdGgodGhpcy5wYXRoKTtcbiAgfVxufVxuXG5leHBvcnQgY2xhc3MgUmVzb3VyY2UgZXh0ZW5kcyBSZXNvdXJjZUJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgcGFyZW50UmVzb3VyY2U/OiBJUmVzb3VyY2U7XG4gIHB1YmxpYyByZWFkb25seSByZXN0QXBpOiBSZXN0QXBpO1xuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nO1xuXG4gIHB1YmxpYyByZWFkb25seSBkZWZhdWx0SW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRNZXRob2RPcHRpb25zPzogTWV0aG9kT3B0aW9ucztcbiAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucz86IENvcnNPcHRpb25zO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBSZXNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHZhbGlkYXRlUmVzb3VyY2VQYXRoUGFydChwcm9wcy5wYXRoUGFydCk7XG5cbiAgICB0aGlzLnBhcmVudFJlc291cmNlID0gcHJvcHMucGFyZW50O1xuXG4gICAgaWYgKHByb3BzLnBhcmVudCBpbnN0YW5jZW9mIFJlc291cmNlQmFzZSkge1xuICAgICAgcHJvcHMucGFyZW50Ll90cmFja0NoaWxkKHByb3BzLnBhdGhQYXJ0LCB0aGlzKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXNvdXJjZVByb3BzOiBDZm5SZXNvdXJjZVByb3BzID0ge1xuICAgICAgcmVzdEFwaUlkOiBwcm9wcy5wYXJlbnQucmVzdEFwaS5yZXN0QXBpSWQsXG4gICAgICBwYXJlbnRJZDogcHJvcHMucGFyZW50LnJlc291cmNlSWQsXG4gICAgICBwYXRoUGFydDogcHJvcHMucGF0aFBhcnRcbiAgICB9O1xuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmblJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHJlc291cmNlUHJvcHMpO1xuXG4gICAgdGhpcy5yZXNvdXJjZUlkID0gcmVzb3VyY2UucmVmO1xuICAgIHRoaXMucmVzdEFwaSA9IHByb3BzLnBhcmVudC5yZXN0QXBpO1xuXG4gICAgLy8gcmVuZGVyIHJlc291cmNlIHBhdGggKHNwZWNpYWwgY2FzZSBmb3Igcm9vdClcbiAgICB0aGlzLnBhdGggPSBwcm9wcy5wYXJlbnQucGF0aDtcbiAgICBpZiAoIXRoaXMucGF0aC5lbmRzV2l0aCgnLycpKSB7IHRoaXMucGF0aCArPSAnLyc7IH1cbiAgICB0aGlzLnBhdGggKz0gcHJvcHMucGF0aFBhcnQ7XG5cbiAgICBjb25zdCBkZXBsb3ltZW50ID0gcHJvcHMucGFyZW50LnJlc3RBcGkubGF0ZXN0RGVwbG95bWVudDtcbiAgICBpZiAoZGVwbG95bWVudCkge1xuICAgICAgZGVwbG95bWVudC5ub2RlLmFkZERlcGVuZGVuY3kocmVzb3VyY2UpO1xuICAgICAgZGVwbG95bWVudC5hZGRUb0xvZ2ljYWxJZCh7IHJlc291cmNlOiByZXNvdXJjZVByb3BzIH0pO1xuICAgIH1cblxuICAgIC8vIHNldHVwIGRlZmF1bHRzIGJhc2VkIG9uIHByb3BlcnRpZXMgYW5kIGluaGVyaXQgZnJvbSBwYXJlbnQuIG1ldGhvZCBkZWZhdWx0c1xuICAgIC8vIGFyZSBpbmhlcml0ZWQgcGVyIHByb3BlcnR5LCBzbyBjaGlsZHJlbiBjYW4gb3ZlcnJpZGUgcGllY2VtZWFsLlxuICAgIHRoaXMuZGVmYXVsdEludGVncmF0aW9uID0gcHJvcHMuZGVmYXVsdEludGVncmF0aW9uIHx8IHByb3BzLnBhcmVudC5kZWZhdWx0SW50ZWdyYXRpb247XG4gICAgdGhpcy5kZWZhdWx0TWV0aG9kT3B0aW9ucyA9IHtcbiAgICAgIC4uLnByb3BzLnBhcmVudC5kZWZhdWx0TWV0aG9kT3B0aW9ucyxcbiAgICAgIC4uLnByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zXG4gICAgfTtcbiAgICB0aGlzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucyA9IHByb3BzLmRlZmF1bHRDb3JzUHJlZmxpZ2h0T3B0aW9ucyB8fCBwcm9wcy5wYXJlbnQuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zO1xuXG4gICAgaWYgKHRoaXMuZGVmYXVsdENvcnNQcmVmbGlnaHRPcHRpb25zKSB7XG4gICAgICB0aGlzLmFkZENvcnNQcmVmbGlnaHQodGhpcy5kZWZhdWx0Q29yc1ByZWZsaWdodE9wdGlvbnMpO1xuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgaW50ZXJmYWNlIFByb3h5UmVzb3VyY2VPcHRpb25zIGV4dGVuZHMgUmVzb3VyY2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIEFkZHMgYW4gXCJBTllcIiBtZXRob2QgdG8gdGhpcyByZXNvdXJjZS4gSWYgc2V0IHRvIGBmYWxzZWAsIHlvdSB3aWxsIGhhdmUgdG8gZXhwbGljaXRseVxuICAgKiBhZGQgbWV0aG9kcyB0byB0aGlzIHJlc291cmNlIGFmdGVyIGl0J3MgY3JlYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYW55TWV0aG9kPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcm94eVJlc291cmNlUHJvcHMgZXh0ZW5kcyBQcm94eVJlc291cmNlT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgcGFyZW50IHJlc291cmNlIG9mIHRoaXMgcmVzb3VyY2UuIFlvdSBjYW4gZWl0aGVyIHBhc3MgYW5vdGhlclxuICAgKiBgUmVzb3VyY2VgIG9iamVjdCBvciBhIGBSZXN0QXBpYCBvYmplY3QgaGVyZS5cbiAgICovXG4gIHJlYWRvbmx5IHBhcmVudDogSVJlc291cmNlO1xufVxuXG4vKipcbiAqIERlZmluZXMgYSB7cHJveHkrfSBncmVlZHkgcmVzb3VyY2UgYW5kIGFuIEFOWSBtZXRob2Qgb24gYSByb3V0ZS5cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXNldC11cC1zaW1wbGUtcHJveHkuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgUHJveHlSZXNvdXJjZSBleHRlbmRzIFJlc291cmNlIHtcbiAgLyoqXG4gICAqIElmIGBwcm9wcy5hbnlNZXRob2RgIGlzIGB0cnVlYCwgdGhpcyB3aWxsIGJlIHRoZSByZWZlcmVuY2UgdG8gdGhlICdBTlknXG4gICAqIG1ldGhvZCBhc3NvY2lhdGVkIHdpdGggdGhpcyBwcm94eSByZXNvdXJjZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhbnlNZXRob2Q/OiBNZXRob2Q7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFByb3h5UmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgcGFyZW50OiBwcm9wcy5wYXJlbnQsXG4gICAgICBwYXRoUGFydDogJ3twcm94eSt9JyxcbiAgICAgIGRlZmF1bHRJbnRlZ3JhdGlvbjogcHJvcHMuZGVmYXVsdEludGVncmF0aW9uLFxuICAgICAgZGVmYXVsdE1ldGhvZE9wdGlvbnM6IHByb3BzLmRlZmF1bHRNZXRob2RPcHRpb25zLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYW55TWV0aG9kID0gcHJvcHMuYW55TWV0aG9kICE9PSB1bmRlZmluZWQgPyBwcm9wcy5hbnlNZXRob2QgOiB0cnVlO1xuICAgIGlmIChhbnlNZXRob2QpIHtcbiAgICAgIHRoaXMuYW55TWV0aG9kID0gdGhpcy5hZGRNZXRob2QoJ0FOWScpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhZGRNZXRob2QoaHR0cE1ldGhvZDogc3RyaW5nLCBpbnRlZ3JhdGlvbj86IEludGVncmF0aW9uLCBvcHRpb25zPzogTWV0aG9kT3B0aW9ucyk6IE1ldGhvZCB7XG4gICAgLy8gSW4gY2FzZSB0aGlzIHByb3h5IGlzIG1vdW50ZWQgdW5kZXIgdGhlIHJvb3QsIGFsc28gYWRkIHRoaXMgbWV0aG9kIHRvXG4gICAgLy8gdGhlIHJvb3Qgc28gdGhhdCBlbXB0eSBwYXRocyBhcmUgcHJveGllZCBhcyB3ZWxsLlxuICAgIGlmICh0aGlzLnBhcmVudFJlc291cmNlICYmIHRoaXMucGFyZW50UmVzb3VyY2UucGF0aCA9PT0gJy8nKSB7XG4gICAgICAvLyBza2lwIGlmIHRoZSByb290IHJlc291cmNlIGFscmVhZHkgaGFzIHRoaXMgbWV0aG9kIGRlZmluZWRcbiAgICAgIGlmICghKHRoaXMucGFyZW50UmVzb3VyY2Uubm9kZS50cnlGaW5kQ2hpbGQoaHR0cE1ldGhvZCkgaW5zdGFuY2VvZiBNZXRob2QpKSB7XG4gICAgICAgIHRoaXMucGFyZW50UmVzb3VyY2UuYWRkTWV0aG9kKGh0dHBNZXRob2QsIGludGVncmF0aW9uLCBvcHRpb25zKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIHN1cGVyLmFkZE1ldGhvZChodHRwTWV0aG9kLCBpbnRlZ3JhdGlvbiwgb3B0aW9ucyk7XG4gIH1cbn1cblxuZnVuY3Rpb24gdmFsaWRhdGVSZXNvdXJjZVBhdGhQYXJ0KHBhcnQ6IHN0cmluZykge1xuICAvLyBzdHJpcCB7fSB3aGljaCBpbmRpY2F0ZSB0aGlzIGlzIGEgcGFyYW1ldGVyXG4gIGlmIChwYXJ0LnN0YXJ0c1dpdGgoJ3snKSAmJiBwYXJ0LmVuZHNXaXRoKCd9JykpIHtcbiAgICBwYXJ0ID0gcGFydC5zdWJzdHIoMSwgcGFydC5sZW5ndGggLSAyKTtcblxuICAgIC8vIHByb3h5IHJlc291cmNlcyBhcmUgYWxsb3dlZCB0byBlbmQgd2l0aCBhICcrJ1xuICAgIGlmIChwYXJ0LmVuZHNXaXRoKCcrJykpIHtcbiAgICAgIHBhcnQgPSBwYXJ0LnN1YnN0cigwLCBwYXJ0Lmxlbmd0aCAtIDEpO1xuICAgIH1cbiAgfVxuXG4gIGlmICghL15bYS16QS1aMC05XFwuXFxfXFwtXSskLy50ZXN0KHBhcnQpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBSZXNvdXJjZSdzIHBhdGggcGFydCBvbmx5IGFsbG93IFthLXpBLVowLTkuXy1dLCBhbiBvcHRpb25hbCB0cmFpbGluZyAnKydcbiAgICAgIGFuZCBjdXJseSBicmFjZXMgYXQgdGhlIGJlZ2lubmluZyBhbmQgdGhlIGVuZDogJHtwYXJ0fWApO1xuICB9XG59XG4iXX0=