import { IVpcEndpoint } from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import { Construct, IResource as IResourceBase, Resource } from '@aws-cdk/core';
import { IApiKey } from './api-key';
import { Deployment } from './deployment';
import { DomainName, DomainNameOptions } from './domain-name';
import { Method } from './method';
import { Model, ModelOptions } from './model';
import { RequestValidator, RequestValidatorOptions } from './requestvalidator';
import { IResource, ResourceOptions } from './resource';
import { Stage, StageOptions } from './stage';
import { UsagePlan, UsagePlanProps } from './usage-plan';
export interface IRestApi extends IResourceBase {
    /**
     * The ID of this API Gateway RestApi.
     * @attribute
     */
    readonly restApiId: string;
}
export interface RestApiProps extends ResourceOptions {
    /**
     * Indicates if a Deployment should be automatically created for this API,
     * and recreated when the API model (resources, methods) changes.
     *
     * Since API Gateway deployments are immutable, When this option is enabled
     * (by default), an AWS::ApiGateway::Deployment resource will automatically
     * created with a logical ID that hashes the API model (methods, resources
     * and options). This means that when the model changes, the logical ID of
     * this CloudFormation resource will change, and a new deployment will be
     * created.
     *
     * If this is set, `latestDeployment` will refer to the `Deployment` object
     * and `deploymentStage` will refer to a `Stage` that points to this
     * deployment. To customize the stage options, use the `deployOptions`
     * property.
     *
     * A CloudFormation Output will also be defined with the root URL endpoint
     * of this REST API.
     *
     * @default true
     */
    readonly deploy?: boolean;
    /**
     * Options for the API Gateway stage that will always point to the latest
     * deployment when `deploy` is enabled. If `deploy` is disabled,
     * this value cannot be set.
     *
     * @default - Based on defaults of `StageOptions`.
     */
    readonly deployOptions?: StageOptions;
    /**
     * Retains old deployment resources when the API changes. This allows
     * manually reverting stages to point to old deployments via the AWS
     * Console.
     *
     * @default false
     */
    readonly retainDeployments?: boolean;
    /**
     * A name for the API Gateway RestApi resource.
     *
     * @default - ID of the RestApi construct.
     */
    readonly restApiName?: string;
    /**
     * Custom header parameters for the request.
     * @see https://docs.aws.amazon.com/cli/latest/reference/apigateway/import-rest-api.html
     *
     * @default - No parameters.
     */
    readonly parameters?: {
        [key: string]: string;
    };
    /**
     * A policy document that contains the permissions for this RestApi
     *
     * @default - No policy.
     */
    readonly policy?: iam.PolicyDocument;
    /**
     * A description of the purpose of this API Gateway RestApi resource.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * The EndpointConfiguration property type specifies the endpoint types of a REST API
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html
     *
     * @default - No endpoint configuration
     */
    readonly endpointConfiguration?: EndpointConfiguration;
    /**
     * A list of the endpoint types of the API. Use this property when creating
     * an API.
     *
     * @default - No endpoint types.
     * @deprecated this property is deprecated, use endpointConfiguration instead
     */
    readonly endpointTypes?: EndpointType[];
    /**
     * The source of the API key for metering requests according to a usage
     * plan.
     *
     * @default - Metering is disabled.
     */
    readonly apiKeySourceType?: ApiKeySourceType;
    /**
     * The list of binary media mime-types that are supported by the RestApi
     * resource, such as "image/png" or "application/octet-stream"
     *
     * @default - RestApi supports only UTF-8-encoded text payloads.
     */
    readonly binaryMediaTypes?: string[];
    /**
     * Indicates whether to roll back the resource if a warning occurs while API
     * Gateway is creating the RestApi resource.
     *
     * @default false
     */
    readonly failOnWarnings?: boolean;
    /**
     * A nullable integer that is used to enable compression (with non-negative
     * between 0 and 10485760 (10M) bytes, inclusive) or disable compression
     * (when undefined) on an API. When compression is enabled, compression or
     * decompression is not applied on the payload if the payload size is
     * smaller than this value. Setting it to zero allows compression for any
     * payload size.
     *
     * @default - Compression is disabled.
     */
    readonly minimumCompressionSize?: number;
    /**
     * The ID of the API Gateway RestApi resource that you want to clone.
     *
     * @default - None.
     */
    readonly cloneFrom?: IRestApi;
    /**
     * Automatically configure an AWS CloudWatch role for API Gateway.
     *
     * @default true
     */
    readonly cloudWatchRole?: boolean;
    /**
     * Configure a custom domain name and map it to this API.
     *
     * @default - no domain name is defined, use `addDomainName` or directly define a `DomainName`.
     */
    readonly domainName?: DomainNameOptions;
    /**
     * Export name for the CfnOutput containing the API endpoint
     *
     * @default - when no export name is given, output will be created without export
     */
    readonly endpointExportName?: string;
}
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
export declare class RestApi extends Resource implements IRestApi {
    static fromRestApiId(scope: Construct, id: string, restApiId: string): IRestApi;
    /**
     * The ID of this API Gateway RestApi.
     */
    readonly restApiId: string;
    /**
     * The resource ID of the root resource.
     *
     * @attribute
     */
    readonly restApiRootResourceId: string;
    /**
     * Represents the root resource ("/") of this API. Use it to define the API model:
     *
     *    api.root.addMethod('ANY', redirectToHomePage); // "ANY /"
     *    api.root.addResource('friends').addMethod('GET', getFriendsHandler); // "GET /friends"
     *
     */
    readonly root: IResource;
    /**
     * API Gateway stage that points to the latest deployment (if defined).
     *
     * If `deploy` is disabled, you will need to explicitly assign this value in order to
     * set up integrations.
     */
    deploymentStage: Stage;
    /**
     * The list of methods bound to this RestApi
     */
    readonly methods: Method[];
    private _domainName?;
    private _latestDeployment;
    constructor(scope: Construct, id: string, props?: RestApiProps);
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName(): DomainName | undefined;
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment(): Deployment | undefined;
    /**
     * The deployed root URL of this REST API.
     */
    get url(): string;
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path?: string): string;
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id: string, options: DomainNameOptions): DomainName;
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id: string, props?: UsagePlanProps): UsagePlan;
    /**
     * Add an ApiKey
     */
    addApiKey(id: string): IApiKey;
    /**
     * Adds a new model.
     */
    addModel(id: string, props: ModelOptions): Model;
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id: string, props: RequestValidatorOptions): RequestValidator;
    /**
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method?: string, path?: string, stage?: string): string;
    /**
     * Performs validation of the REST API.
     */
    protected validate(): string[];
    private configureDeployment;
    private configureCloudWatchRole;
    private configureEndpoints;
}
/**
 * The endpoint configuration of a REST API, including VPCs and endpoint types.
 *
 * EndpointConfiguration is a property of the AWS::ApiGateway::RestApi resource.
 */
export interface EndpointConfiguration {
    /**
     * A list of endpoint types of an API or its custom domain name.
     *
     * @default - no endpoint types.
     */
    readonly types: EndpointType[];
    /**
     * A list of VPC Endpoints against which to create Route53 ALIASes
     *
     * @default - no ALIASes are created for the endpoint.
     */
    readonly vpcEndpoints?: IVpcEndpoint[];
}
export declare enum ApiKeySourceType {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    HEADER = "HEADER",
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    AUTHORIZER = "AUTHORIZER"
}
export declare enum EndpointType {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EDGE = "EDGE",
    /**
     * For a regional API and its custom domain name.
     */
    REGIONAL = "REGIONAL",
    /**
     * For a private API and its custom domain name.
     */
    PRIVATE = "PRIVATE"
}
