"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 */
class RestApi extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.restApiName || id,
        });
        /**
         * The list of methods bound to this RestApi
         */
        this.methods = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this.configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.configureDeployment(props);
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
    }
    static fromRestApiId(scope, id, restApiId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName`
     * configuration prop, or added via `addDomainName`
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The deployed root URL of this REST API.
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     * @param id The construct id
     * @param options custom domain options
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Add an ApiKey
     */
    addApiKey(id) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this]
        });
    }
    /**
     * Adds a new model.
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this
        });
    }
    /**
     * Adds a new request validator.
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this
        });
    }
    /**
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @param method The method (default `*`)
     * @param path The resource path. Must start with '/' (default `*`)
     * @param stage The stage (default `*`)
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
    }
    /**
     * Performs validation of the REST API.
     */
    validate() {
        if (this.methods.length === 0) {
            return [`The REST API doesn't contain any methods`];
        }
        return [];
    }
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error(`Cannot set 'deployOptions' if 'deploy' is disabled`);
            }
        }
    }
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn
        });
        resource.node.addDependency(apiResource);
    }
    configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId)
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApi = RestApi;
var ApiKeySourceType;
(function (ApiKeySourceType) {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     */
    ApiKeySourceType["HEADER"] = "HEADER";
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     */
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
var EndpointType;
(function (EndpointType) {
    /**
     * For an edge-optimized API and its custom domain name.
     */
    EndpointType["EDGE"] = "EDGE";
    /**
     * For a regional API and its custom domain name.
     */
    EndpointType["REGIONAL"] = "REGIONAL";
    /**
     * For a private API and its custom domain name.
     */
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.restApi = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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