"use strict";
// tslint:disable:object-literal-key-quotes
const assert_1 = require("@aws-cdk/assert");
const acm = require("@aws-cdk/aws-certificatemanager");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'can define either an EDGE or REGIONAL domain name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        const regionalDomain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.REGIONAL
        });
        const edgeDomain = new apigw.DomainName(stack, 'your-domain', {
            domainName: 'example.com',
            certificate: cert,
            endpointType: apigw.EndpointType.EDGE
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "example.com",
            "EndpointConfiguration": { "Types": ["REGIONAL"] },
            "RegionalCertificateArn": { "Ref": "Cert5C9FAEC1" }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "example.com",
            "EndpointConfiguration": { "Types": ["EDGE"] },
            "CertificateArn": { "Ref": "Cert5C9FAEC1" }
        }));
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalDomainName'] });
        test.deepEqual(stack.resolve(regionalDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['mydomain592C948B', 'RegionalHostedZoneId'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasDomainName), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionDomainName'] });
        test.deepEqual(stack.resolve(edgeDomain.domainNameAliasHostedZoneId), { 'Fn::GetAtt': ['yourdomain5FE30C81', 'DistributionHostedZoneId'] });
        test.done();
    },
    'default endpoint type is REGIONAL'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: cert,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "example.com",
            "EndpointConfiguration": { "Types": ["REGIONAL"] },
            "RegionalCertificateArn": { "Ref": "Cert5C9FAEC1" }
        }));
        test.done();
    },
    'accepts different security policies'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cert = new acm.Certificate(stack, 'Cert', { domainName: 'example.com' });
        // WHEN
        new apigw.DomainName(stack, 'my-domain', {
            domainName: 'old.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_0
        });
        new apigw.DomainName(stack, 'your-domain', {
            domainName: 'new.example.com',
            certificate: cert,
            securityPolicy: apigw.SecurityPolicy.TLS_1_2
        });
        new apigw.DomainName(stack, 'default-domain', {
            domainName: 'default.example.com',
            certificate: cert
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "old.example.com",
            "EndpointConfiguration": { "Types": ["REGIONAL"] },
            "RegionalCertificateArn": { "Ref": "Cert5C9FAEC1" },
            "SecurityPolicy": "TLS_1_0"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "new.example.com",
            "EndpointConfiguration": { "Types": ["REGIONAL"] },
            "RegionalCertificateArn": { "Ref": "Cert5C9FAEC1" },
            "SecurityPolicy": "TLS_1_2"
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "default.example.com",
            "EndpointConfiguration": { "Types": ["REGIONAL"] },
            "RegionalCertificateArn": { "Ref": "Cert5C9FAEC1" },
            "SecurityPolicy": assert_1.ABSENT
        }));
        test.done();
    },
    '"mapping" can be used to automatically map this domain to the deployment stage of an API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        new apigw.DomainName(stack, 'Domain', {
            domainName: 'foo.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.EDGE,
            mapping: api
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "Domain66AC69E0"
            },
            "RestApiId": {
                "Ref": "apiC8550315"
            },
            "Stage": {
                "Ref": "apiDeploymentStageprod896C8101"
            }
        }));
        test.done();
    },
    '"addBasePathMapping" can be used to add base path mapping to the domain'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api1 = new apigw.RestApi(stack, 'api1');
        const api2 = new apigw.RestApi(stack, 'api2');
        const domain = new apigw.DomainName(stack, 'my-domain', {
            domainName: 'example.com',
            certificate: acm.Certificate.fromCertificateArn(stack, 'cert', 'arn:aws:acm:us-east-1:1111111:certificate/11-3336f1-44483d-adc7-9cd375c5169d'),
            endpointType: apigw.EndpointType.REGIONAL
        });
        api1.root.addMethod('GET');
        api2.root.addMethod('GET');
        // WHEN
        domain.addBasePathMapping(api1, { basePath: 'api1' });
        domain.addBasePathMapping(api2, { basePath: 'api2' });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "mydomain592C948B"
            },
            "BasePath": "api1",
            "RestApiId": {
                "Ref": "api1A91238E2"
            },
            "Stage": {
                "Ref": "api1DeploymentStageprod362746F6"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "mydomain592C948B"
            },
            "BasePath": "api2",
            "RestApiId": {
                "Ref": "api2C4850CEA"
            },
            "Stage": {
                "Ref": "api2DeploymentStageprod4120D74E"
            }
        }));
        test.done();
    },
    'a domain name can be defined with the API'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            domainName: { domainName, certificate }
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": "my.domain.com",
            "EndpointConfiguration": {
                "Types": [
                    "REGIONAL"
                ]
            },
            "RegionalCertificateArn": {
                "Ref": "cert56CA94EB"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "apiCustomDomain64773C4F"
            },
            "RestApiId": {
                "Ref": "apiC8550315"
            },
            "Stage": {
                "Ref": "apiDeploymentStageprod896C8101"
            }
        }));
        test.done();
    },
    'a domain name can be added later'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        api.addDomainName('domainId', { domainName, certificate });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": domainName,
            "EndpointConfiguration": {
                "Types": [
                    "REGIONAL"
                ]
            },
            "RegionalCertificateArn": {
                "Ref": "cert56CA94EB"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "apidomainId102F8DAA"
            },
            "RestApiId": {
                "Ref": "apiC8550315"
            },
            "Stage": {
                "Ref": "apiDeploymentStageprod896C8101"
            }
        }));
        test.done();
    },
    'multiple domain names can be added'(test) {
        // GIVEN
        const domainName = 'my.domain.com';
        const stack = new core_1.Stack();
        const certificate = new acm.Certificate(stack, 'cert', { domainName: 'my.domain.com' });
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {});
        api.root.addMethod('GET');
        const domainName1 = api.addDomainName('domainId', { domainName, certificate });
        api.addDomainName('domainId1', { domainName: 'your.domain.com', certificate });
        api.addDomainName('domainId2', { domainName: 'our.domain.com', certificate });
        test.deepEqual(api.domainName, domainName1);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": 'my.domain.com',
            "EndpointConfiguration": {
                "Types": [
                    "REGIONAL"
                ]
            },
            "RegionalCertificateArn": {
                "Ref": "cert56CA94EB"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": 'your.domain.com',
            "EndpointConfiguration": {
                "Types": [
                    "REGIONAL"
                ]
            },
            "RegionalCertificateArn": {
                "Ref": "cert56CA94EB"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::DomainName', {
            "DomainName": 'our.domain.com',
            "EndpointConfiguration": {
                "Types": [
                    "REGIONAL"
                ]
            },
            "RegionalCertificateArn": {
                "Ref": "cert56CA94EB"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::BasePathMapping', {
            "DomainName": {
                "Ref": "apidomainId102F8DAA"
            },
            "RestApiId": {
                "Ref": "apiC8550315"
            },
            "Stage": {
                "Ref": "apiDeploymentStageprod896C8101"
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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