"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const apigateway = require("../lib");
const API_KEY_RESOURCE_TYPE = 'AWS::ApiGateway::ApiKey';
const USAGE_PLAN_RESOURCE_TYPE = 'AWS::ApiGateway::UsagePlan';
const USAGE_PLAN_KEY_RESOURCE_TYPE = 'AWS::ApiGateway::UsagePlanKey';
module.exports = {
    'default setup'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'my-api', { cloudWatchRole: false, deploy: false });
        api.root.addMethod('GET'); // Need at least one method on the api
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'my-api-key');
        // THEN
        // should have an api key with no props defined.
        assert_1.expect(stack).to(assert_1.haveResource(API_KEY_RESOURCE_TYPE, undefined, assert_1.ResourcePart.CompleteDefinition));
        // should not have a usage plan.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE));
        // should not have a usage plan key.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE));
        test.done();
    },
    'only api key is created when rate limiting properties are not provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: true, deployOptions: { stageName: 'test' } });
        api.root.addMethod('GET'); // api must have atleast one method.
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'test-api-key', {
            customerId: 'test-customer',
            resources: [api]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            CustomerId: 'test-customer',
            StageKeys: [
                {
                    RestApiId: { Ref: "testapiD6451F70" },
                    StageName: { Ref: "testapiDeploymentStagetest5869DF71" }
                }
            ]
        }));
        // should not have a usage plan.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE));
        // should not have a usage plan key.
        assert_1.expect(stack).notTo(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE));
        test.done();
    },
    'api key and usage plan are created and linked when rate limiting properties are provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const api = new apigateway.RestApi(stack, 'test-api', { cloudWatchRole: false, deploy: true, deployOptions: { stageName: 'test' } });
        api.root.addMethod('GET'); // api must have atleast one method.
        // WHEN
        new apigateway.RateLimitedApiKey(stack, 'test-api-key', {
            customerId: 'test-customer',
            resources: [api],
            quota: {
                limit: 10000,
                period: apigateway.Period.MONTH
            }
        });
        // THEN
        // should have an api key
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::ApiKey', {
            CustomerId: 'test-customer',
            StageKeys: [
                {
                    RestApiId: { Ref: "testapiD6451F70" },
                    StageName: { Ref: "testapiDeploymentStagetest5869DF71" }
                }
            ]
        }));
        // should have a usage plan with specified quota.
        assert_1.expect(stack).to(assert_1.haveResource(USAGE_PLAN_RESOURCE_TYPE, {
            Quota: {
                Limit: 10000,
                Period: 'MONTH'
            }
        }, assert_1.ResourcePart.Properties));
        // should have a usage plan key linking the api key and usage plan
        assert_1.expect(stack).to(assert_1.haveResource(USAGE_PLAN_KEY_RESOURCE_TYPE, {
            KeyId: {
                Ref: 'testapikey998028B6'
            },
            KeyType: 'API_KEY',
            UsagePlanId: {
                Ref: 'testapikeyUsagePlanResource66DB63D6'
            }
        }, assert_1.ResourcePart.Properties));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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