"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'ProxyResource defines a "{proxy+}" resource with ANY method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        new apigw.ProxyResource(stack, 'proxy', {
            parent: api.root,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            "ParentId": {
                "Fn::GetAtt": [
                    "apiC8550315",
                    "RootResourceId"
                ]
            },
            "PathPart": "{proxy+}",
            "RestApiId": {
                "Ref": "apiC8550315"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            "HttpMethod": "ANY",
            "ResourceId": {
                "Ref": "proxy3A1DA9C7"
            },
            "RestApiId": {
                "Ref": "apiC8550315"
            },
            "AuthorizationType": "NONE",
            "Integration": {
                "Type": "MOCK"
            }
        }));
        test.done();
    },
    'if "anyMethod" is false, then an ANY method will not be defined'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        const proxy = new apigw.ProxyResource(stack, 'proxy', {
            parent: api.root,
            anyMethod: false
        });
        proxy.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', { "HttpMethod": "GET" }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ApiGateway::Method', { "HttpMethod": "ANY" }));
        test.done();
    },
    'addProxy can be used on any resource to attach a proxy from that route'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', {
            deploy: false,
            cloudWatchRole: false,
        });
        const v2 = api.root.addResource('v2');
        v2.addProxy();
        assert_1.expect(stack).toMatch({
            "Resources": {
                "apiC8550315": {
                    "Type": "AWS::ApiGateway::RestApi",
                    "Properties": {
                        "Name": "api"
                    }
                },
                "apiv25206B108": {
                    "Type": "AWS::ApiGateway::Resource",
                    "Properties": {
                        "ParentId": {
                            "Fn::GetAtt": [
                                "apiC8550315",
                                "RootResourceId"
                            ]
                        },
                        "PathPart": "v2",
                        "RestApiId": {
                            "Ref": "apiC8550315"
                        }
                    }
                },
                "apiv2proxyAEA4DAC8": {
                    "Type": "AWS::ApiGateway::Resource",
                    "Properties": {
                        "ParentId": {
                            "Ref": "apiv25206B108"
                        },
                        "PathPart": "{proxy+}",
                        "RestApiId": {
                            "Ref": "apiC8550315"
                        }
                    }
                },
                "apiv2proxyANY889F4CE1": {
                    "Type": "AWS::ApiGateway::Method",
                    "Properties": {
                        "HttpMethod": "ANY",
                        "ResourceId": {
                            "Ref": "apiv2proxyAEA4DAC8"
                        },
                        "RestApiId": {
                            "Ref": "apiC8550315"
                        },
                        "AuthorizationType": "NONE",
                        "Integration": {
                            "Type": "MOCK"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'if proxy is added to root, proxy methods are automatically duplicated (with integration and options)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const proxy = api.root.addProxy({
            anyMethod: false
        });
        const deleteInteg = new apigw.MockIntegration({
            requestParameters: {
                foo: 'bar'
            }
        });
        // WHEN
        proxy.addMethod('DELETE', deleteInteg, {
            operationName: 'DeleteMe'
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            ResourceId: { Ref: 'apiproxy4EA44110' },
            Integration: {
                RequestParameters: { foo: "bar" },
                Type: 'MOCK'
            },
            OperationName: 'DeleteMe'
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            ResourceId: { "Fn::GetAtt": ["apiC8550315", "RootResourceId"] },
            Integration: {
                RequestParameters: { foo: "bar" },
                Type: 'MOCK'
            },
            OperationName: 'DeleteMe'
        }));
        test.done();
    },
    'if proxy is added to root, proxy methods are only added if they are not defined already on the root resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('POST');
        const proxy = api.root.addProxy({ anyMethod: false });
        // WHEN
        proxy.addMethod('POST');
        // THEN
        test.done();
    },
    'url for a resource'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        // WHEN
        const aResource = api.root.addResource('a');
        const cResource = aResource.addResource('b').addResource('c');
        // THEN
        test.deepEqual(stack.resolve(aResource.url), {
            'Fn::Join': [
                '',
                [
                    'https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/a'
                ]
            ]
        });
        test.deepEqual(stack.resolve(cResource.url), {
            'Fn::Join': [
                '',
                [
                    'https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    '.',
                    { Ref: 'AWS::URLSuffix' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/a/b/c'
                ]
            ]
        });
        test.done();
    },
    'getResource': {
        'root resource': {
            'returns undefined if not found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // THEN
                test.deepEqual(api.root.getResource('boom'), undefined);
                test.done();
            },
            'returns the resource if found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const r1 = api.root.addResource('hello');
                const r2 = api.root.addResource('world');
                // THEN
                test.deepEqual(api.root.getResource('hello'), r1);
                test.deepEqual(api.root.getResource('world'), r2);
                test.done();
            },
            'returns the resource even if it was created using "new"'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const r1 = new apigw.Resource(stack, 'child', {
                    parent: api.root,
                    pathPart: 'yello'
                });
                // THEN
                test.deepEqual(api.root.getResource('yello'), r1);
                test.done();
            }
        },
        'non-root': {
            'returns undefined if not found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const res = api.root.addResource('boom');
                // THEN
                test.deepEqual(res.getResource('child-of-boom'), undefined);
                test.done();
            },
            'returns the resource if found'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const child = api.root.addResource('boom');
                // WHEN
                const r1 = child.addResource('hello');
                const r2 = child.addResource('world');
                // THEN
                test.deepEqual(child.getResource('hello'), r1);
                test.deepEqual(child.getResource('world'), r2);
                test.done();
            },
            'returns the resource even if created with "new"'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                const child = api.root.addResource('boom');
                // WHEN
                const r1 = child.addResource('hello');
                const r2 = new apigw.Resource(stack, 'world', {
                    parent: child,
                    pathPart: 'outside-world'
                });
                // THEN
                test.deepEqual(child.getResource('hello'), r1);
                test.deepEqual(child.getResource('outside-world'), r2);
                test.done();
            }
        },
        'resourceForPath': {
            'empty path or "/" (on root) returns this'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // THEN
                test.deepEqual(api.root.resourceForPath(''), api.root);
                test.deepEqual(api.root.resourceForPath('/'), api.root);
                test.deepEqual(api.root.resourceForPath('///'), api.root);
                test.done();
            },
            'returns a resource for that path'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const resource = api.root.resourceForPath('/boom/trach');
                // THEN
                test.deepEqual(resource.path, '/boom/trach');
                test.done();
            },
            'resources not created if not needed'(test) {
                // GIVEN
                const stack = new core_1.Stack();
                const api = new apigw.RestApi(stack, 'MyRestApi');
                // WHEN
                const trach = api.root.resourceForPath('/boom/trach');
                const bam1 = api.root.resourceForPath('/boom/bam');
                // THEN
                const parent = api.root.getResource('boom');
                test.ok(parent);
                test.deepEqual(parent.path, '/boom');
                test.same(trach.parentResource, parent);
                test.deepEqual(trach.parentResource.path, '/boom');
                const bam2 = api.root.resourceForPath('/boom/bam');
                test.same(bam1, bam2);
                test.deepEqual(bam1.parentResource.path, '/boom');
                test.done();
            }
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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