"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'minimal setup'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'my-api');
        api.root.addMethod('GET'); // must have at least one method
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                myapi4C7BF186: {
                    Type: "AWS::ApiGateway::RestApi",
                    Properties: {
                        Name: "my-api"
                    }
                },
                myapiGETF990CE3C: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "GET",
                        ResourceId: { "Fn::GetAtt": ["myapi4C7BF186", "RootResourceId"] },
                        RestApiId: { Ref: "myapi4C7BF186" },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                },
                myapiDeployment92F2CB49916eaecf87f818f1e175215b8d086029: {
                    Type: "AWS::ApiGateway::Deployment",
                    Properties: {
                        RestApiId: { Ref: "myapi4C7BF186" },
                        Description: "Automatically created by the RestApi construct"
                    },
                    DependsOn: ["myapiGETF990CE3C"]
                },
                myapiDeploymentStageprod298F01AF: {
                    Type: "AWS::ApiGateway::Stage",
                    Properties: {
                        RestApiId: { Ref: "myapi4C7BF186" },
                        DeploymentId: { Ref: "myapiDeployment92F2CB49916eaecf87f818f1e175215b8d086029" },
                        StageName: "prod"
                    }
                },
                myapiCloudWatchRole095452E5: {
                    Type: "AWS::IAM::Role",
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: "sts:AssumeRole",
                                    Effect: "Allow",
                                    Principal: { Service: "apigateway.amazonaws.com" }
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        ManagedPolicyArns: [
                            { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs"]] }
                        ]
                    }
                },
                myapiAccountEC421A0A: {
                    Type: "AWS::ApiGateway::Account",
                    Properties: {
                        CloudWatchRoleArn: { "Fn::GetAtt": ["myapiCloudWatchRole095452E5", "Arn"] }
                    },
                    DependsOn: ["myapi4C7BF186"]
                }
            },
            Outputs: {
                myapiEndpoint3628AFE3: {
                    Value: {
                        "Fn::Join": ["", [
                                "https://",
                                { Ref: "myapi4C7BF186" },
                                ".execute-api.",
                                { Ref: "AWS::Region" },
                                ".",
                                { Ref: "AWS::URLSuffix" },
                                "/",
                                { Ref: "myapiDeploymentStageprod298F01AF" },
                                "/"
                            ]]
                    }
                }
            }
        });
        test.done();
    },
    'defaultChild is set correctly'(test) {
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'my-api');
        test.ok(api.node.defaultChild instanceof apigw.CfnRestApi);
        test.done();
    },
    '"name" is defaulted to resource physical name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            Name: 'restapi'
        }));
        test.done();
    },
    'fails in synthesis if there are no methods'(test) {
        // GIVEN
        const app = new core_1.App();
        const stack = new core_1.Stack(app, 'my-stack');
        const api = new apigw.RestApi(stack, 'API');
        // WHEN
        api.root.addResource('foo');
        api.root.addResource('bar').addResource('goo');
        // THEN
        test.throws(() => app.synth(), /The REST API doesn't contain any methods/);
        test.done();
    },
    '"addResource" can be used on "IRestApiResource" to form a tree'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        api.root.addMethod('GET');
        // WHEN
        const foo = api.root.addResource('foo');
        api.root.addResource('bar');
        foo.addResource('{hello}');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: "foo",
            ParentId: { "Fn::GetAtt": ["restapiC5611D27", "RootResourceId"] }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: "bar",
            ParentId: { "Fn::GetAtt": ["restapiC5611D27", "RootResourceId"] }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: "{hello}",
            ParentId: { Ref: "restapifooF697E056" }
        }));
        test.done();
    },
    '"addResource" allows configuration of proxy paths'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', {
            deploy: false,
            cloudWatchRole: false,
            restApiName: 'my-rest-api',
        });
        // WHEN
        const proxy = api.root.addResource('{proxy+}');
        proxy.addMethod('ANY');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: "{proxy+}",
            ParentId: { "Fn::GetAtt": ["restapiC5611D27", "RootResourceId"] }
        }));
        test.done();
    },
    '"addMethod" can be used to add methods to resources'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi', { deploy: false, cloudWatchRole: false });
        const r1 = api.root.addResource('r1');
        // WHEN
        api.root.addMethod('GET');
        r1.addMethod('POST');
        // THEN
        assert_1.expect(stack).toMatch({
            Resources: {
                restapiC5611D27: {
                    Type: "AWS::ApiGateway::RestApi",
                    Properties: {
                        Name: "restapi"
                    }
                },
                restapir1CF2997EA: {
                    Type: "AWS::ApiGateway::Resource",
                    Properties: {
                        ParentId: {
                            "Fn::GetAtt": [
                                "restapiC5611D27",
                                "RootResourceId"
                            ]
                        },
                        PathPart: "r1",
                        RestApiId: {
                            Ref: "restapiC5611D27"
                        }
                    }
                },
                restapir1POST766920C4: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "POST",
                        ResourceId: {
                            Ref: "restapir1CF2997EA"
                        },
                        RestApiId: {
                            Ref: "restapiC5611D27"
                        },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                },
                restapiGET6FC1785A: {
                    Type: "AWS::ApiGateway::Method",
                    Properties: {
                        HttpMethod: "GET",
                        ResourceId: {
                            "Fn::GetAtt": [
                                "restapiC5611D27",
                                "RootResourceId"
                            ]
                        },
                        RestApiId: {
                            Ref: "restapiC5611D27"
                        },
                        AuthorizationType: "NONE",
                        Integration: {
                            Type: "MOCK"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'resourcePath returns the full path of the resource within the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // WHEN
        const r1 = api.root.addResource('r1');
        const r11 = r1.addResource('r1_1');
        const r12 = r1.addResource('r1_2');
        const r121 = r12.addResource('r1_2_1');
        const r2 = api.root.addResource('r2');
        // THEN
        test.deepEqual(api.root.path, '/');
        test.deepEqual(r1.path, '/r1');
        test.deepEqual(r11.path, '/r1/r1_1');
        test.deepEqual(r12.path, '/r1/r1_2');
        test.deepEqual(r121.path, '/r1/r1_2/r1_2_1');
        test.deepEqual(r2.path, '/r2');
        test.done();
    },
    'resource path part validation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'restapi');
        // THEN
        test.throws(() => api.root.addResource('foo/'));
        api.root.addResource('boom-bam');
        test.throws(() => api.root.addResource('illegal()'));
        api.root.addResource('{foo}');
        test.throws(() => api.root.addResource('foo{bar}'));
        test.done();
    },
    'fails if "deployOptions" is set with "deploy" disabled'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'myapi', {
            deploy: false,
            deployOptions: { cachingEnabled: true }
        }), /Cannot set 'deployOptions' if 'deploy' is disabled/);
        test.done();
    },
    'CloudWatch role is created for API Gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Role'));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Account'));
        test.done();
    },
    'fromRestApiId'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const imported = apigw.RestApi.fromRestApiId(stack, 'imported-api', 'api-rxt4498f');
        // THEN
        test.deepEqual(stack.resolve(imported.restApiId), 'api-rxt4498f');
        test.done();
    },
    '"url" and "urlForPath" return the URL endpoints of the deployed API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(stack.resolve(api.url), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    ".",
                    { Ref: "AWS::URLSuffix" },
                    "/",
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/']] });
        test.deepEqual(stack.resolve(api.urlForPath('/foo/bar')), { 'Fn::Join': ['',
                ['https://',
                    { Ref: 'apiC8550315' },
                    '.execute-api.',
                    { Ref: 'AWS::Region' },
                    ".",
                    { Ref: "AWS::URLSuffix" },
                    "/",
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/foo/bar']] });
        test.done();
    },
    '"urlForPath" would not work if there is no deployment'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api', { deploy: false });
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.url, /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.throws(() => api.urlForPath('/foo'), /Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"/);
        test.done();
    },
    '"urlForPath" requires that path will begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.urlForPath('foo'), /Path must begin with \"\/\": foo/);
        test.done();
    },
    '"executeApiArn" returns the execute-api ARN for a resource/method'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // WHEN
        const arn = api.arnForExecuteApi('method', '/path', 'stage');
        // THEN
        test.deepEqual(stack.resolve(arn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/stage/method/path']] });
        test.done();
    },
    '"executeApiArn" path must begin with "/"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        api.root.addMethod('GET');
        // THEN
        test.throws(() => api.arnForExecuteApi('method', 'hey-path', 'stage'), /"path" must begin with a "\/": 'hey-path'/);
        test.done();
    },
    '"executeApiArn" will convert ANY to "*"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'api');
        const method = api.root.addMethod('ANY');
        // THEN
        test.deepEqual(stack.resolve(method.methodArn), { 'Fn::Join': ['',
                ['arn:',
                    { Ref: 'AWS::Partition' },
                    ':execute-api:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':',
                    { Ref: 'apiC8550315' },
                    '/',
                    { Ref: 'apiDeploymentStageprod896C8101' },
                    '/*/']] });
        test.done();
    },
    '"endpointTypes" can be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointTypes: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE]
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    "EDGE",
                    "PRIVATE"
                ]
            }
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify endpoint types for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE]
            }
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: ["EDGE", "PRIVATE"]
            }
        }));
        test.done();
    },
    '"endpointConfiguration" can be used to specify vpc endpoints on the API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.EDGE, apigw.EndpointType.PRIVATE],
                vpcEndpoints: [
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint'),
                    aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint2', 'vpcEndpoint2'),
                ]
            }
        });
        api.root.addMethod('GET');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            EndpointConfiguration: {
                Types: [
                    "EDGE",
                    "PRIVATE"
                ],
                VpcEndpointIds: [
                    "vpcEndpoint",
                    "vpcEndpoint2"
                ]
            }
        }));
        test.done();
    },
    '"endpointTypes" and "endpointConfiguration" can NOT both be used to specify endpoint configuration for the api'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.throws(() => new apigw.RestApi(stack, 'api', {
            endpointConfiguration: {
                types: [apigw.EndpointType.PRIVATE],
                vpcEndpoints: [aws_ec2_1.GatewayVpcEndpoint.fromGatewayVpcEndpointId(stack, 'ImportedEndpoint', 'vpcEndpoint')]
            },
            endpointTypes: [apigw.EndpointType.PRIVATE]
        }), /Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed/);
        test.done();
    },
    '"cloneFrom" can be used to clone an existing API'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const cloneFrom = apigw.RestApi.fromRestApiId(stack, 'RestApi', 'foobar');
        // WHEN
        const api = new apigw.RestApi(stack, 'api', {
            cloneFrom
        });
        api.root.addMethod('GET');
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RestApi', {
            CloneFrom: "foobar",
            Name: "api"
        }));
        test.done();
    },
    'allow taking a dependency on the rest api (includes deployment and stage)'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        const resource = new core_1.CfnResource(stack, 'DependsOnRestApi', { type: 'My::Resource' });
        // WHEN
        resource.node.addDependency(api);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('My::Resource', {
            DependsOn: [
                "myapiAccountC3A4750C",
                "myapiCloudWatchRoleEB425128",
                "myapiGET9B7CD29E",
                "myapiDeploymentB7EF8EB75c091a668064a3f3a1f6d68a3fb22cf9",
                "myapiDeploymentStageprod329F21FF",
                "myapi162F20B8"
            ]
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'defaultIntegration and defaultMethodOptions can be used at any level'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const rootInteg = new apigw.AwsIntegration({
            service: 's3',
            action: 'GetObject'
        });
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', {
            defaultIntegration: rootInteg,
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID' },
                authorizationType: apigw.AuthorizationType.IAM,
            }
        });
        // CASE #1: should inherit integration and options from root resource
        api.root.addMethod('GET');
        const child = api.root.addResource('child');
        // CASE #2: should inherit integration from root and method options, but
        // "authorizationType" will be overridden to "None" instead of "IAM"
        child.addMethod('POST', undefined, {
            authorizationType: apigw.AuthorizationType.COGNITO
        });
        const child2 = api.root.addResource('child2', {
            defaultIntegration: new apigw.MockIntegration(),
            defaultMethodOptions: {
                authorizer: { authorizerId: 'AUTHID2' },
            }
        });
        // CASE #3: integartion and authorizer ID are inherited from child2
        child2.addMethod('DELETE');
        // CASE #4: same as case #3, but integration is customized
        child2.addMethod('PUT', new apigw.AwsIntegration({ action: 'foo', service: 'bar' }));
        // THEN
        // CASE #1
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            ResourceId: { "Fn::GetAtt": ["myapi162F20B8", "RootResourceId"] },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'AWS_IAM',
        }));
        // CASE #2
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            ResourceId: { Ref: "myapichildA0A65412" },
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID',
            AuthorizationType: 'COGNITO_USER_POOLS',
        }));
        // CASE #3
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'DELETE',
            Integration: { Type: 'MOCK' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM'
        }));
        // CASE #4
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApiGateway::Method', {
            HttpMethod: 'PUT',
            Integration: { Type: 'AWS' },
            AuthorizerId: 'AUTHID2',
            AuthorizationType: 'AWS_IAM'
        }));
        test.done();
    },
    'addModel is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addModel('model', {
            schema: {
                schema: apigw.JsonSchemaVersion.DRAFT4,
                title: "test",
                type: apigw.JsonSchemaType.OBJECT,
                properties: { message: { type: apigw.JsonSchemaType.STRING } }
            }
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Model', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Schema: {
                $schema: "http://json-schema.org/draft-04/schema#",
                title: "test",
                type: "object",
                properties: { message: { type: "string" } }
            }
        }));
        test.done();
    },
    'addRequestValidator is supported'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('OPTIONS');
        // WHEN
        api.addRequestValidator('params-validator', {
            requestValidatorName: 'Parameters',
            validateRequestBody: false,
            validateRequestParameters: true
        });
        api.addRequestValidator('body-validator', {
            requestValidatorName: "Body",
            validateRequestBody: true,
            validateRequestParameters: false
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: "Parameters",
            ValidateRequestBody: false,
            ValidateRequestParameters: true
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::RequestValidator', {
            RestApiId: { Ref: stack.getLogicalId(api.node.findChild('Resource')) },
            Name: "Body",
            ValidateRequestBody: true,
            ValidateRequestParameters: false
        }));
        test.done();
    },
    'creates output with given "exportName"'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi', { endpointExportName: 'my-given-export-name' });
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/'
                        ]
                    ]
                },
                Export: { Name: 'my-given-export-name' }
            }
        });
        test.done();
    },
    'creates output when "exportName" is not specified'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const api = new apigw.RestApi(stack, 'myapi');
        api.root.addMethod('GET');
        // THEN
        test.deepEqual(assert_1.SynthUtils.toCloudFormation(stack).Outputs, {
            myapiEndpoint8EB17201: {
                Value: {
                    'Fn::Join': [
                        '',
                        [
                            'https://',
                            { Ref: 'myapi162F20B8' },
                            '.execute-api.',
                            { Ref: 'AWS::Region' },
                            '.',
                            { Ref: 'AWS::URLSuffix' },
                            '/',
                            { Ref: 'myapiDeploymentStageprod329F21FF' },
                            '/'
                        ]
                    ]
                }
            }
        });
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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