"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessLogFormat = exports.AccessLogField = exports.LogGroupLogDestination = void 0;
/**
 * Use CloudWatch Logs as a custom access log destination for API Gateway.
 *
 * @stability stable
 */
class LogGroupLogDestination {
    /**
     * @stability stable
     */
    constructor(logGroup) {
        this.logGroup = logGroup;
    }
    /**
     * Binds this destination to the CloudWatch Logs.
     *
     * @stability stable
     */
    bind(_stage) {
        return {
            destinationArn: this.logGroup.logGroupArn,
        };
    }
}
exports.LogGroupLogDestination = LogGroupLogDestination;
/**
 * $context variables that can be used to customize access log pattern.
 *
 * @stability stable
 */
class AccessLogField {
    /**
     * The API owner's AWS account ID.
     *
     * @stability stable
     */
    static contextAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * The identifier API Gateway assigns to your API.
     *
     * @stability stable
     */
    static contextApiId() {
        return '$context.apiId';
    }
    /**
     * A property of the claims returned from the Amazon Cognito user pool after the method caller is successfully authenticated.
     *
     * @param property A property key of the claims.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-integrate-with-cognito.html
     * @stability stable
     */
    static contextAuthorizerClaims(property) {
        return `$context.authorizer.claims.${property}`;
    }
    /**
     * The principal user identification associated with the token sent by the client and returned from an API Gateway Lambda authorizer (formerly known as a custom authorizer).
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @stability stable
     */
    static contextAuthorizerPrincipalId() {
        return '$context.authorizer.principalId';
    }
    /**
     * The stringified value of the specified key-value pair of the `context` map returned from an API Gateway Lambda authorizer function.
     *
     * @param property key of the context map.
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-use-lambda-authorizer.html
     * @stability stable
     */
    static contextAuthorizer(property) {
        return `$context.authorizer.${property}`;
    }
    /**
     * The AWS endpoint's request ID.
     *
     * @stability stable
     */
    static contextAwsEndpointRequestId() {
        return '$context.awsEndpointRequestId';
    }
    /**
     * The full domain name used to invoke the API.
     *
     * This should be the same as the incoming `Host` header.
     *
     * @stability stable
     */
    static contextDomainName() {
        return '$context.domainName';
    }
    /**
     * The first label of the `$context.domainName`. This is often used as a caller/customer identifier.
     *
     * @stability stable
     */
    static contextDomainPrefix() {
        return '$context.domainPrefix';
    }
    /**
     * A string containing an API Gateway error message.
     *
     * @stability stable
     */
    static contextErrorMessage() {
        return '$context.error.message';
    }
    /**
     * The quoted value of $context.error.message, namely "$context.error.message".
     *
     * @stability stable
     */
    static contextErrorMessageString() {
        return '$context.error.messageString';
    }
    /**
     * A type of GatewayResponse.
     *
     * This variable can only be used for simple variable substitution in a GatewayResponse body-mapping template,
     * which is not processed by the Velocity Template Language engine, and in access logging.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/customize-gateway-responses.html
     * @stability stable
     */
    static contextErrorResponseType() {
        return '$context.error.responseType';
    }
    /**
     * A string containing a detailed validation error message.
     *
     * @stability stable
     */
    static contextErrorValidationErrorString() {
        return '$context.error.validationErrorString';
    }
    /**
     * The extended ID that API Gateway assigns to the API request, which contains more useful information for debugging/troubleshooting.
     *
     * @stability stable
     */
    static contextExtendedRequestId() {
        return '$context.extendedRequestId';
    }
    /**
     * The HTTP method used.
     *
     * Valid values include: `DELETE`, `GET`, `HEAD`, `OPTIONS`, `PATCH`, `POST`, and `PUT`.
     *
     * @stability stable
     */
    static contextHttpMethod() {
        return '$context.httpMethod';
    }
    /**
     * The AWS account ID associated with the request.
     *
     * @stability stable
     */
    static contextIdentityAccountId() {
        return '$context.identity.accountId';
    }
    /**
     * For API methods that require an API key, this variable is the API key associated with the method request.
     *
     * For methods that don't require an API key, this variable is
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-api-usage-plans.html
     * @stability stable
     */
    static contextIdentityApiKey() {
        return '$context.identity.apiKey';
    }
    /**
     * The API key ID associated with an API request that requires an API key.
     *
     * @stability stable
     */
    static contextIdentityApiKeyId() {
        return '$context.identity.apiKeyId';
    }
    /**
     * The principal identifier of the caller making the request.
     *
     * @stability stable
     */
    static contextIdentityCaller() {
        return '$context.identity.caller';
    }
    /**
     * The Amazon Cognito authentication provider used by the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-identity.html
     * @stability stable
     */
    static contextIdentityCognitoAuthenticationProvider() {
        return '$context.identity.cognitoAuthenticationProvider';
    }
    /**
     * The Amazon Cognito authentication type of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoAuthenticationType() {
        return '$context.identity.cognitoAuthenticationType';
    }
    /**
     * The Amazon Cognito identity ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoIdentityId() {
        return '$context.identity.cognitoIdentityId';
    }
    /**
     * The Amazon Cognito identity pool ID of the caller making the request.
     *
     * Available only if the request was signed with Amazon Cognito credentials.
     *
     * @stability stable
     */
    static contextIdentityCognitoIdentityPoolId() {
        return '$context.identity.cognitoIdentityPoolId';
    }
    /**
     * The AWS organization ID.
     *
     * @stability stable
     */
    static contextIdentityPrincipalOrgId() {
        return '$context.identity.principalOrgId';
    }
    /**
     * The source IP address of the TCP connection making the request to API Gateway.
     *
     * Warning: You should not trust this value if there is any chance that the `X-Forwarded-For` header could be forged.
     *
     * @stability stable
     */
    static contextIdentitySourceIp() {
        return '$context.identity.sourceIp';
    }
    /**
     * The principal identifier of the user making the request.
     *
     * Used in Lambda authorizers.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-lambda-authorizer-output.html
     * @stability stable
     */
    static contextIdentityUser() {
        return '$context.identity.user';
    }
    /**
     * The User-Agent header of the API caller.
     *
     * @stability stable
     */
    static contextIdentityUserAgent() {
        return '$context.identity.userAgent';
    }
    /**
     * The Amazon Resource Name (ARN) of the effective user identified after authentication.
     *
     * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
     * @stability stable
     */
    static contextIdentityUserArn() {
        return '$context.identity.userArn';
    }
    /**
     * The request path.
     *
     * For example, for a non-proxy request URL of https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child,
     * this value is /{stage}/root/child.
     *
     * @stability stable
     */
    static contextPath() {
        return '$context.path';
    }
    /**
     * The request protocol, for example, HTTP/1.1.
     *
     * @stability stable
     */
    static contextProtocol() {
        return '$context.protocol';
    }
    /**
     * The ID that API Gateway assigns to the API request.
     *
     * @stability stable
     */
    static contextRequestId() {
        return '$context.requestId';
    }
    /**
     * The request header override.
     *
     * If this parameter is defined, it contains the headers to be used instead of the HTTP Headers that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextRequestOverrideHeader(headerName) {
        return `$context.requestOverride.header.${headerName}`;
    }
    /**
     * The request path override.
     *
     * If this parameter is defined,
     * it contains the request path to be used instead of the URL Path Parameters that are defined in the Integration Request pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextRequestOverridePath(pathName) {
        return `$context.requestOverride.path.${pathName}`;
    }
    /**
     * The request query string override.
     *
     * If this parameter is defined, it contains the request query strings to be used instead
     * of the URL Query String Parameters that are defined in the Integration Request pane.
     *
     * @stability stable
     */
    static contextRequestOverrideQuerystring(querystringName) {
        return `$context.requestOverride.querystring.${querystringName}`;
    }
    /**
     * The response header override.
     *
     * If this parameter is defined, it contains the header to be returned instead of the Response header
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextResponseOverrideHeader(headerName) {
        return `$context.responseOverride.header.${headerName}`;
    }
    /**
     * The response status code override.
     *
     * If this parameter is defined, it contains the status code to be returned instead of the Method response status
     * that is defined as the Default mapping in the Integration Response pane.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-override-request-response-parameters.html
     * @stability stable
     */
    static contextResponseOverrideStatus() {
        return '$context.responseOverride.status';
    }
    /**
     * The CLF-formatted request time (dd/MMM/yyyy:HH:mm:ss +-hhmm).
     *
     * @stability stable
     */
    static contextRequestTime() {
        return '$context.requestTime';
    }
    /**
     * The Epoch-formatted request time.
     *
     * @stability stable
     */
    static contextRequestTimeEpoch() {
        return '$context.requestTimeEpoch';
    }
    /**
     * The identifier that API Gateway assigns to your resource.
     *
     * @stability stable
     */
    static contextResourceId() {
        return '$context.resourceId';
    }
    /**
     * The path to your resource.
     *
     * For example, for the non-proxy request URI of `https://{rest-api-id.execute-api.{region}.amazonaws.com/{stage}/root/child`,
     * The $context.resourcePath value is `/root/child`.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-create-api-step-by-step.html
     * @stability stable
     */
    static contextResourcePath() {
        return '$context.resourcePath';
    }
    /**
     * The deployment stage of the API request (for example, `Beta` or `Prod`).
     *
     * @stability stable
     */
    static contextStage() {
        return '$context.stage';
    }
    /**
     * The response received from AWS WAF: `WAF_ALLOW` or `WAF_BLOCK`.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     * @stability stable
     */
    static contextWafResponseCode() {
        return '$context.wafResponseCode';
    }
    /**
     * The complete ARN of the web ACL that is used to decide whether to allow or block the request.
     *
     * Will not be set if the stage is not associated with a web ACL.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-control-access-aws-waf.html
     * @stability stable
     */
    static contextWebaclArn() {
        return '$context.webaclArn';
    }
    /**
     * The trace ID for the X-Ray trace.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/apigateway-enabling-xray.html
     * @stability stable
     */
    static contextXrayTraceId() {
        return '$context.xrayTraceId';
    }
    /**
     * The authorizer latency in ms.
     *
     * @stability stable
     */
    static contextAuthorizerIntegrationLatency() {
        return '$context.authorizer.integrationLatency';
    }
    /**
     * The integration latency in ms.
     *
     * @stability stable
     */
    static contextIntegrationLatency() {
        return '$context.integrationLatency';
    }
    /**
     * For Lambda proxy integration, this parameter represents the status code returned from AWS Lambda, not from the backend Lambda function.
     *
     * @stability stable
     */
    static contextIntegrationStatus() {
        return '$context.integrationStatus.';
    }
    /**
     * The response latency in ms.
     *
     * @stability stable
     */
    static contextResponseLatency() {
        return '$context.responseLatency';
    }
    /**
     * The response payload length.
     *
     * @stability stable
     */
    static contextResponseLength() {
        return '$context.responseLength';
    }
    /**
     * The method response status.
     *
     * @stability stable
     */
    static contextStatus() {
        return '$context.status';
    }
}
exports.AccessLogField = AccessLogField;
/**
 * factory methods for access log format.
 *
 * @stability stable
 */
class AccessLogFormat {
    constructor(format) {
        this.format = format;
    }
    /**
     * Custom log format.
     *
     * You can create any log format string. You can easily get the $ context variable by using the methods of AccessLogField.
     *
     * @stability stable
     * @example
     *
     * custom(JSON.stringify({
     * requestId: AccessLogField.contextRequestId(),
     * sourceIp: AccessLogField.contextIdentitySourceIp(),
     * method: AccessLogFiled.contextHttpMethod(),
     * userContext: {
     * sub: AccessLogField.contextAuthorizerClaims('sub'),
     * email: AccessLogField.contextAuthorizerClaims('email')
     * }
     * }))
     */
    static custom(format) {
        return new AccessLogFormat(format);
    }
    /**
     * Generate Common Log Format.
     *
     * @stability stable
     */
    static clf() {
        const requester = [AccessLogField.contextIdentitySourceIp(), AccessLogField.contextIdentityCaller(), AccessLogField.contextIdentityUser()].join(' ');
        const requestTime = AccessLogField.contextRequestTime();
        const request = [AccessLogField.contextHttpMethod(), AccessLogField.contextResourcePath(), AccessLogField.contextProtocol()].join(' ');
        const status = [AccessLogField.contextStatus(), AccessLogField.contextResponseLength(), AccessLogField.contextRequestId()].join(' ');
        return new AccessLogFormat(`${requester} [${requestTime}] "${request}" ${status}`);
    }
    /**
     * Access log will be produced in the JSON format with a set of fields most useful in the access log.
     *
     * All fields are turned on by default with the
     * option to turn off specific fields.
     *
     * @stability stable
     */
    static jsonWithStandardFields(fields = {
        ip: true,
        user: true,
        caller: true,
        requestTime: true,
        httpMethod: true,
        resourcePath: true,
        status: true,
        protocol: true,
        responseLength: true,
    }) {
        return this.custom(JSON.stringify({
            requestId: AccessLogField.contextRequestId(),
            ip: fields.ip ? AccessLogField.contextIdentitySourceIp() : undefined,
            user: fields.user ? AccessLogField.contextIdentityUser() : undefined,
            caller: fields.caller ? AccessLogField.contextIdentityCaller() : undefined,
            requestTime: fields.requestTime ? AccessLogField.contextRequestTime() : undefined,
            httpMethod: fields.httpMethod ? AccessLogField.contextHttpMethod() : undefined,
            resourcePath: fields.resourcePath ? AccessLogField.contextResourcePath() : undefined,
            status: fields.status ? AccessLogField.contextStatus() : undefined,
            protocol: fields.protocol ? AccessLogField.contextProtocol() : undefined,
            responseLength: fields.responseLength ? AccessLogField.contextResponseLength() : undefined,
        }));
    }
    /**
     * Output a format string to be used with CloudFormation.
     *
     * @stability stable
     */
    toString() {
        return this.format;
    }
}
exports.AccessLogFormat = AccessLogFormat;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWNjZXNzLWxvZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFjY2Vzcy1sb2cudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7Ozs7QUEwQkEsTUFBYSxzQkFBc0I7Ozs7SUFDakMsWUFBNkIsUUFBbUI7UUFBbkIsYUFBUSxHQUFSLFFBQVEsQ0FBVztJQUNoRCxDQUFDOzs7Ozs7SUFLTSxJQUFJLENBQUMsTUFBYztRQUN4QixPQUFPO1lBQ0wsY0FBYyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVztTQUMxQyxDQUFDO0lBQ0osQ0FBQztDQUNGO0FBWkQsd0RBWUM7Ozs7OztBQUtELE1BQWEsY0FBYzs7Ozs7O0lBSWxCLE1BQU0sQ0FBQyxnQkFBZ0I7UUFDNUIsT0FBTyw2QkFBNkIsQ0FBQztJQUN2QyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsWUFBWTtRQUN4QixPQUFPLGdCQUFnQixDQUFDO0lBQzFCLENBQUM7Ozs7Ozs7O0lBUU0sTUFBTSxDQUFDLHVCQUF1QixDQUFDLFFBQWdCO1FBQ3BELE9BQU8sOEJBQThCLFFBQVEsRUFBRSxDQUFDO0lBQ2xELENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsNEJBQTRCO1FBQ3hDLE9BQU8saUNBQWlDLENBQUM7SUFDM0MsQ0FBQzs7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsaUJBQWlCLENBQUMsUUFBZ0I7UUFDOUMsT0FBTyx1QkFBdUIsUUFBUSxFQUFFLENBQUM7SUFDM0MsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLDJCQUEyQjtRQUN2QyxPQUFPLCtCQUErQixDQUFDO0lBQ3pDLENBQUM7Ozs7Ozs7O0lBS00sTUFBTSxDQUFDLGlCQUFpQjtRQUM3QixPQUFPLHFCQUFxQixDQUFDO0lBQy9CLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxtQkFBbUI7UUFDL0IsT0FBTyx1QkFBdUIsQ0FBQztJQUNqQyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsbUJBQW1CO1FBQy9CLE9BQU8sd0JBQXdCLENBQUM7SUFDbEMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLHlCQUF5QjtRQUNyQyxPQUFPLDhCQUE4QixDQUFDO0lBQ3hDLENBQUM7Ozs7Ozs7Ozs7SUFTTSxNQUFNLENBQUMsd0JBQXdCO1FBQ3BDLE9BQU8sNkJBQTZCLENBQUM7SUFDdkMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLGlDQUFpQztRQUM3QyxPQUFPLHNDQUFzQyxDQUFDO0lBQ2hELENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyx3QkFBd0I7UUFDcEMsT0FBTyw0QkFBNEIsQ0FBQztJQUN0QyxDQUFDOzs7Ozs7OztJQUtNLE1BQU0sQ0FBQyxpQkFBaUI7UUFDN0IsT0FBTyxxQkFBcUIsQ0FBQztJQUMvQixDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsd0JBQXdCO1FBQ3BDLE9BQU8sNkJBQTZCLENBQUM7SUFDdkMsQ0FBQzs7Ozs7Ozs7O0lBT00sTUFBTSxDQUFDLHFCQUFxQjtRQUNqQyxPQUFPLDBCQUEwQixDQUFDO0lBQ3BDLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyx1QkFBdUI7UUFDbkMsT0FBTyw0QkFBNEIsQ0FBQztJQUN0QyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMscUJBQXFCO1FBQ2pDLE9BQU8sMEJBQTBCLENBQUM7SUFDcEMsQ0FBQzs7Ozs7Ozs7O0lBT00sTUFBTSxDQUFDLDRDQUE0QztRQUN4RCxPQUFPLGlEQUFpRCxDQUFDO0lBQzNELENBQUM7Ozs7Ozs7O0lBTU0sTUFBTSxDQUFDLHdDQUF3QztRQUNwRCxPQUFPLDZDQUE2QyxDQUFDO0lBQ3ZELENBQUM7Ozs7Ozs7O0lBS00sTUFBTSxDQUFDLGdDQUFnQztRQUM1QyxPQUFPLHFDQUFxQyxDQUFDO0lBQy9DLENBQUM7Ozs7Ozs7O0lBTU0sTUFBTSxDQUFDLG9DQUFvQztRQUNoRCxPQUFPLHlDQUF5QyxDQUFDO0lBQ25ELENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyw2QkFBNkI7UUFDekMsT0FBTyxrQ0FBa0MsQ0FBQztJQUM1QyxDQUFDOzs7Ozs7OztJQU1NLE1BQU0sQ0FBQyx1QkFBdUI7UUFDbkMsT0FBTyw0QkFBNEIsQ0FBQztJQUN0QyxDQUFDOzs7Ozs7Ozs7SUFNTSxNQUFNLENBQUMsbUJBQW1CO1FBQy9CLE9BQU8sd0JBQXdCLENBQUM7SUFDbEMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLHdCQUF3QjtRQUNwQyxPQUFPLDZCQUE2QixDQUFDO0lBQ3ZDLENBQUM7Ozs7Ozs7SUFNTSxNQUFNLENBQUMsc0JBQXNCO1FBQ2xDLE9BQU8sMkJBQTJCLENBQUM7SUFDckMsQ0FBQzs7Ozs7Ozs7O0lBT00sTUFBTSxDQUFDLFdBQVc7UUFDdkIsT0FBTyxlQUFlLENBQUM7SUFDekIsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLGVBQWU7UUFDM0IsT0FBTyxtQkFBbUIsQ0FBQztJQUM3QixDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsZ0JBQWdCO1FBQzVCLE9BQU8sb0JBQW9CLENBQUM7SUFDOUIsQ0FBQzs7Ozs7Ozs7O0lBU00sTUFBTSxDQUFDLDRCQUE0QixDQUFDLFVBQWtCO1FBQzNELE9BQU8sbUNBQW1DLFVBQVUsRUFBRSxDQUFDO0lBQ3pELENBQUM7Ozs7Ozs7Ozs7SUFTTSxNQUFNLENBQUMsMEJBQTBCLENBQUMsUUFBZ0I7UUFDdkQsT0FBTyxpQ0FBaUMsUUFBUSxFQUFFLENBQUM7SUFDckQsQ0FBQzs7Ozs7Ozs7O0lBU00sTUFBTSxDQUFDLGlDQUFpQyxDQUFDLGVBQXVCO1FBQ3JFLE9BQU8sd0NBQXdDLGVBQWUsRUFBRSxDQUFDO0lBQ25FLENBQUM7Ozs7Ozs7Ozs7SUFVTSxNQUFNLENBQUMsNkJBQTZCLENBQUMsVUFBa0I7UUFDNUQsT0FBTyxvQ0FBb0MsVUFBVSxFQUFFLENBQUM7SUFDMUQsQ0FBQzs7Ozs7Ozs7OztJQVFNLE1BQU0sQ0FBQyw2QkFBNkI7UUFDekMsT0FBTyxrQ0FBa0MsQ0FBQztJQUM1QyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMsa0JBQWtCO1FBQzlCLE9BQU8sc0JBQXNCLENBQUM7SUFDaEMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLHVCQUF1QjtRQUNuQyxPQUFPLDJCQUEyQixDQUFDO0lBQ3JDLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxpQkFBaUI7UUFDN0IsT0FBTyxxQkFBcUIsQ0FBQztJQUMvQixDQUFDOzs7Ozs7Ozs7O0lBUU0sTUFBTSxDQUFDLG1CQUFtQjtRQUMvQixPQUFPLHVCQUF1QixDQUFDO0lBQ2pDLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxZQUFZO1FBQ3hCLE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQzs7Ozs7Ozs7O0lBTU0sTUFBTSxDQUFDLHNCQUFzQjtRQUNsQyxPQUFPLDBCQUEwQixDQUFDO0lBQ3BDLENBQUM7Ozs7Ozs7OztJQU9NLE1BQU0sQ0FBQyxnQkFBZ0I7UUFDNUIsT0FBTyxvQkFBb0IsQ0FBQztJQUM5QixDQUFDOzs7Ozs7O0lBTU0sTUFBTSxDQUFDLGtCQUFrQjtRQUM5QixPQUFPLHNCQUFzQixDQUFDO0lBQ2hDLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxtQ0FBbUM7UUFDL0MsT0FBTyx3Q0FBd0MsQ0FBQztJQUNsRCxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMseUJBQXlCO1FBQ3JDLE9BQU8sNkJBQTZCLENBQUM7SUFDdkMsQ0FBQzs7Ozs7O0lBTU0sTUFBTSxDQUFDLHdCQUF3QjtRQUNwQyxPQUFPLDZCQUE2QixDQUFDO0lBQ3ZDLENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxzQkFBc0I7UUFDbEMsT0FBTywwQkFBMEIsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7SUFLTSxNQUFNLENBQUMscUJBQXFCO1FBQ2pDLE9BQU8seUJBQXlCLENBQUM7SUFDbkMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLGFBQWE7UUFDekIsT0FBTyxpQkFBaUIsQ0FBQztJQUMzQixDQUFDO0NBQ0Y7QUF6WUQsd0NBeVlDOzs7Ozs7QUErQ0QsTUFBYSxlQUFlO0lBa0UxQixZQUFvQixNQUFjO1FBQ2hDLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO0lBQ3ZCLENBQUM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7SUFyRE0sTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFjO1FBQ2pDLE9BQU8sSUFBSSxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDckMsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLEdBQUc7UUFDZixNQUFNLFNBQVMsR0FBRyxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRSxFQUFFLGNBQWMsQ0FBQyxxQkFBcUIsRUFBRSxFQUFFLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3JKLE1BQU0sV0FBVyxHQUFHLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1FBQ3hELE1BQU0sT0FBTyxHQUFHLENBQUMsY0FBYyxDQUFDLGlCQUFpQixFQUFFLEVBQUUsY0FBYyxDQUFDLG1CQUFtQixFQUFFLEVBQUUsY0FBYyxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZJLE1BQU0sTUFBTSxHQUFHLENBQUMsY0FBYyxDQUFDLGFBQWEsRUFBRSxFQUFFLGNBQWMsQ0FBQyxxQkFBcUIsRUFBRSxFQUFFLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRXJJLE9BQU8sSUFBSSxlQUFlLENBQUMsR0FBRyxTQUFTLEtBQUssV0FBVyxNQUFNLE9BQU8sS0FBSyxNQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQ3JGLENBQUM7Ozs7Ozs7OztJQU1NLE1BQU0sQ0FBQyxzQkFBc0IsQ0FDbEMsU0FBcUM7UUFDbkMsRUFBRSxFQUFFLElBQUk7UUFDUixJQUFJLEVBQUUsSUFBSTtRQUNWLE1BQU0sRUFBRSxJQUFJO1FBQ1osV0FBVyxFQUFFLElBQUk7UUFDakIsVUFBVSxFQUFFLElBQUk7UUFDaEIsWUFBWSxFQUFFLElBQUk7UUFDbEIsTUFBTSxFQUFFLElBQUk7UUFDWixRQUFRLEVBQUUsSUFBSTtRQUNkLGNBQWMsRUFBRSxJQUFJO0tBQ3JCO1FBQ0QsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDaEMsU0FBUyxFQUFFLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRTtZQUM1QyxFQUFFLEVBQUUsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDcEUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3BFLE1BQU0sRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMscUJBQXFCLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMxRSxXQUFXLEVBQUUsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDakYsVUFBVSxFQUFFLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQzlFLFlBQVksRUFBRSxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNwRixNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xFLFFBQVEsRUFBRSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsZUFBZSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDeEUsY0FBYyxFQUFFLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxxQkFBcUIsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1NBQzNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQzs7Ozs7O0lBY00sUUFBUTtRQUNiLE9BQU8sSUFBSSxDQUFDLE1BQU0sQ0FBQztJQUNyQixDQUFDO0NBQ0Y7QUE1RUQsMENBNEVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSUxvZ0dyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0IHsgSVN0YWdlIH0gZnJvbSAnLi9zdGFnZSc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJQWNjZXNzTG9nRGVzdGluYXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBiaW5kKHN0YWdlOiBJU3RhZ2UpOiBBY2Nlc3NMb2dEZXN0aW5hdGlvbkNvbmZpZ1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQWNjZXNzTG9nRGVzdGluYXRpb25Db25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGRlc3RpbmF0aW9uQXJuOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBMb2dHcm91cExvZ0Rlc3RpbmF0aW9uIGltcGxlbWVudHMgSUFjY2Vzc0xvZ0Rlc3RpbmF0aW9uIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBsb2dHcm91cDogSUxvZ0dyb3VwKSB7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBiaW5kKF9zdGFnZTogSVN0YWdlKTogQWNjZXNzTG9nRGVzdGluYXRpb25Db25maWcge1xuICAgIHJldHVybiB7XG4gICAgICBkZXN0aW5hdGlvbkFybjogdGhpcy5sb2dHcm91cC5sb2dHcm91cEFybixcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBY2Nlc3NMb2dGaWVsZCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0QWNjb3VudElkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkuYWNjb3VudElkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dEFwaUlkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuYXBpSWQnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dEF1dGhvcml6ZXJDbGFpbXMocHJvcGVydHk6IHN0cmluZykge1xuICAgIHJldHVybiBgJGNvbnRleHQuYXV0aG9yaXplci5jbGFpbXMuJHtwcm9wZXJ0eX1gO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0QXV0aG9yaXplclByaW5jaXBhbElkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuYXV0aG9yaXplci5wcmluY2lwYWxJZCc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0QXV0aG9yaXplcihwcm9wZXJ0eTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIGAkY29udGV4dC5hdXRob3JpemVyLiR7cHJvcGVydHl9YDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRBd3NFbmRwb2ludFJlcXVlc3RJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmF3c0VuZHBvaW50UmVxdWVzdElkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHREb21haW5OYW1lKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuZG9tYWluTmFtZSc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dERvbWFpblByZWZpeCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmRvbWFpblByZWZpeCc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dEVycm9yTWVzc2FnZSgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmVycm9yLm1lc3NhZ2UnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRFcnJvck1lc3NhZ2VTdHJpbmcoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5lcnJvci5tZXNzYWdlU3RyaW5nJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0RXJyb3JSZXNwb25zZVR5cGUoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5lcnJvci5yZXNwb25zZVR5cGUnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0RXJyb3JWYWxpZGF0aW9uRXJyb3JTdHJpbmcoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5lcnJvci52YWxpZGF0aW9uRXJyb3JTdHJpbmcnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRFeHRlbmRlZFJlcXVlc3RJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmV4dGVuZGVkUmVxdWVzdElkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dEh0dHBNZXRob2QoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5odHRwTWV0aG9kJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dElkZW50aXR5QWNjb3VudElkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkuYWNjb3VudElkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SWRlbnRpdHlBcGlLZXkoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5pZGVudGl0eS5hcGlLZXknO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SWRlbnRpdHlBcGlLZXlJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmlkZW50aXR5LmFwaUtleUlkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SWRlbnRpdHlDYWxsZXIoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5pZGVudGl0eS5jYWxsZXInO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJZGVudGl0eUNvZ25pdG9BdXRoZW50aWNhdGlvblByb3ZpZGVyKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkuY29nbml0b0F1dGhlbnRpY2F0aW9uUHJvdmlkZXInO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJZGVudGl0eUNvZ25pdG9BdXRoZW50aWNhdGlvblR5cGUoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5pZGVudGl0eS5jb2duaXRvQXV0aGVudGljYXRpb25UeXBlJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJZGVudGl0eUNvZ25pdG9JZGVudGl0eUlkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkuY29nbml0b0lkZW50aXR5SWQnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJZGVudGl0eUNvZ25pdG9JZGVudGl0eVBvb2xJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmlkZW50aXR5LmNvZ25pdG9JZGVudGl0eVBvb2xJZCc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SWRlbnRpdHlQcmluY2lwYWxPcmdJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmlkZW50aXR5LnByaW5jaXBhbE9yZ0lkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJZGVudGl0eVNvdXJjZUlwKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkuc291cmNlSXAnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SWRlbnRpdHlVc2VyKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkudXNlcic7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dElkZW50aXR5VXNlckFnZW50KCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuaWRlbnRpdHkudXNlckFnZW50JztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dElkZW50aXR5VXNlckFybigpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmlkZW50aXR5LnVzZXJBcm4nO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UGF0aCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LnBhdGgnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UHJvdG9jb2woKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5wcm90b2NvbCc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UmVxdWVzdElkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQucmVxdWVzdElkJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UmVxdWVzdE92ZXJyaWRlSGVhZGVyKGhlYWRlck5hbWU6IHN0cmluZykge1xuICAgIHJldHVybiBgJGNvbnRleHQucmVxdWVzdE92ZXJyaWRlLmhlYWRlci4ke2hlYWRlck5hbWV9YDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXF1ZXN0T3ZlcnJpZGVQYXRoKHBhdGhOYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gYCRjb250ZXh0LnJlcXVlc3RPdmVycmlkZS5wYXRoLiR7cGF0aE5hbWV9YDtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXF1ZXN0T3ZlcnJpZGVRdWVyeXN0cmluZyhxdWVyeXN0cmluZ05hbWU6IHN0cmluZykge1xuICAgIHJldHVybiBgJGNvbnRleHQucmVxdWVzdE92ZXJyaWRlLnF1ZXJ5c3RyaW5nLiR7cXVlcnlzdHJpbmdOYW1lfWA7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UmVzcG9uc2VPdmVycmlkZUhlYWRlcihoZWFkZXJOYW1lOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gYCRjb250ZXh0LnJlc3BvbnNlT3ZlcnJpZGUuaGVhZGVyLiR7aGVhZGVyTmFtZX1gO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXNwb25zZU92ZXJyaWRlU3RhdHVzKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQucmVzcG9uc2VPdmVycmlkZS5zdGF0dXMnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXF1ZXN0VGltZSgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LnJlcXVlc3RUaW1lJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXF1ZXN0VGltZUVwb2NoKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQucmVxdWVzdFRpbWVFcG9jaCc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0UmVzb3VyY2VJZCgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LnJlc291cmNlSWQnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRSZXNvdXJjZVBhdGgoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5yZXNvdXJjZVBhdGgnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dFN0YWdlKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQuc3RhZ2UnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0V2FmUmVzcG9uc2VDb2RlKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQud2FmUmVzcG9uc2VDb2RlJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0V2ViYWNsQXJuKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQud2ViYWNsQXJuJztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dFhyYXlUcmFjZUlkKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQueHJheVRyYWNlSWQnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0QXV0aG9yaXplckludGVncmF0aW9uTGF0ZW5jeSgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LmF1dGhvcml6ZXIuaW50ZWdyYXRpb25MYXRlbmN5JztcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGNvbnRleHRJbnRlZ3JhdGlvbkxhdGVuY3koKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5pbnRlZ3JhdGlvbkxhdGVuY3knO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBjb250ZXh0SW50ZWdyYXRpb25TdGF0dXMoKSB7XG4gICAgcmV0dXJuICckY29udGV4dC5pbnRlZ3JhdGlvblN0YXR1cy4nO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dFJlc3BvbnNlTGF0ZW5jeSgpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LnJlc3BvbnNlTGF0ZW5jeSc7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dFJlc3BvbnNlTGVuZ3RoKCkge1xuICAgIHJldHVybiAnJGNvbnRleHQucmVzcG9uc2VMZW5ndGgnO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY29udGV4dFN0YXR1cygpIHtcbiAgICByZXR1cm4gJyRjb250ZXh0LnN0YXR1cyc7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSnNvbldpdGhTdGFuZGFyZEZpZWxkUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaXA6IGJvb2xlYW4sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNhbGxlcjogYm9vbGVhbixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHVzZXI6IGJvb2xlYW4sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSByZXF1ZXN0VGltZTogYm9vbGVhbixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGh0dHBNZXRob2Q6IGJvb2xlYW4sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVzb3VyY2VQYXRoOiBib29sZWFuLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdGF0dXM6IGJvb2xlYW4sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHByb3RvY29sOiBib29sZWFuLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVzcG9uc2VMZW5ndGg6IGJvb2xlYW5cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFjY2Vzc0xvZ0Zvcm1hdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY3VzdG9tKGZvcm1hdDogc3RyaW5nKTogQWNjZXNzTG9nRm9ybWF0IHtcbiAgICByZXR1cm4gbmV3IEFjY2Vzc0xvZ0Zvcm1hdChmb3JtYXQpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgY2xmKCk6IEFjY2Vzc0xvZ0Zvcm1hdCB7XG4gICAgY29uc3QgcmVxdWVzdGVyID0gW0FjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRJZGVudGl0eVNvdXJjZUlwKCksIEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRJZGVudGl0eUNhbGxlcigpLCBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0SWRlbnRpdHlVc2VyKCldLmpvaW4oJyAnKTtcbiAgICBjb25zdCByZXF1ZXN0VGltZSA9IEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRSZXF1ZXN0VGltZSgpO1xuICAgIGNvbnN0IHJlcXVlc3QgPSBbQWNjZXNzTG9nRmllbGQuY29udGV4dEh0dHBNZXRob2QoKSwgQWNjZXNzTG9nRmllbGQuY29udGV4dFJlc291cmNlUGF0aCgpLCBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0UHJvdG9jb2woKV0uam9pbignICcpO1xuICAgIGNvbnN0IHN0YXR1cyA9IFtBY2Nlc3NMb2dGaWVsZC5jb250ZXh0U3RhdHVzKCksIEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRSZXNwb25zZUxlbmd0aCgpLCBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0UmVxdWVzdElkKCldLmpvaW4oJyAnKTtcblxuICAgIHJldHVybiBuZXcgQWNjZXNzTG9nRm9ybWF0KGAke3JlcXVlc3Rlcn0gWyR7cmVxdWVzdFRpbWV9XSBcIiR7cmVxdWVzdH1cIiAke3N0YXR1c31gKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBqc29uV2l0aFN0YW5kYXJkRmllbGRzKFxuICAgIGZpZWxkczogSnNvbldpdGhTdGFuZGFyZEZpZWxkUHJvcHMgPSB7XG4gICAgICBpcDogdHJ1ZSxcbiAgICAgIHVzZXI6IHRydWUsXG4gICAgICBjYWxsZXI6IHRydWUsXG4gICAgICByZXF1ZXN0VGltZTogdHJ1ZSxcbiAgICAgIGh0dHBNZXRob2Q6IHRydWUsXG4gICAgICByZXNvdXJjZVBhdGg6IHRydWUsXG4gICAgICBzdGF0dXM6IHRydWUsXG4gICAgICBwcm90b2NvbDogdHJ1ZSxcbiAgICAgIHJlc3BvbnNlTGVuZ3RoOiB0cnVlLFxuICAgIH0pOiBBY2Nlc3NMb2dGb3JtYXQge1xuICAgIHJldHVybiB0aGlzLmN1c3RvbShKU09OLnN0cmluZ2lmeSh7XG4gICAgICByZXF1ZXN0SWQ6IEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRSZXF1ZXN0SWQoKSxcbiAgICAgIGlwOiBmaWVsZHMuaXAgPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0SWRlbnRpdHlTb3VyY2VJcCgpIDogdW5kZWZpbmVkLFxuICAgICAgdXNlcjogZmllbGRzLnVzZXIgPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0SWRlbnRpdHlVc2VyKCkgOiB1bmRlZmluZWQsXG4gICAgICBjYWxsZXI6IGZpZWxkcy5jYWxsZXIgPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0SWRlbnRpdHlDYWxsZXIoKSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlcXVlc3RUaW1lOiBmaWVsZHMucmVxdWVzdFRpbWUgPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0UmVxdWVzdFRpbWUoKSA6IHVuZGVmaW5lZCxcbiAgICAgIGh0dHBNZXRob2Q6IGZpZWxkcy5odHRwTWV0aG9kID8gQWNjZXNzTG9nRmllbGQuY29udGV4dEh0dHBNZXRob2QoKSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlc291cmNlUGF0aDogZmllbGRzLnJlc291cmNlUGF0aCA/IEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRSZXNvdXJjZVBhdGgoKSA6IHVuZGVmaW5lZCxcbiAgICAgIHN0YXR1czogZmllbGRzLnN0YXR1cyA/IEFjY2Vzc0xvZ0ZpZWxkLmNvbnRleHRTdGF0dXMoKSA6IHVuZGVmaW5lZCxcbiAgICAgIHByb3RvY29sOiBmaWVsZHMucHJvdG9jb2wgPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0UHJvdG9jb2woKSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlc3BvbnNlTGVuZ3RoOiBmaWVsZHMucmVzcG9uc2VMZW5ndGggPyBBY2Nlc3NMb2dGaWVsZC5jb250ZXh0UmVzcG9uc2VMZW5ndGgoKSA6IHVuZGVmaW5lZCxcbiAgICB9KSk7XG4gIH1cblxuICAvKipcbiAgICogQSBBUEkgR2F0ZXdheSBjdXN0b20gYWNjZXNzIGxvZyBmb3JtYXRcbiAgICovXG4gIHByaXZhdGUgcmVhZG9ubHkgZm9ybWF0OiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihmb3JtYXQ6IHN0cmluZykge1xuICAgIHRoaXMuZm9ybWF0ID0gZm9ybWF0O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyB0b1N0cmluZygpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmZvcm1hdDtcbiAgfVxufVxuIl19