"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    /**
     *
     */
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : options.allowTestInvoke;
    }
    /**
     * Can be overridden by subclasses to allow the integration to interact with the method being integrated, access the REST API object, method ARNs, etc.
     */
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${core_1.Names.nodeUniqueId(method.api.node)}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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