"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
/**
 *
 */
class UsagePlan extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.anyValue({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Adds an ApiKey.
     */
    addApiKey(apiKey) {
        const prefix = 'UsagePlanKeyResource';
        // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodifed.
        const id = this.node.tryFindChild(prefix) ? `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}` : prefix;
        new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
    }
    /**
     * Adds an apiStage.
     */
    addApiStage(apiStage) {
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateInteger(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
//# sourceMappingURL=data:application/json;base64,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