"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcLink = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
/**
 * Define a new VPC Link Specifies an API Gateway VPC link for a RestApi to access resources in an Amazon Virtual Private Cloud (VPC).
 */
class VpcLink extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.vpcLinkName ||
                core_1.Lazy.stringValue({ produce: () => core_1.Names.nodeUniqueId(this.node) }),
        });
        this._targets = new Array();
        const cfnResource = new apigateway_generated_1.CfnVpcLink(this, 'Resource', {
            name: this.physicalName,
            description: props.description,
            targetArns: core_1.Lazy.listValue({ produce: () => this.renderTargets() }),
        });
        this.vpcLinkId = cfnResource.ref;
        if (props.targets) {
            this.addTargets(...props.targets);
        }
    }
    /**
     * Import a VPC Link by its Id.
     */
    static fromVpcLinkId(scope, id, vpcLinkId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcLinkId = vpcLinkId;
            }
        }
        return new Import(scope, id);
    }
    /**
     *
     */
    addTargets(...targets) {
        this._targets.push(...targets);
    }
    /**
     * Return the list of DNS names from the target NLBs.
     * @internal
     * */
    get _targetDnsNames() {
        return this._targets.map(t => t.loadBalancerDnsName);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     */
    validate() {
        if (this._targets.length === 0) {
            return ['No targets added to vpc link'];
        }
        return [];
    }
    renderTargets() {
        return this._targets.map(nlb => nlb.loadBalancerArn);
    }
}
exports.VpcLink = VpcLink;
//# sourceMappingURL=data:application/json;base64,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