"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Channel = exports.ChannelType = exports.LatencyMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("aws-cdk-lib");
const aws_ivs_1 = require("aws-cdk-lib/aws-ivs");
const stream_key_1 = require("./stream-key");
/**
 * Reference to a new or existing IVS Channel
 */
class ChannelBase extends core.Resource {
    addStreamKey(id) {
        return new stream_key_1.StreamKey(this, id, {
            channel: this,
        });
    }
}
/**
  Channel latency mode
*/
var LatencyMode;
(function (LatencyMode) {
    /**
     * Use LOW to minimize broadcaster-to-viewer latency for interactive broadcasts.
     */
    LatencyMode["LOW"] = "LOW";
    /**
     * Use NORMAL for broadcasts that do not require viewer interaction.
     */
    LatencyMode["NORMAL"] = "NORMAL";
})(LatencyMode = exports.LatencyMode || (exports.LatencyMode = {}));
/**
  * The channel type, which determines the allowable resolution and bitrate.
  * If you exceed the allowable resolution or bitrate, the stream probably will disconnect immediately.
*/
var ChannelType;
(function (ChannelType) {
    /**
     * Multiple qualities are generated from the original input, to automatically give viewers the best experience for
     * their devices and network conditions.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ivs-channel.html
     */
    ChannelType["STANDARD"] = "STANDARD";
    /**
     * delivers the original input to viewers. The viewer’s video-quality choice is limited to the original input.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ivs-channel.html
     */
    ChannelType["BASIC"] = "BASIC";
})(ChannelType = exports.ChannelType || (exports.ChannelType = {}));
/**
  A new IVS channel
*/
class Channel extends ChannelBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ivs_alpha_ChannelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        if (props.name && !core.Token.isUnresolved(props.name) && !/^[a-zA-Z0-9-_]*$/.test(props.name)) {
            throw new Error(`name must contain only numbers, letters, hyphens and underscores, got: '${props.name}'`);
        }
        const resource = new aws_ivs_1.CfnChannel(this, 'Resource', {
            authorized: props.authorized,
            latencyMode: props.latencyMode,
            name: props.name,
            type: props.type,
        });
        this.channelArn = resource.attrArn;
        this.channelIngestEndpoint = resource.attrIngestEndpoint;
        this.channelPlaybackUrl = resource.attrPlaybackUrl;
    }
    /**
     * Import an existing channel
     */
    static fromChannelArn(scope, id, channelArn) {
        // This will throw an error if the arn cannot be parsed
        let arnComponents = core.Arn.split(channelArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!core.Token.isUnresolved(arnComponents.service) && arnComponents.service !== 'ivs') {
            throw new Error(`Invalid service, expected 'ivs', got '${arnComponents.service}'`);
        }
        if (!core.Token.isUnresolved(arnComponents.resource) && arnComponents.resource !== 'channel') {
            throw new Error(`Invalid resource, expected 'channel', got '${arnComponents.resource}'`);
        }
        class Import extends ChannelBase {
            constructor() {
                super(...arguments);
                this.channelArn = channelArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Channel = Channel;
_a = JSII_RTTI_SYMBOL_1;
Channel[_a] = { fqn: "@aws-cdk/aws-ivs-alpha.Channel", version: "2.26.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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