# **EndPoint Function Documentation**  

## **Overview**  
The `EndPoint` function is used to make HTTP requests in a local or custom environment. It sets up the required environment variables and interacts with a WSGI application to process the request.

---

## **Install**
```bash
pip install polyframe-endpoint
```

## **Function Signature**  
```python
def EndPoint(
    script_name: str = None,
    cwd: str = None,
    method: str = "GET",
    q: str = None,
    headers: dict = None,
    body: str = None,
    output: str = None
)
```

---

## **Parameters**  

| Parameter   | Type   | Default | Description |
|------------|--------|---------|-------------|
| `script_name` | `str` | `None` | The API endpoint or script filename to execute (e.g., `"api/v1/play_song"`). |
| `cwd` | `str` | `None` | The current working directory. If not provided, it defaults to the current directory. |
| `method` | `str` | `"GET"` | The HTTP method to use (e.g., `"GET"`, `"POST"`, `"PUT"`, `"DELETE"`). |
| `q` | `str` | `None` | The query string parameters (e.g., `"track_id=22"`). |
| `headers` | `dict` | `None` | A dictionary containing HTTP headers (e.g., `{"Content-Type": "application/json"}`). |
| `body` | `str` | `None` | The request body as a JSON string for POST/PUT requests (e.g., `'{"track_id": 22, "action": "play"}'`). |
| `output` | `str` | `None` | Specifies how the response should be displayed (`"s"`, `"h"`, `"d"`, or `None`). |

---

## **Output Methods**  

| **Output Value** | **Description** | **Example Output** |
|------------------|---------------|------------------|
| `"s"` | **Status only** | `200 OK` |
| `"h"` | **Headers only** | `Content-Type: application/json`<br>`Content-Length: 58` |
| `"d"` | **Body only** | `{ "success": true, "message": "Track is now playing" }` |
| `None` (default) | **Full response (status, headers, and body)** | ```json { "status": "200 OK", "body": { "success": true, "message": "Track is now playing" }, "headers": { "Content-Type": "application/json" } } ``` |

---

## **Examples**  

### **1. Sending a GET request with query parameters**
```python
from polyframe_endpoint.endpoint import EndPoint
get_song = EndPoint(
    script_name="api/v1/get_song.py",
    method="GET",
    q="track_id=22",
    headers={
        "Accept": "application/json",
        "Authorization": "Bearer your_token_here"
    },
    output="d"
)
print(get_song)
```
📌 **This will print only the response body.**  

---

### **2. Sending a POST request with JSON body**  
```python
from polyframe_endpoint.endpoint import EndPoint
import json

play_song = EndPoint(
    script_name="api/v1/play_song.py",
    method="POST",
    headers={
        "Content-Type": "application/json",
        "Authorization": "Bearer your_token_here"
    },
    body=json.dumps({
        "track_id": 22,
        "action": "play"
    }),
    output="?"
)
print(play_song['status')
```
📌 **This will print the full response (status, headers, and body).**  

---

### **3. Displaying only the response headers**  
```python
from polyframe_endpoint.endpoint import EndPoint
get_h = EndPoint(
    script_name="api/v1/play_song.py",
    method="POST",
    output="h"
)

print(get_h )
```
📌 **This will print only the response headers.**  

---

## **Error Handling**
- If a required parameter is missing, the function will print an error message.
- If the WSGI application fails to execute, an error message is displayed and the program exits.

---

## **Notes**
- The function supports both `GET` and `POST` requests.
- It simulates a WSGI environment to execute scripts as API endpoints.
- It handles HTTP headers dynamically and includes official HTTP header support.
- The function reads from `sys.argv` for command-line execution support.

---
