# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange

# -----------------------------------------------------------------------------

try:
    basestring  # Python 3
except NameError:
    basestring = str  # Python 2
import math
import json
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.errors import OnMaintenance
from ccxt.base.errors import InvalidNonce


class bitstamp(Exchange):

    def describe(self):
        return self.deep_extend(super(bitstamp, self).describe(), {
            'id': 'bitstamp',
            'name': 'Bitstamp',
            'countries': ['GB'],
            'rateLimit': 1000,
            'version': 'v2',
            'userAgent': self.userAgents['chrome'],
            'pro': True,
            'has': {
                'CORS': True,
                'cancelOrder': True,
                'createOrder': True,
                'fetchBalance': True,
                'fetchDepositAddress': True,
                'fetchMarkets': True,
                'fetchCurrencies': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchTicker': True,
                'fetchTrades': True,
                'fetchTransactions': True,
                'fetchWithdrawals': True,
                'withdraw': True,
                'fetchTradingFee': True,
                'fetchTradingFees': True,
                'fetchFundingFees': True,
                'fetchFees': True,
                'fetchLedger': True,
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/27786377-8c8ab57e-5fe9-11e7-8ea4-2b05b6bcceec.jpg',
                'api': {
                    'public': 'https://www.bitstamp.net/api',
                    'private': 'https://www.bitstamp.net/api',
                    'v1': 'https://www.bitstamp.net/api',
                },
                'www': 'https://www.bitstamp.net',
                'doc': 'https://www.bitstamp.net/api',
            },
            'timeframes': {
                '1m': '60',
                '3m': '180',
                '5m': '300',
                '15m': '900',
                '30m': '1800',
                '1h': '3600',
                '2h': '7200',
                '4h': '14400',
                '6h': '21600',
                '12h': '43200',
                '1d': '86400',
                '1w': '259200',
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
                'uid': True,
            },
            'api': {
                'public': {
                    'get': [
                        'ohlc/{pair}/',
                        'order_book/{pair}/',
                        'ticker_hour/{pair}/',
                        'ticker/{pair}/',
                        'transactions/{pair}/',
                        'trading-pairs-info/',
                    ],
                },
                'private': {
                    'post': [
                        'balance/',
                        'balance/{pair}/',
                        'bch_withdrawal/',
                        'bch_address/',
                        'user_transactions/',
                        'user_transactions/{pair}/',
                        'open_orders/all/',
                        'open_orders/{pair}/',
                        'order_status/',
                        'cancel_order/',
                        'buy/{pair}/',
                        'buy/market/{pair}/',
                        'buy/instant/{pair}/',
                        'sell/{pair}/',
                        'sell/market/{pair}/',
                        'sell/instant/{pair}/',
                        'ltc_withdrawal/',
                        'ltc_address/',
                        'eth_withdrawal/',
                        'eth_address/',
                        'xrp_withdrawal/',
                        'xrp_address/',
                        'xlm_withdrawal/',
                        'xlm_address/',
                        'pax_withdrawal/',
                        'pax_address/',
                        'link_withdrawal/',
                        'link_address/',
                        'usdc_withdrawal/',
                        'usdc_address/',
                        'omg_withdrawal/',
                        'omg_address/',
                        'transfer-to-main/',
                        'transfer-from-main/',
                        'withdrawal-requests/',
                        'withdrawal/open/',
                        'withdrawal/status/',
                        'withdrawal/cancel/',
                        'liquidation_address/new/',
                        'liquidation_address/info/',
                    ],
                },
                'v1': {
                    'post': [
                        'bitcoin_deposit_address/',
                        'unconfirmed_btc/',
                        'bitcoin_withdrawal/',
                        'ripple_withdrawal/',
                        'ripple_address/',
                    ],
                },
            },
            'fees': {
                'trading': {
                    'tierBased': True,
                    'percentage': True,
                    'taker': 0.5 / 100,
                    'maker': 0.5 / 100,
                    'tiers': {
                        'taker': [
                            [0, 0.5 / 100],
                            [20000, 0.25 / 100],
                            [100000, 0.24 / 100],
                            [200000, 0.22 / 100],
                            [400000, 0.20 / 100],
                            [600000, 0.15 / 100],
                            [1000000, 0.14 / 100],
                            [2000000, 0.13 / 100],
                            [4000000, 0.12 / 100],
                            [20000000, 0.11 / 100],
                            [50000000, 0.10 / 100],
                            [100000000, 0.07 / 100],
                            [500000000, 0.05 / 100],
                            [2000000000, 0.03 / 100],
                            [6000000000, 0.01 / 100],
                            [10000000000, 0.005 / 100],
                            [10000000001, 0.0],
                        ],
                        'maker': [
                            [0, 0.5 / 100],
                            [20000, 0.25 / 100],
                            [100000, 0.24 / 100],
                            [200000, 0.22 / 100],
                            [400000, 0.20 / 100],
                            [600000, 0.15 / 100],
                            [1000000, 0.14 / 100],
                            [2000000, 0.13 / 100],
                            [4000000, 0.12 / 100],
                            [20000000, 0.11 / 100],
                            [50000000, 0.10 / 100],
                            [100000000, 0.07 / 100],
                            [500000000, 0.05 / 100],
                            [2000000000, 0.03 / 100],
                            [6000000000, 0.01 / 100],
                            [10000000000, 0.005 / 100],
                            [10000000001, 0.0],
                        ],
                    },
                },
                'funding': {
                    'tierBased': False,
                    'percentage': False,
                    'withdraw': {
                        'BTC': 0.0005,
                        'BCH': 0.0001,
                        'LTC': 0.001,
                        'ETH': 0.001,
                        'XRP': 0.02,
                        'XLM': 0.005,
                        'PAX': 0.5,
                        'USD': 25,
                        'EUR': 3.0,
                    },
                    'deposit': {
                        'BTC': 0,
                        'BCH': 0,
                        'LTC': 0,
                        'ETH': 0,
                        'XRP': 0,
                        'XLM': 0,
                        'PAX': 0,
                        'USD': 7.5,
                        'EUR': 0,
                    },
                },
            },
            'exceptions': {
                'exact': {
                    'No permission found': PermissionDenied,
                    'API key not found': AuthenticationError,
                    'IP address not allowed': PermissionDenied,
                    'Invalid nonce': InvalidNonce,
                    'Invalid signature': AuthenticationError,
                    'Authentication failed': AuthenticationError,
                    'Missing key, signature and nonce parameters': AuthenticationError,
                    'Your account is frozen': PermissionDenied,
                    'Please update your profile with your FATCA information, before using API.': PermissionDenied,
                    'Order not found': OrderNotFound,
                    'Price is more than 20% below market price.': InvalidOrder,
                    'Bitstamp.net is under scheduled maintenance.': OnMaintenance,  # {"error": "Bitstamp.net is under scheduled maintenance. We'll be back soon."}
                },
                'broad': {
                    'Minimum order size is': InvalidOrder,  # Minimum order size is 5.0 EUR.
                    'Check your account balance for details.': InsufficientFunds,  # You have only 0.00100000 BTC available. Check your account balance for details.
                    'Ensure self value has at least': InvalidAddress,  # Ensure self value has at least 25 characters(it has 4).
                },
            },
        })

    async def fetch_markets(self, params={}):
        response = await self.fetch_markets_from_cache(params)
        result = []
        for i in range(0, len(response)):
            market = response[i]
            name = self.safe_string(market, 'name')
            base, quote = name.split('/')
            baseId = base.lower()
            quoteId = quote.lower()
            base = self.safe_currency_code(base)
            quote = self.safe_currency_code(quote)
            symbol = base + '/' + quote
            symbolId = baseId + '_' + quoteId
            id = self.safe_string(market, 'url_symbol')
            precision = {
                'amount': market['base_decimals'],
                'price': market['counter_decimals'],
            }
            parts = market['minimum_order'].split(' ')
            cost = parts[0]
            # cost, currency = market['minimum_order'].split(' ')
            active = (market['trading'] == 'Enabled')
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'baseId': baseId,
                'quoteId': quoteId,
                'symbolId': symbolId,
                'info': market,
                'active': active,
                'precision': precision,
                'limits': {
                    'amount': {
                        'min': math.pow(10, -precision['amount']),
                        'max': None,
                    },
                    'price': {
                        'min': math.pow(10, -precision['price']),
                        'max': None,
                    },
                    'cost': {
                        'min': float(cost),
                        'max': None,
                    },
                },
            })
        return result

    def construct_currency_object(self, id, code, name, precision, minCost, originalPayload):
        currencyType = 'crypto'
        description = self.describe()
        if self.is_fiat(code):
            currencyType = 'fiat'
        return {
            'id': id,
            'code': code,
            'info': originalPayload,  # the original payload
            'type': currencyType,
            'name': name,
            'active': True,
            'fee': self.safe_float(description['fees']['funding']['withdraw'], code),
            'precision': precision,
            'limits': {
                'amount': {
                    'min': math.pow(10, -precision),
                    'max': None,
                },
                'price': {
                    'min': math.pow(10, -precision),
                    'max': None,
                },
                'cost': {
                    'min': minCost,
                    'max': None,
                },
                'withdraw': {
                    'min': None,
                    'max': None,
                },
            },
        }

    async def fetch_markets_from_cache(self, params={}):
        # self method is now redundant
        # currencies are now fetched before markets
        options = self.safe_value(self.options, 'fetchMarkets', {})
        timestamp = self.safe_integer(options, 'timestamp')
        expires = self.safe_integer(options, 'expires', 1000)
        now = self.milliseconds()
        if (timestamp is None) or ((now - timestamp) > expires):
            response = await self.publicGetTradingPairsInfo(params)
            self.options['fetchMarkets'] = self.extend(options, {
                'response': response,
                'timestamp': now,
            })
        return self.safe_value(self.options['fetchMarkets'], 'response')

    async def fetch_currencies(self, params={}):
        response = await self.fetch_markets_from_cache(params)
        result = {}
        for i in range(0, len(response)):
            market = response[i]
            name = self.safe_string(market, 'name')
            base, quote = name.split('/')
            baseId = base.lower()
            quoteId = quote.lower()
            base = self.safe_currency_code(base)
            quote = self.safe_currency_code(quote)
            description = self.safe_string(market, 'description')
            baseDescription, quoteDescription = description.split(' / ')
            parts = market['minimum_order'].split(' ')
            cost = parts[0]
            if not (base in result):
                result[base] = self.construct_currency_object(baseId, base, baseDescription, market['base_decimals'], None, market)
            if not (quote in result):
                result[quote] = self.construct_currency_object(quoteId, quote, quoteDescription, market['counter_decimals'], float(cost), market)
        return result

    async def fetch_order_book(self, symbol, limit=None, params={}):
        await self.load_markets()
        request = {
            'pair': self.market_id(symbol),
        }
        response = await self.publicGetOrderBookPair(self.extend(request, params))
        #
        #     {
        #         "timestamp": "1583652948",
        #         "microtimestamp": "1583652948955826",
        #         "bids": [
        #             ["8750.00", "1.33685271"],
        #             ["8749.39", "0.07700000"],
        #             ["8746.98", "0.07400000"],
        #         ]
        #         "asks": [
        #             ["8754.10", "1.51995636"],
        #             ["8754.71", "1.40000000"],
        #             ["8754.72", "2.50000000"],
        #         ]
        #     }
        #
        microtimestamp = self.safe_integer(response, 'microtimestamp')
        timestamp = int(microtimestamp / 1000)
        orderbook = self.parse_order_book(response, timestamp)
        orderbook['nonce'] = microtimestamp
        return orderbook

    async def fetch_ticker(self, symbol, params={}):
        await self.load_markets()
        request = {
            'pair': self.market_id(symbol),
        }
        ticker = await self.publicGetTickerPair(self.extend(request, params))
        timestamp = self.safe_timestamp(ticker, 'timestamp')
        vwap = self.safe_float(ticker, 'vwap')
        baseVolume = self.safe_float(ticker, 'volume')
        quoteVolume = None
        if baseVolume is not None and vwap is not None:
            quoteVolume = baseVolume * vwap
        last = self.safe_float(ticker, 'last')
        return {
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_float(ticker, 'high'),
            'low': self.safe_float(ticker, 'low'),
            'bid': self.safe_float(ticker, 'bid'),
            'bidVolume': None,
            'ask': self.safe_float(ticker, 'ask'),
            'askVolume': None,
            'vwap': vwap,
            'open': self.safe_float(ticker, 'open'),
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': None,
            'average': None,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        }

    def get_currency_id_from_transaction(self, transaction):
        #
        #     {
        #         "fee": "0.00000000",
        #         "btc_usd": "0.00",
        #         "datetime": XXX,
        #         "usd": 0.0,
        #         "btc": 0.0,
        #         "eth": "0.05000000",
        #         "type": "0",
        #         "id": XXX,
        #         "eur": 0.0
        #     }
        #
        currencyId = self.safe_string_lower(transaction, 'currency')
        if currencyId is not None:
            return currencyId
        transaction = self.omit(transaction, [
            'fee',
            'price',
            'datetime',
            'type',
            'status',
            'id',
        ])
        ids = list(transaction.keys())
        for i in range(0, len(ids)):
            id = ids[i]
            if id.find('_') < 0:
                value = self.safe_float(transaction, id)
                if (value is not None) and (value != 0):
                    return id
        return None

    def get_market_from_trade(self, trade):
        trade = self.omit(trade, [
            'fee',
            'price',
            'datetime',
            'tid',
            'type',
            'order_id',
            'side',
        ])
        currencyIds = list(trade.keys())
        numCurrencyIds = len(currencyIds)
        if numCurrencyIds > 2:
            raise ExchangeError(self.id + ' getMarketFromTrade too many keys: ' + self.json(currencyIds) + ' in the trade: ' + self.json(trade))
        if numCurrencyIds == 2:
            marketId = currencyIds[0] + currencyIds[1]
            if marketId in self.markets_by_id:
                return self.markets_by_id[marketId]
            marketId = currencyIds[1] + currencyIds[0]
            if marketId in self.markets_by_id:
                return self.markets_by_id[marketId]
        return None

    def get_market_from_trades(self, trades):
        tradesBySymbol = self.index_by(trades, 'symbol')
        symbols = list(tradesBySymbol.keys())
        numSymbols = len(symbols)
        if numSymbols == 1:
            return self.markets[symbols[0]]
        return None

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades(public)
        #
        #     {
        #         date: '1551814435',
        #         tid: '83581898',
        #         price: '0.03532850',
        #         type: '1',
        #         amount: '0.85945907'
        #     },
        #
        # fetchMyTrades, trades returned within fetchOrder(private)
        #
        #     {
        #         "usd": "6.0134400000000000",
        #         "price": "4008.96000000",
        #         "datetime": "2019-03-28 23:07:37.233599",
        #         "fee": "0.02",
        #         "btc": "0.00150000",
        #         "tid": 84452058,
        #         "type": 2
        #     }
        #
        # from fetchOrder:
        #    {fee: '0.000019',
        #     price: '0.00015803',
        #     datetime: '2018-01-07 10:45:34.132551',
        #     btc: '0.0079015000000000',
        #     tid: 42777395,
        #     type: 2,  #(0 - deposit; 1 - withdrawal; 2 - market trade) NOT buy/sell
        #     xrp: '50.00000000'}
        id = self.safe_string_2(trade, 'id', 'tid')
        symbol = None
        side = None
        price = self.safe_float(trade, 'price')
        amount = self.safe_float(trade, 'amount')
        orderId = self.safe_string(trade, 'order_id')
        type = None
        cost = self.safe_float(trade, 'cost')
        if market is None:
            keys = list(trade.keys())
            for i in range(0, len(keys)):
                if keys[i].find('_') >= 0:
                    marketId = keys[i].replace('_', '')
                    if marketId in self.markets_by_id:
                        market = self.markets_by_id[marketId]
            # if the market is still not defined
            # try to deduce it from used keys
            if market is None:
                market = self.get_market_from_trade(trade)
        feeCost = self.safe_float(trade, 'fee')
        feeCurrency = None
        if market is not None:
            price = self.safe_float(trade, market['symbolId'], price)
            amount = self.safe_float(trade, market['baseId'], amount)
            cost = self.safe_float(trade, market['quoteId'], cost)
            feeCurrency = market['quote']
            symbol = market['symbol']
        timestamp = self.safe_string_2(trade, 'date', 'datetime')
        if timestamp is not None:
            if timestamp.find(' ') >= 0:
                # iso8601
                timestamp = self.parse8601(timestamp)
            else:
                # string unix epoch in seconds
                timestamp = int(timestamp)
                timestamp = timestamp * 1000
        # if it is a private trade
        if 'id' in trade:
            if amount is not None:
                if amount < 0:
                    side = 'sell'
                    amount = -amount
                else:
                    side = 'buy'
        else:
            side = self.safe_string(trade, 'type')
            if side == '1':
                side = 'sell'
            elif side == '0':
                side = 'buy'
        if cost is None:
            if price is not None:
                if amount is not None:
                    cost = price * amount
        if cost is not None:
            cost = abs(cost)
        fee = None
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': feeCurrency,
            }
        return {
            'id': id,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'order': orderId,
            'type': type,
            'side': side,
            'takerOrMaker': None,
            'price': price,
            'amount': amount,
            'cost': cost,
            'fee': fee,
        }

    def parse_trading_fee(self, balances, symbol):
        market = self.market(symbol)
        tradeFee = self.safe_float(balances, market['id'] + '_fee')
        return {
            'symbol': symbol,
            'maker': tradeFee,
            'taker': tradeFee,
        }

    async def fetch_trades(self, symbol, since=None, limit=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'pair': market['id'],
            'time': 'hour',
        }
        response = await self.publicGetTransactionsPair(self.extend(request, params))
        #
        #     [
        #         {
        #             date: '1551814435',
        #             tid: '83581898',
        #             price: '0.03532850',
        #             type: '1',
        #             amount: '0.85945907'
        #         },
        #         {
        #             date: '1551814434',
        #             tid: '83581896',
        #             price: '0.03532851',
        #             type: '1',
        #             amount: '11.34130961'
        #         },
        #     ]
        #
        return self.parse_trades(response, market, since, limit)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     {
        #         "high": "9064.77",
        #         "timestamp": "1593961440",
        #         "volume": "18.49436608",
        #         "low": "9040.87",
        #         "close": "9064.77",
        #         "open": "9040.87"
        #     }
        #
        return [
            self.safe_timestamp(ohlcv, 'timestamp'),
            self.safe_float(ohlcv, 'open'),
            self.safe_float(ohlcv, 'high'),
            self.safe_float(ohlcv, 'low'),
            self.safe_float(ohlcv, 'close'),
            self.safe_float(ohlcv, 'volume'),
        ]

    async def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'pair': market['id'],
            'step': self.timeframes[timeframe],
        }
        duration = self.parse_timeframe(timeframe)
        if limit is None:
            if since is None:
                raise ArgumentsRequired(self.id + ' fetchOHLCV requires a since argument or a limit argument')
            else:
                limit = 1000
                start = int(since / 1000)
                request['start'] = start
                request['end'] = self.sum(start, limit * duration)
                request['limit'] = limit
        else:
            if since is not None:
                start = int(since / 1000)
                request['start'] = start
                request['end'] = self.sum(start, limit * duration)
            request['limit'] = min(limit, 1000)  # min 1, max 1000
        response = await self.publicGetOhlcPair(self.extend(request, params))
        #
        #     {
        #         "data": {
        #             "pair": "BTC/USD",
        #             "ohlc": [
        #                 {"high": "9064.77", "timestamp": "1593961440", "volume": "18.49436608", "low": "9040.87", "close": "9064.77", "open": "9040.87"},
        #                 {"high": "9071.59", "timestamp": "1593961500", "volume": "3.48631711", "low": "9058.76", "close": "9061.07", "open": "9064.66"},
        #                 {"high": "9067.33", "timestamp": "1593961560", "volume": "0.04142833", "low": "9061.94", "close": "9061.94", "open": "9067.33"},
        #             ],
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        ohlc = self.safe_value(data, 'ohlc', [])
        return self.parse_ohlcvs(ohlc, market, timeframe, since, limit)

    async def fetch_balance(self, params={}):
        await self.load_markets()
        balance = await self.privatePostBalance(params)
        result = {'info': balance}
        codes = list(self.currencies.keys())
        for i in range(0, len(codes)):
            code = codes[i]
            currency = self.currency(code)
            currencyId = currency['id']
            account = self.account()
            account['free'] = self.safe_float(balance, currencyId + '_available')
            account['used'] = self.safe_float(balance, currencyId + '_reserved')
            account['total'] = self.safe_float(balance, currencyId + '_balance')
            result[code] = account
        return self.parse_balance(result)

    async def fetch_trading_fee(self, symbol, params={}):
        await self.load_markets()
        request = {}
        method = 'privatePostBalance'
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['pair'] = market['id']
            method += 'Pair'
        balance = await getattr(self, method)(self.extend(request, params))
        return {
            'info': balance,
            'symbol': symbol,
            'maker': balance['fee'],
            'taker': balance['fee'],
        }

    def prase_trading_fees(self, balance):
        result = {'info': balance}
        markets = list(self.markets.keys())
        for i in range(0, len(markets)):
            symbol = markets[i]
            fee = self.parse_trading_fee(balance, symbol)
            result[symbol] = fee
        return result

    async def fetch_trading_fees(self, params={}):
        await self.load_markets()
        balance = await self.privatePostBalance(params)
        return self.prase_trading_fees(balance)

    def parse_funding_fees(self, balance):
        withdraw = {}
        ids = list(balance.keys())
        for i in range(0, len(ids)):
            id = ids[i]
            if id.find('_withdrawal_fee') >= 0:
                currencyId = id.split('_')[0]
                code = self.safe_currency_code(currencyId)
                withdraw[code] = self.safe_float(balance, id)
        return {
            'info': balance,
            'withdraw': withdraw,
            'deposit': {},
        }

    async def fetch_funding_fees(self, params={}):
        await self.load_markets()
        balance = await self.privatePostBalance(params)
        return self.parse_funding_fees(balance)

    async def fetch_fees(self, params={}):
        await self.load_markets()
        balance = await self.privatePostBalance(params)
        tradingFees = self.prase_trading_fees(balance)
        del tradingFees['info']
        fundingFees = self.parse_funding_fees(balance)
        del fundingFees['info']
        return {
            'info': balance,
            'trading': tradingFees,
            'funding': fundingFees,
        }

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        method = 'privatePost' + self.capitalize(side)
        request = {
            'pair': market['id'],
            'amount': self.amount_to_precision(symbol, amount),
        }
        if type == 'market':
            method += 'Market'
        elif type == 'instant':
            method += 'Instant'
        else:
            request['price'] = self.price_to_precision(symbol, price)
        method += 'Pair'
        response = await getattr(self, method)(self.extend(request, params))
        order = self.parse_order(response, market)
        return self.extend(order, {
            'type': type,
        })

    async def cancel_order(self, id, symbol=None, params={}):
        await self.load_markets()
        request = {
            'id': id,
        }
        return await self.privatePostCancelOrder(self.extend(request, params))

    def parse_order_status(self, status):
        statuses = {
            'In Queue': 'open',
            'Open': 'open',
            'Finished': 'closed',
            'Canceled': 'canceled',
        }
        return self.safe_string(statuses, status, status)

    async def fetch_order_status(self, id, symbol=None, params={}):
        await self.load_markets()
        request = {
            'id': id,
        }
        response = await self.privatePostOrderStatus(self.extend(request, params))
        return self.parse_order_status(self.safe_string(response, 'status'))

    async def fetch_order(self, id, symbol=None, params={}):
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        request = {'id': id}
        response = await self.privatePostOrderStatus(self.extend(request, params))
        #
        #     {
        #         "status": "Finished",
        #         "id": 3047704374,
        #         "transactions": [
        #             {
        #                 "usd": "6.0134400000000000",
        #                 "price": "4008.96000000",
        #                 "datetime": "2019-03-28 23:07:37.233599",
        #                 "fee": "0.02",
        #                 "btc": "0.00150000",
        #                 "tid": 84452058,
        #                 "type": 2
        #             }
        #         ]
        #     }
        return self.parse_order(response, market)

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {}
        method = 'privatePostUserTransactions'
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['pair'] = market['id']
            method += 'Pair'
        if limit is not None:
            request['limit'] = limit
        response = await getattr(self, method)(self.extend(request, params))
        result = self.filter_by(response, 'type', '2')
        return self.parse_trades(result, market, since, limit)

    async def fetch_transactions(self, code=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {}
        if limit is not None:
            request['limit'] = limit
        response = await self.privatePostUserTransactions(self.extend(request, params))
        #
        #     [
        #         {
        #             "fee": "0.00000000",
        #             "btc_usd": "0.00",
        #             "id": 1234567894,
        #             "usd": 0,
        #             "btc": 0,
        #             "datetime": "2018-09-08 09:00:31",
        #             "type": "1",
        #             "xrp": "-20.00000000",
        #             "eur": 0,
        #         },
        #         {
        #             "fee": "0.00000000",
        #             "btc_usd": "0.00",
        #             "id": 1134567891,
        #             "usd": 0,
        #             "btc": 0,
        #             "datetime": "2018-09-07 18:47:52",
        #             "type": "0",
        #             "xrp": "20.00000000",
        #             "eur": 0,
        #         },
        #     ]
        #
        currency = None
        if code is not None:
            currency = self.currency(code)
        transactions = self.filter_by_array(response, 'type', ['0', '1'], False)
        return self.parse_transactions(transactions, currency, since, limit)

    async def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {}
        if since is not None:
            request['timedelta'] = self.milliseconds() - since
        else:
            request['timedelta'] = 50000000  # use max bitstamp approved value
        response = await self.privatePostWithdrawalRequests(self.extend(request, params))
        #
        #     [
        #         {
        #             status: 2,
        #             datetime: '2018-10-17 10:58:13',
        #             currency: 'BTC',
        #             amount: '0.29669259',
        #             address: 'aaaaa',
        #             type: 1,
        #             id: 111111,
        #             transaction_id: 'xxxx',
        #         },
        #         {
        #             status: 2,
        #             datetime: '2018-10-17 10:55:17',
        #             currency: 'ETH',
        #             amount: '1.11010664',
        #             address: 'aaaa',
        #             type: 16,
        #             id: 222222,
        #             transaction_id: 'xxxxx',
        #         },
        #     ]
        #
        return self.parse_transactions(response, None, since, limit)

    def parse_transaction(self, transaction, currency=None):
        #
        # fetchTransactions
        #
        #     {
        #         "fee": "0.00000000",
        #         "btc_usd": "0.00",
        #         "id": 1234567894,
        #         "usd": 0,
        #         "btc": 0,
        #         "datetime": "2018-09-08 09:00:31",
        #         "type": "1",
        #         "xrp": "-20.00000000",
        #         "eur": 0,
        #     }
        #
        # fetchWithdrawals
        #
        #     {
        #         status: 2,
        #         datetime: '2018-10-17 10:58:13',
        #         currency: 'BTC',
        #         amount: '0.29669259',
        #         address: 'aaaaa',
        #         type: 1,
        #         id: 111111,
        #         transaction_id: 'xxxx',
        #     }
        #
        #     {
        #         "id": 3386432,
        #         "type": 14,
        #         "amount": "863.21332500",
        #         "status": 2,
        #         "address": "rE1sdh25BJQ3qFwngiTBwaq3zPGGYcrjp1?dt=1455",
        #         "currency": "XRP",
        #         "datetime": "2018-01-05 15:27:55",
        #         "transaction_id": "001743B03B0C79BA166A064AC0142917B050347B4CB23BA2AB4B91B3C5608F4C"
        #     }
        #
        timestamp = self.parse8601(self.safe_string(transaction, 'datetime'))
        id = self.safe_string(transaction, 'id')
        currencyId = self.get_currency_id_from_transaction(transaction)
        code = self.safe_currency_code(currencyId, currency)
        feeCost = self.safe_float(transaction, 'fee')
        feeCurrency = None
        amount = None
        if 'amount' in transaction:
            amount = self.safe_float(transaction, 'amount')
        elif currency is not None:
            amount = self.safe_float(transaction, currency['id'], amount)
            feeCurrency = currency['code']
        elif (code is not None) and (currencyId is not None):
            amount = self.safe_float(transaction, currencyId, amount)
            feeCurrency = code
        if amount is not None:
            # withdrawals have a negative amount
            amount = abs(amount)
        status = 'ok'
        if 'status' in transaction:
            status = self.parse_transaction_status(self.safe_string(transaction, 'status'))
        type = None
        if 'type' in transaction:
            # from fetchTransactions
            rawType = self.safe_string(transaction, 'type')
            if rawType == '0':
                type = 'deposit'
            elif rawType == '1':
                type = 'withdrawal'
        else:
            # from fetchWithdrawals
            type = 'withdrawal'
        txid = self.safe_string(transaction, 'transaction_id')
        tag = None
        address = self.safe_string(transaction, 'address')
        if address is not None:
            # dt(destination tag) is embedded into the address field
            addressParts = address.split('?dt=')
            numParts = len(addressParts)
            if numParts > 1:
                address = addressParts[0]
                tag = addressParts[1]
        addressFrom = None
        addressTo = address
        tagFrom = None
        tagTo = tag
        fee = None
        if feeCost is not None:
            fee = {
                'currency': feeCurrency,
                'cost': feeCost,
                'rate': None,
            }
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'addressFrom': addressFrom,
            'addressTo': addressTo,
            'address': address,
            'tagFrom': tagFrom,
            'tagTo': tagTo,
            'tag': tag,
            'type': type,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': None,
            'fee': fee,
        }

    def parse_transaction_status(self, status):
        # withdrawals:
        # 0(open), 1(in process), 2(finished), 3(canceled) or 4(failed).
        statuses = {
            '0': 'pending',  # Open
            '1': 'pending',  # In process
            '2': 'ok',  # Finished
            '3': 'canceled',  # Canceled
            '4': 'failed',  # Failed
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        # from fetch order:
        #   {status: 'Finished',
        #     id: 731693945,
        #     transactions:
        #     [{fee: '0.000019',
        #         price: '0.00015803',
        #         datetime: '2018-01-07 10:45:34.132551',
        #         btc: '0.0079015000000000',
        #         tid: 42777395,
        #         type: 2,
        #         xrp: '50.00000000'}]}
        #
        # partially filled order:
        #   {"id": 468646390,
        #     "status": "Canceled",
        #     "transactions": [{
        #         "eth": "0.23000000",
        #         "fee": "0.09",
        #         "tid": 25810126,
        #         "usd": "69.8947000000000000",
        #         "type": 2,
        #         "price": "303.89000000",
        #         "datetime": "2017-11-11 07:22:20.710567"
        #     }]}
        #
        # from create order response:
        #     {
        #         price: '0.00008012',
        #         currency_pair: 'XRP/BTC',
        #         datetime: '2019-01-31 21:23:36',
        #         amount: '15.00000000',
        #         type: '0',
        #         id: '2814205012'
        #     }
        #
        id = self.safe_string(order, 'id')
        side = self.safe_string(order, 'type')
        if side is not None:
            side = 'sell' if (side == '1') else 'buy'
        # there is no timestamp from fetchOrder
        timestamp = self.parse8601(self.safe_string(order, 'datetime'))
        lastTradeTimestamp = None
        symbol = None
        marketId = self.safe_string_lower(order, 'currency_pair')
        if marketId is not None:
            marketId = marketId.replace('/', '')
            if marketId in self.markets_by_id:
                market = self.markets_by_id[marketId]
                symbol = market['symbol']
        amount = self.safe_float(order, 'amount')
        filled = 0.0
        trades = []
        transactions = self.safe_value(order, 'transactions', [])
        feeCost = None
        cost = None
        numTransactions = len(transactions)
        if numTransactions > 0:
            feeCost = 0.0
            for i in range(0, numTransactions):
                trade = self.parse_trade(self.extend({
                    'order_id': id,
                    'side': side,
                }, transactions[i]), market)
                filled = self.sum(filled, trade['amount'])
                feeCost = self.sum(feeCost, trade['fee']['cost'])
                if cost is None:
                    cost = 0.0
                cost = self.sum(cost, trade['cost'])
                trades.append(trade)
            lastTradeTimestamp = trades[numTransactions - 1]['timestamp']
        status = self.parse_order_status(self.safe_string(order, 'status'))
        if (status == 'closed') and (amount is None):
            amount = filled
        remaining = None
        if amount is not None:
            remaining = amount - filled
        price = self.safe_float(order, 'price')
        if market is None:
            market = self.get_market_from_trades(trades)
        feeCurrency = None
        if market is not None:
            if symbol is None:
                symbol = market['symbol']
            feeCurrency = market['quote']
        if cost is None:
            if price is not None:
                cost = price * filled
        elif price is None:
            if filled > 0:
                price = cost / filled
        fee = None
        if feeCost is not None:
            if feeCurrency is not None:
                fee = {
                    'cost': feeCost,
                    'currency': feeCurrency,
                }
        return {
            'id': id,
            'clientOrderId': None,
            'datetime': self.iso8601(timestamp),
            'timestamp': timestamp,
            'lastTradeTimestamp': lastTradeTimestamp,
            'status': status,
            'symbol': symbol,
            'type': None,
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': price,
            'stopPrice': None,
            'cost': cost,
            'amount': amount,
            'filled': filled,
            'remaining': remaining,
            'trades': trades,
            'fee': fee,
            'info': order,
            'average': None,
        }

    def parse_ledger_entry_type(self, type):
        types = {
            '0': 'transaction',
            '1': 'transaction',
            '2': 'trade',
            '14': 'transfer',
        }
        return self.safe_string(types, type, type)

    def parse_ledger_entry(self, item, currency=None):
        #
        #     [
        #         {
        #             "fee": "0.00000000",
        #             "btc_usd": "0.00",
        #             "id": 1234567894,
        #             "usd": 0,
        #             "btc": 0,
        #             "datetime": "2018-09-08 09:00:31",
        #             "type": "1",
        #             "xrp": "-20.00000000",
        #             "eur": 0,
        #         },
        #         {
        #             "fee": "0.00000000",
        #             "btc_usd": "0.00",
        #             "id": 1134567891,
        #             "usd": 0,
        #             "btc": 0,
        #             "datetime": "2018-09-07 18:47:52",
        #             "type": "0",
        #             "xrp": "20.00000000",
        #             "eur": 0,
        #         },
        #     ]
        #
        type = self.parse_ledger_entry_type(self.safe_string(item, 'type'))
        if type == 'trade':
            parsedTrade = self.parse_trade(item)
            market = None
            keys = list(item.keys())
            for i in range(0, len(keys)):
                if keys[i].find('_') >= 0:
                    marketId = keys[i].replace('_', '')
                    if marketId in self.markets_by_id:
                        market = self.markets_by_id[marketId]
            # if the market is still not defined
            # try to deduce it from used keys
            if market is None:
                market = self.get_market_from_trade(item)
            direction = parsedTrade['side'] == 'in' if 'buy' else 'out'
            return {
                'id': parsedTrade['id'],
                'info': item,
                'timestamp': parsedTrade['timestamp'],
                'datetime': parsedTrade['datetime'],
                'direction': direction,
                'account': None,
                'referenceId': parsedTrade['order'],
                'referenceAccount': None,
                'type': type,
                'currency': market['base'],
                'amount': parsedTrade['amount'],
                'before': None,
                'after': None,
                'status': 'ok',
                'fee': parsedTrade['fee'],
            }
        else:
            parsedTransaction = self.parse_transaction(item)
            direction = None
            if 'amount' in item:
                amount = self.safe_float(item, 'amount')
                direction = amount > 'in' if 0 else 'out'
            elif ('currency' in parsedTransaction) and parsedTransaction['currency'] is not None:
                currencyId = self.currency_id(parsedTransaction['currency'])
                amount = self.safe_float(item, currencyId)
                direction = amount > 'in' if 0 else 'out'
            return {
                'id': parsedTransaction['id'],
                'info': item,
                'timestamp': parsedTransaction['timestamp'],
                'datetime': parsedTransaction['datetime'],
                'direction': direction,
                'account': None,
                'referenceId': parsedTransaction['txid'],
                'referenceAccount': None,
                'type': type,
                'currency': parsedTransaction['currency'],
                'amount': parsedTransaction['amount'],
                'before': None,
                'after': None,
                'status': parsedTransaction['status'],
                'fee': parsedTransaction['fee'],
            }

    async def fetch_ledger(self, code=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {}
        if limit is not None:
            request['limit'] = limit
        response = await self.privatePostUserTransactions(self.extend(request, params))
        currency = None
        if code is not None:
            currency = self.currency(code)
        return self.parse_ledger(response, currency, since, limit)

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        market = None
        await self.load_markets()
        if symbol is not None:
            market = self.market(symbol)
        response = await self.privatePostOpenOrdersAll(params)
        #     [
        #         {
        #             price: '0.00008012',
        #             currency_pair: 'XRP/BTC',
        #             datetime: '2019-01-31 21:23:36',
        #             amount: '15.00000000',
        #             type: '0',
        #             id: '2814205012',
        #         }
        #     ]
        #
        return self.parse_orders(response, market, since, limit, {
            'status': 'open',
            'type': 'limit',
        })

    def get_currency_name(self, code):
        if code == 'BTC':
            return 'bitcoin'
        return code.lower()

    def is_fiat(self, code):
        return code == 'USD' or code == 'EUR' or code == 'GBP'

    async def fetch_deposit_address(self, code, params={}):
        if self.is_fiat(code):
            raise NotSupported(self.id + ' fiat fetchDepositAddress() for ' + code + ' is not supported!')
        name = self.get_currency_name(code)
        v1 = (code == 'BTC')
        method = 'v1' if v1 else 'private'  # v1 or v2
        method += 'Post' + self.capitalize(name)
        method += 'Deposit' if v1 else ''
        method += 'Address'
        response = await getattr(self, method)(params)
        if v1:
            response = json.loads(response)
        address = response if v1 else self.safe_string(response, 'address')
        tag = None if v1 else self.safe_string_2(response, 'memo_id', 'destination_tag')
        self.check_address(address)
        return {
            'currency': code,
            'address': address,
            'tag': tag,
            'info': response,
        }

    async def withdraw(self, code, amount, address, tag=None, params={}):
        # For fiat withdrawals please provide all required additional parameters in the 'params'
        # Check https://www.bitstamp.net/api/ under 'Open bank withdrawal' for list and description.
        await self.load_markets()
        self.check_address(address)
        request = {
            'amount': amount,
        }
        method = None
        if not self.is_fiat(code):
            name = self.get_currency_name(code)
            v1 = (code == 'BTC')
            method = 'v1' if v1 else 'private'  # v1 or v2
            method += 'Post' + self.capitalize(name) + 'Withdrawal'
            if code == 'XRP':
                if tag is not None:
                    request['destination_tag'] = tag
            request['address'] = address
        else:
            method = 'privatePostWithdrawalOpen'
            currency = self.currency(code)
            request['iban'] = address
            request['account_currency'] = currency['id']
        response = await getattr(self, method)(self.extend(request, params))
        return {
            'info': response,
            'id': response['id'],
        }

    def nonce(self):
        return self.milliseconds()

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        url = self.urls['api'][api] + '/'
        if api != 'v1':
            url += self.version + '/'
        url += self.implode_params(path, params)
        query = self.omit(params, self.extract_params(path))
        if api == 'public':
            if query:
                url += '?' + self.urlencode(query)
        else:
            self.check_required_credentials()
            authVersion = self.safe_value(self.options, 'auth', 'v2')
            if (authVersion == 'v1') or (api == 'v1'):
                nonce = str(self.nonce())
                auth = nonce + self.uid + self.apiKey
                signature = self.encode(self.hmac(self.encode(auth), self.encode(self.secret)))
                query = self.extend({
                    'key': self.apiKey,
                    'signature': signature.upper(),
                    'nonce': nonce,
                }, query)
                body = self.urlencode(query)
                headers = {
                    'Content-Type': 'application/x-www-form-urlencoded',
                }
            else:
                xAuth = 'BITSTAMP ' + self.apiKey
                xAuthNonce = self.uuid()
                xAuthTimestamp = str(self.milliseconds())
                xAuthVersion = 'v2'
                contentType = ''
                headers = {
                    'X-Auth': xAuth,
                    'X-Auth-Nonce': xAuthNonce,
                    'X-Auth-Timestamp': xAuthTimestamp,
                    'X-Auth-Version': xAuthVersion,
                }
                if method == 'POST':
                    if query:
                        body = self.urlencode(query)
                        contentType = 'application/x-www-form-urlencoded'
                        headers['Content-Type'] = contentType
                    else:
                        # sending an empty POST request will trigger
                        # an API0020 error returned by the exchange
                        # therefore for empty requests we send a dummy object
                        # https://github.com/ccxt/ccxt/issues/6846
                        body = self.urlencode({'foo': 'bar'})
                        contentType = 'application/x-www-form-urlencoded'
                        headers['Content-Type'] = contentType
                authBody = body if body else ''
                auth = xAuth + method + url.replace('https://', '') + contentType + xAuthNonce + xAuthTimestamp + xAuthVersion + authBody
                signature = self.hmac(self.encode(auth), self.encode(self.secret))
                headers['X-Auth-Signature'] = signature
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, httpCode, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return
        #
        #     {"error": "No permission found"}  # fetchDepositAddress returns self on apiKeys that don't have the permission required
        #     {"status": "error", "reason": {"__all__": ["Minimum order size is 5.0 EUR."]}}
        #     reuse of a nonce gives: {status: 'error', reason: 'Invalid nonce', code: 'API0004'}
        status = self.safe_string(response, 'status')
        error = self.safe_value(response, 'error')
        if (status == 'error') or (error is not None):
            errors = []
            if isinstance(error, basestring):
                errors.append(error)
            elif error is not None:
                keys = list(error.keys())
                for i in range(0, len(keys)):
                    key = keys[i]
                    value = self.safe_value(error, key)
                    if isinstance(value, list):
                        errors = self.array_concat(errors, value)
                    else:
                        errors.append(value)
            reason = self.safe_value(response, 'reason', {})
            if isinstance(reason, basestring):
                errors.append(reason)
            else:
                all = self.safe_value(reason, '__all__', [])
                for i in range(0, len(all)):
                    errors.append(all[i])
            code = self.safe_string(response, 'code')
            if code == 'API0005':
                raise AuthenticationError(self.id + ' invalid signature, use the uid for the main account if you have subaccounts')
            feedback = self.id + ' ' + body
            for i in range(0, len(errors)):
                value = errors[i]
                self.throw_exactly_matched_exception(self.exceptions['exact'], value, feedback)
                self.throw_broadly_matched_exception(self.exceptions['broad'], value, feedback)
            raise ExchangeError(feedback)
