"use strict";
const assert_1 = require("@aws-cdk/assert");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const apigw = require("../lib");
module.exports = {
    'LambdaRestApi defines a REST API with Lambda proxy integration'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const api = new apigw.LambdaRestApi(stack, 'lambda-rest-api', { handler });
        // THEN -- can't customize further
        test.throws(() => {
            api.root.addResource('cant-touch-this');
        });
        // THEN -- template proxies everything
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            'PathPart': '{proxy+}',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            'HttpMethod': 'ANY',
            'ResourceId': {
                'Ref': 'lambdarestapiproxyE3AE07E3',
            },
            'RestApiId': {
                'Ref': 'lambdarestapiAAD10924',
            },
            'AuthorizationType': 'NONE',
            'Integration': {
                'IntegrationHttpMethod': 'POST',
                'Type': 'AWS_PROXY',
                'Uri': {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                'Ref': 'AWS::Partition',
                            },
                            ':apigateway:',
                            {
                                'Ref': 'AWS::Region',
                            },
                            ':lambda:path/2015-03-31/functions/',
                            {
                                'Fn::GetAtt': [
                                    'handlerE1533BD5',
                                    'Arn',
                                ],
                            },
                            '/invocations',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'LambdaRestApi supports function Alias'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const alias = new lambda.Alias(stack, 'alias', {
            aliasName: 'my-alias',
            version: new lambda.Version(stack, 'version', {
                lambda: handler,
            }),
        });
        // WHEN
        const api = new apigw.LambdaRestApi(stack, 'lambda-rest-api', { handler: alias });
        // THEN -- can't customize further
        test.throws(() => {
            api.root.addResource('cant-touch-this');
        });
        // THEN -- template proxies everything
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            'PathPart': '{proxy+}',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            'HttpMethod': 'ANY',
            'ResourceId': {
                'Ref': 'lambdarestapiproxyE3AE07E3',
            },
            'RestApiId': {
                'Ref': 'lambdarestapiAAD10924',
            },
            'AuthorizationType': 'NONE',
            'Integration': {
                'IntegrationHttpMethod': 'POST',
                'Type': 'AWS_PROXY',
                'Uri': {
                    'Fn::Join': [
                        '',
                        [
                            'arn:',
                            {
                                'Ref': 'AWS::Partition',
                            },
                            ':apigateway:',
                            {
                                'Ref': 'AWS::Region',
                            },
                            ':lambda:path/2015-03-31/functions/',
                            {
                                'Ref': 'alias68BF17F5',
                            },
                            '/invocations',
                        ],
                    ],
                },
            },
        }));
        test.done();
    },
    'when "proxy" is set to false, users need to define the model'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        const api = new apigw.LambdaRestApi(stack, 'lambda-rest-api', { handler, proxy: false });
        const tasks = api.root.addResource('tasks');
        tasks.addMethod('GET');
        tasks.addMethod('POST');
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::ApiGateway::Resource', {
            'PathPart': '{proxy+}',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Resource', {
            PathPart: 'tasks',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'GET',
            ResourceId: { Ref: 'lambdarestapitasks224418C8' },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'POST',
            ResourceId: { Ref: 'lambdarestapitasks224418C8' },
        }));
        test.done();
    },
    'fails if options.defaultIntegration is also set'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        test.throws(() => new apigw.LambdaRestApi(stack, 'lambda-rest-api', {
            handler,
            options: { defaultIntegration: new apigw.HttpIntegration('https://foo/bar') },
        }), /Cannot specify \"defaultIntegration\" since Lambda integration is automatically defined/);
        test.throws(() => new apigw.LambdaRestApi(stack, 'lambda-rest-api', {
            handler,
            defaultIntegration: new apigw.HttpIntegration('https://foo/bar'),
        }), /Cannot specify \"defaultIntegration\" since Lambda integration is automatically defined/);
        test.done();
    },
    'LambdaRestApi defines a REST API with CORS enabled'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const handler = new lambda.Function(stack, 'handler', {
            handler: 'index.handler',
            code: lambda.Code.fromInline('boom'),
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        new apigw.LambdaRestApi(stack, 'lambda-rest-api', {
            handler,
            defaultCorsPreflightOptions: {
                allowOrigins: ['https://aws.amazon.com'],
                allowMethods: ['GET', 'PUT'],
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApiGateway::Method', {
            HttpMethod: 'OPTIONS',
            ResourceId: { Ref: 'lambdarestapiproxyE3AE07E3' },
            Integration: {
                IntegrationResponses: [
                    {
                        ResponseParameters: {
                            'method.response.header.Access-Control-Allow-Headers': "'Content-Type,X-Amz-Date,Authorization,X-Api-Key,X-Amz-Security-Token,X-Amz-User-Agent'",
                            'method.response.header.Access-Control-Allow-Origin': "'https://aws.amazon.com'",
                            'method.response.header.Vary': "'Origin'",
                            'method.response.header.Access-Control-Allow-Methods': "'GET,PUT'",
                        },
                        StatusCode: '204',
                    },
                ],
                RequestTemplates: {
                    'application/json': '{ statusCode: 200 }',
                },
                Type: 'MOCK',
            },
            MethodResponses: [
                {
                    ResponseParameters: {
                        'method.response.header.Access-Control-Allow-Headers': true,
                        'method.response.header.Access-Control-Allow-Origin': true,
                        'method.response.header.Vary': true,
                        'method.response.header.Access-Control-Allow-Methods': true,
                    },
                    StatusCode: '204',
                },
            ],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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