"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const s3_assets = require("@aws-cdk/aws-s3-assets");
/**
 * (experimental) Represents an OpenAPI definition asset.
 *
 * @experimental
 */
class ApiDefinition {
    /**
     * (experimental) Creates an API definition from a specification file in an S3 bucket.
     *
     * @experimental
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * (experimental) Create an API definition from an inline object.
     *
     * The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @experimental
     * @example
     *
     *   ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * (experimental) Loads the API specification from a local disk asset.
     *
     * @experimental
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
}
exports.ApiDefinition = ApiDefinition;
/**
 * (experimental) OpenAPI specification from an S3 archive.
 *
 * @experimental
 */
class S3ApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
/**
 * (experimental) OpenAPI specification from an inline JSON object.
 *
 * @experimental
 */
class InlineApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
/**
 * (experimental) OpenAPI specification from a local file.
 *
 * @experimental
 */
class AssetApiDefinition extends ApiDefinition {
    /**
     * @experimental
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    /**
     * (experimental) Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @experimental
     */
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
//# sourceMappingURL=data:application/json;base64,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