"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
/**
 * @stability stable
 */
class ResourceBase extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    /**
     * Defines a new child resource where this resource is the parent.
     *
     * @stability stable
     */
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    /**
     * Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.
     *
     * @stability stable
     */
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    /**
     * Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.
     *
     * Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
     * HTTP headers to tell browsers to give a web application running at one
     * origin, access to selected resources from a different origin. A web
     * application executes a cross-origin HTTP request when it requests a
     * resource that has a different origin (domain, protocol, or port) from its
     * own.
     *
     * @stability stable
     */
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    /**
     * Retrieves a child resource by path part.
     *
     * @stability stable
     */
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    /**
     * Gets or create all resources leading up to the specified path.
     *
     * - Path may only start with "/" if this method is called on the root resource.
     * - All resources are created using default options.
     *
     * @stability stable
     */
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
/**
 * @stability stable
 */
class Resource extends ResourceBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Import an existing resource.
     *
     * @stability stable
     */
    static fromResourceAttributes(scope, id, attrs) {
        class Import extends ResourceBase {
            constructor() {
                super(...arguments);
                this.api = attrs.restApi;
                this.resourceId = attrs.resourceId;
                this.path = attrs.path;
                this.defaultIntegration = undefined;
                this.defaultMethodOptions = undefined;
                this.defaultCorsPreflightOptions = undefined;
            }
            get parentResource() {
                throw new Error('parentResource is not configured for imported resource.');
            }
            get restApi() {
                throw new Error('restApi is not configured for imported resource.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (deprecated) The RestApi associated with this Resource.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
/**
 * Defines a {proxy+} greedy resource and an ANY method on a route.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 * @stability stable
 */
class ProxyResource extends Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    /**
     * Defines a new method for this resource.
     *
     * @stability stable
     */
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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