"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * Returns a token that resolves to the Rest Api Id at the time of synthesis.
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * Token based lambda authorizer that recognizes the caller's identity as a bearer token,
 * such as a JSON Web Token (JWT) or an OAuth token.
 * Based on the token, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * Request-based lambda authorizer that recognizes the caller's identity via request parameters,
 * such as headers, paths, query strings, stage variables, or context variables.
 * Based on the request, authorization is performed by a lambda function.
 *
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    return `arn:${core_1.Stack.of(handler).partition}:apigateway:${core_1.Stack.of(handler).region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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