"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainName = exports.SecurityPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const base_path_mapping_1 = require("./base-path-mapping");
const restapi_1 = require("./restapi");
/**
 * The minimum version of the SSL protocol that you want API Gateway to use for HTTPS connections.
 */
var SecurityPolicy;
(function (SecurityPolicy) {
    /** Cipher suite TLS 1.0 */
    SecurityPolicy["TLS_1_0"] = "TLS_1_0";
    /** Cipher suite TLS 1.2 */
    SecurityPolicy["TLS_1_2"] = "TLS_1_2";
})(SecurityPolicy = exports.SecurityPolicy || (exports.SecurityPolicy = {}));
class DomainName extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        const endpointType = props.endpointType || restapi_1.EndpointType.REGIONAL;
        const edge = endpointType === restapi_1.EndpointType.EDGE;
        if (!core_1.Token.isUnresolved(props.domainName) && /[A-Z]/.test(props.domainName)) {
            throw new Error('domainName does not support uppercase letters. ' +
                `got: '${props.domainName}'`);
        }
        const resource = new apigateway_generated_1.CfnDomainName(this, 'Resource', {
            domainName: props.domainName,
            certificateArn: edge ? props.certificate.certificateArn : undefined,
            regionalCertificateArn: edge ? undefined : props.certificate.certificateArn,
            endpointConfiguration: { types: [endpointType] },
            securityPolicy: props.securityPolicy,
        });
        this.domainName = resource.ref;
        this.domainNameAliasDomainName = edge
            ? resource.attrDistributionDomainName
            : resource.attrRegionalDomainName;
        this.domainNameAliasHostedZoneId = edge
            ? resource.attrDistributionHostedZoneId
            : resource.attrRegionalHostedZoneId;
        if (props.mapping) {
            this.addBasePathMapping(props.mapping);
        }
    }
    /**
     * Imports an existing domain name.
     */
    static fromDomainNameAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = attrs.domainName;
                this.domainNameAliasDomainName = attrs.domainNameAliasTarget;
                this.domainNameAliasHostedZoneId = attrs.domainNameAliasHostedZoneId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Maps this domain to an API endpoint.
     * @param targetApi That target API endpoint, requests will be mapped to the deployment stage.
     * @param options Options for mapping to base path with or without a stage
     */
    addBasePathMapping(targetApi, options = {}) {
        const basePath = options.basePath || '/';
        const id = `Map:${basePath}=>${targetApi.node.uniqueId}`;
        return new base_path_mapping_1.BasePathMapping(this, id, {
            domainName: this,
            restApi: targetApi,
            ...options,
        });
    }
}
exports.DomainName = DomainName;
//# sourceMappingURL=data:application/json;base64,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