"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("@aws-cdk/core");
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(props.integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_a = options.authorizationScopes) !== null && _a !== void 0 ? _a : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ method: methodProps });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled.
     *
     * @attribute
     */
    get methodArn() {
        if (!this.restApi.deploymentStage) {
            throw new Error(`Unable to determine ARN for method "${this.node.id}" since there is no stage associated with this API.\n` +
                'Either use the `deploy` prop or explicitly assign `deploymentStage` on the RestApi');
        }
        const stage = this.restApi.deploymentStage.stageName.toString();
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.restApi.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        if (!integration) {
            // use defaultIntegration from API if defined
            if (this.resource.defaultIntegration) {
                return this.renderIntegration(this.resource.defaultIntegration);
            }
            // fallback to mock
            return this.renderIntegration(new mock_1.MockIntegration());
        }
        integration.bind(this);
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibWV0aG9kLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsibWV0aG9kLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUEyRDtBQUMzRCxpRUFBbUU7QUFDbkUsNkNBQXVEO0FBQ3ZELCtDQUE0RDtBQUM1RCw4Q0FBc0Q7QUFLdEQsdUNBQTJEO0FBQzNELGlDQUE0QztBQWlKNUMsTUFBYSxNQUFPLFNBQVEsZUFBUTtJQVdsQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCOztRQUMxRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUMvQixJQUFJLENBQUMsR0FBRyxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDO1FBQzlCLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxXQUFXLEVBQUUsQ0FBQztRQUVqRCx5QkFBa0IsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFcEMsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUM7UUFFcEMsTUFBTSxvQkFBb0IsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDLG9CQUFvQixJQUFJLEVBQUUsQ0FBQztRQUN2RSxNQUFNLFVBQVUsR0FBRyxPQUFPLENBQUMsVUFBVSxJQUFJLG9CQUFvQixDQUFDLFVBQVUsQ0FBQztRQUN6RSxNQUFNLFlBQVksR0FBRyxVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsWUFBWSxDQUFDO1FBRTlDLE1BQU0sdUJBQXVCLEdBQUcsT0FBTyxDQUFDLGlCQUFpQixJQUFJLG9CQUFvQixDQUFDLGlCQUFpQixDQUFDO1FBQ3BHLE1BQU0saUJBQWlCLEdBQUcsQ0FBQSxVQUFVLGFBQVYsVUFBVSx1QkFBVixVQUFVLENBQUUsaUJBQWlCLEtBQUksdUJBQXVCLElBQUksaUJBQWlCLENBQUMsSUFBSSxDQUFDO1FBRTdHLHdIQUF3SDtRQUN4SCxJQUFJLENBQUEsVUFBVSxhQUFWLFVBQVUsdUJBQVYsVUFBVSxDQUFFLGlCQUFpQixLQUFJLHVCQUF1QixJQUFJLENBQUEsVUFBVSxhQUFWLFVBQVUsdUJBQVYsVUFBVSxDQUFFLGlCQUFpQixNQUFLLHVCQUF1QixFQUFFO1lBQ3pILE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLElBQUksQ0FBQyxVQUFVLG1DQUFtQyx1QkFBdUIsR0FBRztnQkFDOUcsK0RBQStELFVBQVUsQ0FBQyxpQkFBaUIsR0FBRyxDQUFDLENBQUM7U0FDbkc7UUFFRCxJQUFJLHVCQUFVLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQ3ZDLFVBQVUsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ25DO1FBRUQsTUFBTSxXQUFXLEdBQW1CO1lBQ2xDLFVBQVUsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFVBQVU7WUFDckMsU0FBUyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUztZQUM3QixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsYUFBYSxFQUFFLE9BQU8sQ0FBQyxhQUFhLElBQUksb0JBQW9CLENBQUMsYUFBYTtZQUMxRSxjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWMsSUFBSSxvQkFBb0IsQ0FBQyxjQUFjO1lBQzdFLGlCQUFpQjtZQUNqQixZQUFZO1lBQ1osaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQixJQUFJLG9CQUFvQixDQUFDLGlCQUFpQjtZQUN0RixXQUFXLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUM7WUFDdEQsZUFBZSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDO1lBQ3BFLGFBQWEsRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsT0FBTyxDQUFDLGFBQWEsQ0FBQztZQUM5RCxrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDO1lBQ3BELG1CQUFtQixRQUFFLE9BQU8sQ0FBQyxtQkFBbUIsbUNBQUksb0JBQW9CLENBQUMsbUJBQW1CO1NBQzdGLENBQUM7UUFFRixNQUFNLFFBQVEsR0FBRyxJQUFJLGdDQUFTLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUU5RCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFN0IsSUFBSSxxQkFBVyxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ2xELEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUN4QztRQUVELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDO1FBQ3ZELElBQUksVUFBVSxFQUFFO1lBQ2QsVUFBVSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7WUFDeEMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO1NBQ3BEO0lBQ0gsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDO0lBQy9CLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSCxJQUFXLFNBQVM7UUFDbEIsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZUFBZSxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQ2IsdUNBQXVDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSx1REFBdUQ7Z0JBQzFHLG9GQUFvRixDQUFDLENBQUM7U0FDekY7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDaEUsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDL0YsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsYUFBYTtRQUN0QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxtQkFBbUIsQ0FBQyxDQUFDO0lBQzdHLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxXQUF5QjtRQUNqRCxJQUFJLENBQUMsV0FBVyxFQUFFO1lBQ2hCLDZDQUE2QztZQUM3QyxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ3BDLE9BQU8sSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsQ0FBQzthQUNqRTtZQUVELG1CQUFtQjtZQUNuQixPQUFPLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLHNCQUFlLEVBQUUsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV2QixNQUFNLE9BQU8sR0FBRyxXQUFXLENBQUMsTUFBTSxDQUFDLE9BQU8sSUFBSSxFQUFHLENBQUM7UUFFbEQsSUFBSSxXQUFXLENBQUM7UUFDaEIsSUFBSSxPQUFPLENBQUMsc0JBQXNCLEtBQUssU0FBUyxJQUFJLE9BQU8sQ0FBQyxlQUFlLEtBQUssU0FBUyxFQUFFO1lBQ3pGLE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQztTQUM5RjtRQUVELElBQUksT0FBTyxDQUFDLGNBQWMsS0FBSyw0QkFBYyxDQUFDLFFBQVEsSUFBSSxPQUFPLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUN2RixNQUFNLElBQUksS0FBSyxDQUFDLG9FQUFvRSxDQUFDLENBQUM7U0FDdkY7UUFFRCxJQUFJLE9BQU8sQ0FBQyxjQUFjLEtBQUssNEJBQWMsQ0FBQyxRQUFRLElBQUksT0FBTyxDQUFDLE9BQU8sS0FBSyxTQUFTLEVBQUU7WUFDdkYsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1NBQ2hGO1FBRUQsSUFBSSxPQUFPLENBQUMsZUFBZSxFQUFFO1lBQzNCLFdBQVcsR0FBRyxPQUFPLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQztTQUMvQzthQUFNLElBQUksT0FBTyxDQUFDLHNCQUFzQixFQUFFO1lBQ3pDLHdCQUF3QjtZQUN4QiwyQ0FBMkM7WUFDM0MsV0FBVyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsUUFBUSxFQUFFLE1BQU0sRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLFlBQVksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ3JJO1FBRUQsT0FBTztZQUNMLElBQUksRUFBRSxXQUFXLENBQUMsTUFBTSxDQUFDLElBQUk7WUFDN0IsR0FBRyxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsR0FBRztZQUMzQixrQkFBa0IsRUFBRSxPQUFPLENBQUMsa0JBQWtCO1lBQzlDLGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYztZQUN0QyxlQUFlLEVBQUUsT0FBTyxDQUFDLGVBQWU7WUFDeEMscUJBQXFCLEVBQUUsV0FBVyxDQUFDLE1BQU0sQ0FBQyxxQkFBcUI7WUFDL0QsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQjtZQUM1QyxnQkFBZ0IsRUFBRSxPQUFPLENBQUMsZ0JBQWdCO1lBQzFDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxtQkFBbUI7WUFDaEQsb0JBQW9CLEVBQUUsT0FBTyxDQUFDLG9CQUFvQjtZQUNsRCxjQUFjLEVBQUUsT0FBTyxDQUFDLGNBQWM7WUFDdEMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ3JFLFdBQVc7U0FDWixDQUFDO0lBQ0osQ0FBQztJQUVPLHFCQUFxQixDQUFDLGVBQTZDO1FBQ3pFLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDcEIsdUJBQXVCO1lBQ3ZCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsT0FBTyxlQUFlLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFO1lBQzlCLElBQUksY0FBMkQsQ0FBQztZQUVoRSxJQUFJLEVBQUUsQ0FBQyxjQUFjLEVBQUU7Z0JBQ3JCLGNBQWMsR0FBRyxFQUFFLENBQUM7Z0JBQ3BCLEtBQUssTUFBTSxXQUFXLElBQUksRUFBRSxDQUFDLGNBQWMsRUFBRTtvQkFDM0MsSUFBSSxFQUFFLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRTt3QkFDakQsY0FBYyxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUMsT0FBTyxDQUFDO3FCQUN0RTtpQkFDRjthQUNGO1lBRUQsTUFBTSxrQkFBa0IsR0FBRztnQkFDekIsVUFBVSxFQUFFLEVBQUUsQ0FBQyxVQUFVO2dCQUN6QixrQkFBa0IsRUFBRSxFQUFFLENBQUMsa0JBQWtCO2dCQUN6QyxjQUFjO2FBQ2YsQ0FBQztZQUVGLE9BQU8sa0JBQWtCLENBQUM7UUFDNUIsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sbUJBQW1CLENBQUMsYUFBc0Q7UUFDaEYsSUFBSSxDQUFDLGFBQWEsRUFBRTtZQUNsQix1QkFBdUI7WUFDdkIsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxNQUFNLE1BQU0sR0FBOEIsRUFBRSxDQUFDO1FBQzdDLEtBQUssTUFBTSxXQUFXLElBQUksYUFBYSxFQUFFO1lBQ3ZDLElBQUksYUFBYSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDN0MsTUFBTSxDQUFDLFdBQVcsQ0FBQyxHQUFHLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUM7YUFDMUQ7U0FDRjtRQUVELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUM7SUFFTyxrQkFBa0IsQ0FBQyxPQUFzQjs7UUFDL0MsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLHVCQUF1QixFQUFFO1lBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMsb0ZBQW9GLENBQUMsQ0FBQztTQUN2RztRQUVELElBQUksT0FBTyxDQUFDLHVCQUF1QixFQUFFO1lBQ25DLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsbUJBQW1CLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1lBQ2pHLE9BQU8sU0FBUyxDQUFDLGtCQUFrQixDQUFDO1NBQ3JDO1FBRUQsNkJBQTZCO1FBQzdCLGFBQU8sT0FBTyxDQUFDLGdCQUFnQiwwQ0FBRSxrQkFBa0IsQ0FBQztJQUN0RCxDQUFDO0NBQ0Y7QUF6TkQsd0JBeU5DO0FBRUQsSUFBWSxpQkFvQlg7QUFwQkQsV0FBWSxpQkFBaUI7SUFDM0I7O09BRUc7SUFDSCxrQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCxvQ0FBZSxDQUFBO0lBRWY7O09BRUc7SUFDSCxzQ0FBaUIsQ0FBQTtJQUVqQjs7T0FFRztJQUNILG1EQUE4QixDQUFBO0FBQ2hDLENBQUMsRUFwQlcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFvQjVCO0FBRUQsU0FBUyxVQUFVLENBQUMsSUFBWTtJQUM5QixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUMsMERBQTBEO0FBQ3JHLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBDb25zdHJ1Y3QsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuTWV0aG9kLCBDZm5NZXRob2RQcm9wcyB9IGZyb20gJy4vYXBpZ2F0ZXdheS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQXV0aG9yaXplciwgSUF1dGhvcml6ZXIgfSBmcm9tICcuL2F1dGhvcml6ZXInO1xuaW1wb3J0IHsgQ29ubmVjdGlvblR5cGUsIEludGVncmF0aW9uIH0gZnJvbSAnLi9pbnRlZ3JhdGlvbic7XG5pbXBvcnQgeyBNb2NrSW50ZWdyYXRpb24gfSBmcm9tICcuL2ludGVncmF0aW9ucy9tb2NrJztcbmltcG9ydCB7IE1ldGhvZFJlc3BvbnNlIH0gZnJvbSAnLi9tZXRob2RyZXNwb25zZSc7XG5pbXBvcnQgeyBJTW9kZWwgfSBmcm9tICcuL21vZGVsJztcbmltcG9ydCB7IElSZXF1ZXN0VmFsaWRhdG9yLCBSZXF1ZXN0VmFsaWRhdG9yT3B0aW9ucyB9IGZyb20gJy4vcmVxdWVzdHZhbGlkYXRvcic7XG5pbXBvcnQgeyBJUmVzb3VyY2UgfSBmcm9tICcuL3Jlc291cmNlJztcbmltcG9ydCB7IElSZXN0QXBpLCBSZXN0QXBpLCBSZXN0QXBpQmFzZSB9IGZyb20gJy4vcmVzdGFwaSc7XG5pbXBvcnQgeyB2YWxpZGF0ZUh0dHBNZXRob2QgfSBmcm9tICcuL3V0aWwnO1xuXG5leHBvcnQgaW50ZXJmYWNlIE1ldGhvZE9wdGlvbnMge1xuICAvKipcbiAgICogQSBmcmllbmRseSBvcGVyYXRpb24gbmFtZSBmb3IgdGhlIG1ldGhvZC4gRm9yIGV4YW1wbGUsIHlvdSBjYW4gYXNzaWduIHRoZVxuICAgKiBPcGVyYXRpb25OYW1lIG9mIExpc3RQZXRzIGZvciB0aGUgR0VUIC9wZXRzIG1ldGhvZC5cbiAgICovXG4gIHJlYWRvbmx5IG9wZXJhdGlvbk5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE1ldGhvZCBhdXRob3JpemF0aW9uLlxuICAgKiBJZiB0aGUgdmFsdWUgaXMgc2V0IG9mIGBDdXN0b21gLCBhbiBgYXV0aG9yaXplcmAgbXVzdCBhbHNvIGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogSWYgeW91J3JlIHVzaW5nIG9uZSBvZiB0aGUgYXV0aG9yaXplcnMgdGhhdCBhcmUgYXZhaWxhYmxlIHZpYSB0aGUge0BsaW5rIEF1dGhvcml6ZXJ9IGNsYXNzLCBzdWNoIGFzIHtAbGluayBBdXRob3JpemVyI3Rva2VuKCl9LFxuICAgKiBpdCBpcyByZWNvbW1lbmRlZCB0aGF0IHRoaXMgb3B0aW9uIG5vdCBiZSBzcGVjaWZpZWQuIFRoZSBhdXRob3JpemVyIHdpbGwgdGFrZSBjYXJlIG9mIHNldHRpbmcgdGhlIGNvcnJlY3QgYXV0aG9yaXphdGlvbiB0eXBlLlxuICAgKiBIb3dldmVyLCBzcGVjaWZ5aW5nIGFuIGF1dGhvcml6YXRpb24gdHlwZSB1c2luZyB0aGlzIHByb3BlcnR5IHRoYXQgY29uZmxpY3RzIHdpdGggd2hhdCBpcyBleHBlY3RlZCBieSB0aGUge0BsaW5rIEF1dGhvcml6ZXJ9XG4gICAqIHdpbGwgcmVzdWx0IGluIGFuIGVycm9yLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG9wZW4gYWNjZXNzIHVubGVzcyBgYXV0aG9yaXplcmAgaXMgc3BlY2lmaWVkXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uVHlwZT86IEF1dGhvcml6YXRpb25UeXBlO1xuXG4gIC8qKlxuICAgKiBJZiBgYXV0aG9yaXphdGlvblR5cGVgIGlzIGBDdXN0b21gLCB0aGlzIHNwZWNpZmllcyB0aGUgSUQgb2YgdGhlIG1ldGhvZFxuICAgKiBhdXRob3JpemVyIHJlc291cmNlLlxuICAgKiBJZiBzcGVjaWZpZWQsIHRoZSB2YWx1ZSBvZiBgYXV0aG9yaXphdGlvblR5cGVgIG11c3QgYmUgc2V0IHRvIGBDdXN0b21gXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemVyPzogSUF1dGhvcml6ZXI7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoZSBtZXRob2QgcmVxdWlyZXMgY2xpZW50cyB0byBzdWJtaXQgYSB2YWxpZCBBUEkga2V5LlxuICAgKiBAZGVmYXVsdCBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgYXBpS2V5UmVxdWlyZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcmVzcG9uc2VzIHRoYXQgY2FuIGJlIHNlbnQgdG8gdGhlIGNsaWVudCB3aG8gY2FsbHMgdGhlIG1ldGhvZC5cbiAgICogQGRlZmF1bHQgTm9uZVxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IGlzIG5vdCByZXF1aXJlZCwgYnV0IGlmIHRoZXNlIGFyZSBub3Qgc3VwcGxpZWQgZm9yIGEgTGFtYmRhXG4gICAqIHByb3h5IGludGVncmF0aW9uLCB0aGUgTGFtYmRhIGZ1bmN0aW9uIG11c3QgcmV0dXJuIGEgdmFsdWUgb2YgdGhlIGNvcnJlY3QgZm9ybWF0LFxuICAgKiBmb3IgdGhlIGludGVncmF0aW9uIHJlc3BvbnNlIHRvIGJlIGNvcnJlY3RseSBtYXBwZWQgdG8gYSByZXNwb25zZSB0byB0aGUgY2xpZW50LlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1tZXRob2Qtc2V0dGluZ3MtbWV0aG9kLXJlc3BvbnNlLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IG1ldGhvZFJlc3BvbnNlcz86IE1ldGhvZFJlc3BvbnNlW107XG5cbiAgLyoqXG4gICAqIFRoZSByZXF1ZXN0IHBhcmFtZXRlcnMgdGhhdCBBUEkgR2F0ZXdheSBhY2NlcHRzLiBTcGVjaWZ5IHJlcXVlc3QgcGFyYW1ldGVyc1xuICAgKiBhcyBrZXktdmFsdWUgcGFpcnMgKHN0cmluZy10by1Cb29sZWFuIG1hcHBpbmcpLCB3aXRoIGEgc291cmNlIGFzIHRoZSBrZXkgYW5kXG4gICAqIGEgQm9vbGVhbiBhcyB0aGUgdmFsdWUuIFRoZSBCb29sZWFuIHNwZWNpZmllcyB3aGV0aGVyIGEgcGFyYW1ldGVyIGlzIHJlcXVpcmVkLlxuICAgKiBBIHNvdXJjZSBtdXN0IG1hdGNoIHRoZSBmb3JtYXQgbWV0aG9kLnJlcXVlc3QubG9jYXRpb24ubmFtZSwgd2hlcmUgdGhlIGxvY2F0aW9uXG4gICAqIGlzIHF1ZXJ5c3RyaW5nLCBwYXRoLCBvciBoZWFkZXIsIGFuZCBuYW1lIGlzIGEgdmFsaWQsIHVuaXF1ZSBwYXJhbWV0ZXIgbmFtZS5cbiAgICogQGRlZmF1bHQgTm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgcmVxdWVzdFBhcmFtZXRlcnM/OiB7IFtwYXJhbTogc3RyaW5nXTogYm9vbGVhbiB9O1xuXG4gIC8qKlxuICAgKiBUaGUgbW9kZWxzIHdoaWNoIGRlc2NyaWJlIGRhdGEgc3RydWN0dXJlIG9mIHJlcXVlc3QgcGF5bG9hZC4gV2hlblxuICAgKiBjb21iaW5lZCB3aXRoIGByZXF1ZXN0VmFsaWRhdG9yYCBvciBgcmVxdWVzdFZhbGlkYXRvck9wdGlvbnNgLCB0aGUgc2VydmljZVxuICAgKiB3aWxsIHZhbGlkYXRlIHRoZSBBUEkgcmVxdWVzdCBwYXlsb2FkIGJlZm9yZSBpdCByZWFjaGVzIHRoZSBBUEkncyBJbnRlZ3JhdGlvbiAoaW5jbHVkaW5nIHByb3hpZXMpLlxuICAgKiBTcGVjaWZ5IGByZXF1ZXN0TW9kZWxzYCBhcyBrZXktdmFsdWUgcGFpcnMsIHdpdGggYSBjb250ZW50IHR5cGVcbiAgICogKGUuZy4gYCdhcHBsaWNhdGlvbi9qc29uJ2ApIGFzIHRoZSBrZXkgYW5kIGFuIEFQSSBHYXRld2F5IE1vZGVsIGFzIHRoZSB2YWx1ZS5cbiAgICpcbiAgICogQGV4YW1wbGVcbiAgICpcbiAgICogICAgIGNvbnN0IHVzZXJNb2RlbDogYXBpZ2F0ZXdheS5Nb2RlbCA9IGFwaS5hZGRNb2RlbCgnVXNlck1vZGVsJywge1xuICAgKiAgICAgICAgIHNjaGVtYToge1xuICAgKiAgICAgICAgICAgICB0eXBlOiBhcGlnYXRld2F5Lkpzb25TY2hlbWFUeXBlLk9CSkVDVFxuICAgKiAgICAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAqICAgICAgICAgICAgICAgICB1c2VySWQ6IHtcbiAgICogICAgICAgICAgICAgICAgICAgICB0eXBlOiBhcGlnYXRld2F5Lkpzb25TY2hlbWEuU1RSSU5HXG4gICAqICAgICAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICAgICAgbmFtZToge1xuICAgKiAgICAgICAgICAgICAgICAgICAgIHR5cGU6IGFwaWdhdGV3YXkuSnNvblNjaGVtYS5TVFJJTkdcbiAgICogICAgICAgICAgICAgICAgIH1cbiAgICogICAgICAgICAgICAgfSxcbiAgICogICAgICAgICAgICAgcmVxdWlyZWQ6IFsndXNlcklkJ11cbiAgICogICAgICAgICB9XG4gICAqICAgICB9KTtcbiAgICogICAgIGFwaS5yb290LmFkZFJlc291cmNlKCd1c2VyJykuYWRkTWV0aG9kKCdQT1NUJyxcbiAgICogICAgICAgICBuZXcgYXBpZ2F0ZXdheS5MYW1iZGFJbnRlZ3JhdGlvbih1c2VyTGFtYmRhKSwge1xuICAgKiAgICAgICAgICAgICByZXF1ZXN0TW9kZWxzOiB7XG4gICAqICAgICAgICAgICAgICAgICAnYXBwbGljYXRpb24vanNvbic6IHVzZXJNb2RlbFxuICAgKiAgICAgICAgICAgICB9XG4gICAqICAgICAgICAgfVxuICAgKiAgICAgKTtcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYXBpLWdhdGV3YXktbWV0aG9kLXNldHRpbmdzLW1ldGhvZC1yZXF1ZXN0Lmh0bWwjc2V0dXAtbWV0aG9kLXJlcXVlc3QtbW9kZWxcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVlc3RNb2RlbHM/OiB7IFtwYXJhbTogc3RyaW5nXTogSU1vZGVsIH07XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgYXNzb2NpYXRlZCByZXF1ZXN0IHZhbGlkYXRvci5cbiAgICogT25seSBvbmUgb2YgYHJlcXVlc3RWYWxpZGF0b3JgIG9yIGByZXF1ZXN0VmFsaWRhdG9yT3B0aW9uc2AgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIFdvcmtzIHRvZ2V0aGVyIHdpdGggYHJlcXVlc3RNb2RlbHNgIG9yIGByZXF1ZXN0UGFyYW1ldGVyc2AgdG8gdmFsaWRhdGVcbiAgICogdGhlIHJlcXVlc3QgYmVmb3JlIGl0IHJlYWNoZXMgaW50ZWdyYXRpb24gbGlrZSBMYW1iZGEgUHJveHkgSW50ZWdyYXRpb24uXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVmYXVsdCB2YWxpZGF0b3JcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVlc3RWYWxpZGF0b3I/OiBJUmVxdWVzdFZhbGlkYXRvcjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIGF1dGhvcml6YXRpb24gc2NvcGVzIGNvbmZpZ3VyZWQgb24gdGhlIG1ldGhvZC4gVGhlIHNjb3BlcyBhcmUgdXNlZCB3aXRoXG4gICAqIGEgQ09HTklUT19VU0VSX1BPT0xTIGF1dGhvcml6ZXIgdG8gYXV0aG9yaXplIHRoZSBtZXRob2QgaW52b2NhdGlvbi5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQVdTQ2xvdWRGb3JtYXRpb24vbGF0ZXN0L1VzZXJHdWlkZS9hd3MtcmVzb3VyY2UtYXBpZ2F0ZXdheS1tZXRob2QuaHRtbCNjZm4tYXBpZ2F0ZXdheS1tZXRob2QtYXV0aG9yaXphdGlvbnNjb3Blc1xuICAgKiBAZGVmYXVsdCAtIG5vIGF1dGhvcml6YXRpb24gc2NvcGVzXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uU2NvcGVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFJlcXVlc3QgdmFsaWRhdG9yIG9wdGlvbnMgdG8gY3JlYXRlIG5ldyB2YWxpZGF0b3JcbiAgICogT25seSBvbmUgb2YgYHJlcXVlc3RWYWxpZGF0b3JgIG9yIGByZXF1ZXN0VmFsaWRhdG9yT3B0aW9uc2AgbXVzdCBiZSBzcGVjaWZpZWQuXG4gICAqIFdvcmtzIHRvZ2V0aGVyIHdpdGggYHJlcXVlc3RNb2RlbHNgIG9yIGByZXF1ZXN0UGFyYW1ldGVyc2AgdG8gdmFsaWRhdGVcbiAgICogdGhlIHJlcXVlc3QgYmVmb3JlIGl0IHJlYWNoZXMgaW50ZWdyYXRpb24gbGlrZSBMYW1iZGEgUHJveHkgSW50ZWdyYXRpb24uXG4gICAqIEBkZWZhdWx0IC0gTm8gZGVmYXVsdCB2YWxpZGF0b3JcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVlc3RWYWxpZGF0b3JPcHRpb25zPzogUmVxdWVzdFZhbGlkYXRvck9wdGlvbnM7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTWV0aG9kUHJvcHMge1xuICAvKipcbiAgICogVGhlIHJlc291cmNlIHRoaXMgbWV0aG9kIGlzIGFzc29jaWF0ZWQgd2l0aC4gRm9yIHJvb3QgcmVzb3VyY2UgbWV0aG9kcyxcbiAgICogc3BlY2lmeSB0aGUgYFJlc3RBcGlgIG9iamVjdC5cbiAgICovXG4gIHJlYWRvbmx5IHJlc291cmNlOiBJUmVzb3VyY2U7XG5cbiAgLyoqXG4gICAqIFRoZSBIVFRQIG1ldGhvZCAoXCJHRVRcIiwgXCJQT1NUXCIsIFwiUFVUXCIsIC4uLikgdGhhdCBjbGllbnRzIHVzZSB0byBjYWxsIHRoaXMgbWV0aG9kLlxuICAgKi9cbiAgcmVhZG9ubHkgaHR0cE1ldGhvZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgYmFja2VuZCBzeXN0ZW0gdGhhdCB0aGUgbWV0aG9kIGNhbGxzIHdoZW4gaXQgcmVjZWl2ZXMgYSByZXF1ZXN0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IGBNb2NrSW50ZWdyYXRpb25gLlxuICAgKi9cbiAgcmVhZG9ubHkgaW50ZWdyYXRpb24/OiBJbnRlZ3JhdGlvbjtcblxuICAvKipcbiAgICogTWV0aG9kIG9wdGlvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gb3B0aW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IG9wdGlvbnM/OiBNZXRob2RPcHRpb25zO1xufVxuXG5leHBvcnQgY2xhc3MgTWV0aG9kIGV4dGVuZHMgUmVzb3VyY2Uge1xuICAvKiogQGF0dHJpYnV0ZSAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbWV0aG9kSWQ6IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgaHR0cE1ldGhvZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2U6IElSZXNvdXJjZTtcbiAgLyoqXG4gICAqIFRoZSBBUEkgR2F0ZXdheSBSZXN0QXBpIGFzc29jaWF0ZWQgd2l0aCB0aGlzIG1ldGhvZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhcGk6IElSZXN0QXBpO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNZXRob2RQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnJlc291cmNlID0gcHJvcHMucmVzb3VyY2U7XG4gICAgdGhpcy5hcGkgPSBwcm9wcy5yZXNvdXJjZS5hcGk7XG4gICAgdGhpcy5odHRwTWV0aG9kID0gcHJvcHMuaHR0cE1ldGhvZC50b1VwcGVyQ2FzZSgpO1xuXG4gICAgdmFsaWRhdGVIdHRwTWV0aG9kKHRoaXMuaHR0cE1ldGhvZCk7XG5cbiAgICBjb25zdCBvcHRpb25zID0gcHJvcHMub3B0aW9ucyB8fCB7fTtcblxuICAgIGNvbnN0IGRlZmF1bHRNZXRob2RPcHRpb25zID0gcHJvcHMucmVzb3VyY2UuZGVmYXVsdE1ldGhvZE9wdGlvbnMgfHwge307XG4gICAgY29uc3QgYXV0aG9yaXplciA9IG9wdGlvbnMuYXV0aG9yaXplciB8fCBkZWZhdWx0TWV0aG9kT3B0aW9ucy5hdXRob3JpemVyO1xuICAgIGNvbnN0IGF1dGhvcml6ZXJJZCA9IGF1dGhvcml6ZXI/LmF1dGhvcml6ZXJJZDtcblxuICAgIGNvbnN0IGF1dGhvcml6YXRpb25UeXBlT3B0aW9uID0gb3B0aW9ucy5hdXRob3JpemF0aW9uVHlwZSB8fCBkZWZhdWx0TWV0aG9kT3B0aW9ucy5hdXRob3JpemF0aW9uVHlwZTtcbiAgICBjb25zdCBhdXRob3JpemF0aW9uVHlwZSA9IGF1dGhvcml6ZXI/LmF1dGhvcml6YXRpb25UeXBlIHx8IGF1dGhvcml6YXRpb25UeXBlT3B0aW9uIHx8IEF1dGhvcml6YXRpb25UeXBlLk5PTkU7XG5cbiAgICAvLyBpZiB0aGUgYXV0aG9yaXplciBkZWZpbmVzIGFuIGF1dGhvcml6YXRpb24gdHlwZSBhbmQgd2UgYWxzbyBoYXZlIGFuIGV4cGxpY2l0IG9wdGlvbiBzZXQsIGNoZWNrIHRoYXQgdGhleSBhcmUgdGhlIHNhbWVcbiAgICBpZiAoYXV0aG9yaXplcj8uYXV0aG9yaXphdGlvblR5cGUgJiYgYXV0aG9yaXphdGlvblR5cGVPcHRpb24gJiYgYXV0aG9yaXplcj8uYXV0aG9yaXphdGlvblR5cGUgIT09IGF1dGhvcml6YXRpb25UeXBlT3B0aW9uKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCR7dGhpcy5yZXNvdXJjZX0vJHt0aGlzLmh0dHBNZXRob2R9IC0gQXV0aG9yaXphdGlvbiB0eXBlIGlzIHNldCB0byAke2F1dGhvcml6YXRpb25UeXBlT3B0aW9ufSBgICtcbiAgICAgICAgYHdoaWNoIGlzIGRpZmZlcmVudCBmcm9tIHdoYXQgaXMgcmVxdWlyZWQgYnkgdGhlIGF1dGhvcml6ZXIgWyR7YXV0aG9yaXplci5hdXRob3JpemF0aW9uVHlwZX1dYCk7XG4gICAgfVxuXG4gICAgaWYgKEF1dGhvcml6ZXIuaXNBdXRob3JpemVyKGF1dGhvcml6ZXIpKSB7XG4gICAgICBhdXRob3JpemVyLl9hdHRhY2hUb0FwaSh0aGlzLmFwaSk7XG4gICAgfVxuXG4gICAgY29uc3QgbWV0aG9kUHJvcHM6IENmbk1ldGhvZFByb3BzID0ge1xuICAgICAgcmVzb3VyY2VJZDogcHJvcHMucmVzb3VyY2UucmVzb3VyY2VJZCxcbiAgICAgIHJlc3RBcGlJZDogdGhpcy5hcGkucmVzdEFwaUlkLFxuICAgICAgaHR0cE1ldGhvZDogdGhpcy5odHRwTWV0aG9kLFxuICAgICAgb3BlcmF0aW9uTmFtZTogb3B0aW9ucy5vcGVyYXRpb25OYW1lIHx8IGRlZmF1bHRNZXRob2RPcHRpb25zLm9wZXJhdGlvbk5hbWUsXG4gICAgICBhcGlLZXlSZXF1aXJlZDogb3B0aW9ucy5hcGlLZXlSZXF1aXJlZCB8fCBkZWZhdWx0TWV0aG9kT3B0aW9ucy5hcGlLZXlSZXF1aXJlZCxcbiAgICAgIGF1dGhvcml6YXRpb25UeXBlLFxuICAgICAgYXV0aG9yaXplcklkLFxuICAgICAgcmVxdWVzdFBhcmFtZXRlcnM6IG9wdGlvbnMucmVxdWVzdFBhcmFtZXRlcnMgfHwgZGVmYXVsdE1ldGhvZE9wdGlvbnMucmVxdWVzdFBhcmFtZXRlcnMsXG4gICAgICBpbnRlZ3JhdGlvbjogdGhpcy5yZW5kZXJJbnRlZ3JhdGlvbihwcm9wcy5pbnRlZ3JhdGlvbiksXG4gICAgICBtZXRob2RSZXNwb25zZXM6IHRoaXMucmVuZGVyTWV0aG9kUmVzcG9uc2VzKG9wdGlvbnMubWV0aG9kUmVzcG9uc2VzKSxcbiAgICAgIHJlcXVlc3RNb2RlbHM6IHRoaXMucmVuZGVyUmVxdWVzdE1vZGVscyhvcHRpb25zLnJlcXVlc3RNb2RlbHMpLFxuICAgICAgcmVxdWVzdFZhbGlkYXRvcklkOiB0aGlzLnJlcXVlc3RWYWxpZGF0b3JJZChvcHRpb25zKSxcbiAgICAgIGF1dGhvcml6YXRpb25TY29wZXM6IG9wdGlvbnMuYXV0aG9yaXphdGlvblNjb3BlcyA/PyBkZWZhdWx0TWV0aG9kT3B0aW9ucy5hdXRob3JpemF0aW9uU2NvcGVzLFxuICAgIH07XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5NZXRob2QodGhpcywgJ1Jlc291cmNlJywgbWV0aG9kUHJvcHMpO1xuXG4gICAgdGhpcy5tZXRob2RJZCA9IHJlc291cmNlLnJlZjtcblxuICAgIGlmIChSZXN0QXBpQmFzZS5faXNSZXN0QXBpQmFzZShwcm9wcy5yZXNvdXJjZS5hcGkpKSB7XG4gICAgICBwcm9wcy5yZXNvdXJjZS5hcGkuX2F0dGFjaE1ldGhvZCh0aGlzKTtcbiAgICB9XG5cbiAgICBjb25zdCBkZXBsb3ltZW50ID0gcHJvcHMucmVzb3VyY2UuYXBpLmxhdGVzdERlcGxveW1lbnQ7XG4gICAgaWYgKGRlcGxveW1lbnQpIHtcbiAgICAgIGRlcGxveW1lbnQubm9kZS5hZGREZXBlbmRlbmN5KHJlc291cmNlKTtcbiAgICAgIGRlcGxveW1lbnQuYWRkVG9Mb2dpY2FsSWQoeyBtZXRob2Q6IG1ldGhvZFByb3BzIH0pO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgUmVzdEFwaSBhc3NvY2lhdGVkIHdpdGggdGhpcyBNZXRob2RcbiAgICogQGRlcHJlY2F0ZWQgLSBUaHJvd3MgYW4gZXJyb3IgaWYgdGhpcyBSZXNvdXJjZSBpcyBub3QgYXNzb2NpYXRlZCB3aXRoIGFuIGluc3RhbmNlIG9mIGBSZXN0QXBpYC4gVXNlIGBhcGlgIGluc3RlYWQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHJlc3RBcGkoKTogUmVzdEFwaSB7XG4gICAgcmV0dXJuIHRoaXMucmVzb3VyY2UucmVzdEFwaTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGV4ZWN1dGUtYXBpIEFSTiBmb3IgdGhpcyBtZXRob2Q6XG4gICAqXG4gICAqICAgYXJuOmF3czpleGVjdXRlLWFwaTp7cmVnaW9ufTp7YWNjb3VudH06e3Jlc3RBcGlJZH0ve3N0YWdlfS97bWV0aG9kfS97cGF0aH1cbiAgICpcbiAgICogTk9URToge3N0YWdlfSB3aWxsIHJlZmVyIHRvIHRoZSBgcmVzdEFwaS5kZXBsb3ltZW50U3RhZ2VgLCB3aGljaCB3aWxsXG4gICAqIGF1dG9tYXRpY2FsbHkgc2V0IGlmIGF1dG8tZGVwbG95IGlzIGVuYWJsZWQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyBnZXQgbWV0aG9kQXJuKCk6IHN0cmluZyB7XG4gICAgaWYgKCF0aGlzLnJlc3RBcGkuZGVwbG95bWVudFN0YWdlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgIGBVbmFibGUgdG8gZGV0ZXJtaW5lIEFSTiBmb3IgbWV0aG9kIFwiJHt0aGlzLm5vZGUuaWR9XCIgc2luY2UgdGhlcmUgaXMgbm8gc3RhZ2UgYXNzb2NpYXRlZCB3aXRoIHRoaXMgQVBJLlxcbmAgK1xuICAgICAgICAnRWl0aGVyIHVzZSB0aGUgYGRlcGxveWAgcHJvcCBvciBleHBsaWNpdGx5IGFzc2lnbiBgZGVwbG95bWVudFN0YWdlYCBvbiB0aGUgUmVzdEFwaScpO1xuICAgIH1cblxuICAgIGNvbnN0IHN0YWdlID0gdGhpcy5yZXN0QXBpLmRlcGxveW1lbnRTdGFnZS5zdGFnZU5hbWUudG9TdHJpbmcoKTtcbiAgICByZXR1cm4gdGhpcy5yZXN0QXBpLmFybkZvckV4ZWN1dGVBcGkodGhpcy5odHRwTWV0aG9kLCBwYXRoRm9yQXJuKHRoaXMucmVzb3VyY2UucGF0aCksIHN0YWdlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIGFuIGV4ZWN1dGUtYXBpIEFSTiBmb3IgdGhpcyBtZXRob2QncyBcInRlc3QtaW52b2tlLXN0YWdlXCIgc3RhZ2UuXG4gICAqIFRoaXMgc3RhZ2UgaXMgdXNlZCBieSB0aGUgQVdTIENvbnNvbGUgVUkgd2hlbiB0ZXN0aW5nIHRoZSBtZXRob2QuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHRlc3RNZXRob2RBcm4oKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5yZXN0QXBpLmFybkZvckV4ZWN1dGVBcGkodGhpcy5odHRwTWV0aG9kLCBwYXRoRm9yQXJuKHRoaXMucmVzb3VyY2UucGF0aCksICd0ZXN0LWludm9rZS1zdGFnZScpO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJJbnRlZ3JhdGlvbihpbnRlZ3JhdGlvbj86IEludGVncmF0aW9uKTogQ2ZuTWV0aG9kLkludGVncmF0aW9uUHJvcGVydHkge1xuICAgIGlmICghaW50ZWdyYXRpb24pIHtcbiAgICAgIC8vIHVzZSBkZWZhdWx0SW50ZWdyYXRpb24gZnJvbSBBUEkgaWYgZGVmaW5lZFxuICAgICAgaWYgKHRoaXMucmVzb3VyY2UuZGVmYXVsdEludGVncmF0aW9uKSB7XG4gICAgICAgIHJldHVybiB0aGlzLnJlbmRlckludGVncmF0aW9uKHRoaXMucmVzb3VyY2UuZGVmYXVsdEludGVncmF0aW9uKTtcbiAgICAgIH1cblxuICAgICAgLy8gZmFsbGJhY2sgdG8gbW9ja1xuICAgICAgcmV0dXJuIHRoaXMucmVuZGVySW50ZWdyYXRpb24obmV3IE1vY2tJbnRlZ3JhdGlvbigpKTtcbiAgICB9XG5cbiAgICBpbnRlZ3JhdGlvbi5iaW5kKHRoaXMpO1xuXG4gICAgY29uc3Qgb3B0aW9ucyA9IGludGVncmF0aW9uLl9wcm9wcy5vcHRpb25zIHx8IHsgfTtcblxuICAgIGxldCBjcmVkZW50aWFscztcbiAgICBpZiAob3B0aW9ucy5jcmVkZW50aWFsc1Bhc3N0aHJvdWdoICE9PSB1bmRlZmluZWQgJiYgb3B0aW9ucy5jcmVkZW50aWFsc1JvbGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdcXCdjcmVkZW50aWFsc1Bhc3N0aHJvdWdoXFwnIGFuZCBcXCdjcmVkZW50aWFsc1JvbGVcXCcgYXJlIG11dHVhbGx5IGV4Y2x1c2l2ZScpO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLmNvbm5lY3Rpb25UeXBlID09PSBDb25uZWN0aW9uVHlwZS5WUENfTElOSyAmJiBvcHRpb25zLnZwY0xpbmsgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdcXCdjb25uZWN0aW9uVHlwZVxcJyBvZiBWUENfTElOSyByZXF1aXJlcyBcXCd2cGNMaW5rXFwnIHByb3AgdG8gYmUgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMuY29ubmVjdGlvblR5cGUgPT09IENvbm5lY3Rpb25UeXBlLklOVEVSTkVUICYmIG9wdGlvbnMudnBjTGluayAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2Nhbm5vdCBzZXQgXFwndnBjTGlua1xcJyB3aGVyZSBcXCdjb25uZWN0aW9uVHlwZVxcJyBpcyBJTlRFUk5FVCcpO1xuICAgIH1cblxuICAgIGlmIChvcHRpb25zLmNyZWRlbnRpYWxzUm9sZSkge1xuICAgICAgY3JlZGVudGlhbHMgPSBvcHRpb25zLmNyZWRlbnRpYWxzUm9sZS5yb2xlQXJuO1xuICAgIH0gZWxzZSBpZiAob3B0aW9ucy5jcmVkZW50aWFsc1Bhc3N0aHJvdWdoKSB7XG4gICAgICAvLyBhcm46YXdzOmlhbTo6Kjp1c2VyLypcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgIGNyZWRlbnRpYWxzID0gU3RhY2sub2YodGhpcykuZm9ybWF0QXJuKHsgc2VydmljZTogJ2lhbScsIHJlZ2lvbjogJycsIGFjY291bnQ6ICcqJywgcmVzb3VyY2U6ICd1c2VyJywgc2VwOiAnLycsIHJlc291cmNlTmFtZTogJyonIH0pO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICB0eXBlOiBpbnRlZ3JhdGlvbi5fcHJvcHMudHlwZSxcbiAgICAgIHVyaTogaW50ZWdyYXRpb24uX3Byb3BzLnVyaSxcbiAgICAgIGNhY2hlS2V5UGFyYW1ldGVyczogb3B0aW9ucy5jYWNoZUtleVBhcmFtZXRlcnMsXG4gICAgICBjYWNoZU5hbWVzcGFjZTogb3B0aW9ucy5jYWNoZU5hbWVzcGFjZSxcbiAgICAgIGNvbnRlbnRIYW5kbGluZzogb3B0aW9ucy5jb250ZW50SGFuZGxpbmcsXG4gICAgICBpbnRlZ3JhdGlvbkh0dHBNZXRob2Q6IGludGVncmF0aW9uLl9wcm9wcy5pbnRlZ3JhdGlvbkh0dHBNZXRob2QsXG4gICAgICByZXF1ZXN0UGFyYW1ldGVyczogb3B0aW9ucy5yZXF1ZXN0UGFyYW1ldGVycyxcbiAgICAgIHJlcXVlc3RUZW1wbGF0ZXM6IG9wdGlvbnMucmVxdWVzdFRlbXBsYXRlcyxcbiAgICAgIHBhc3N0aHJvdWdoQmVoYXZpb3I6IG9wdGlvbnMucGFzc3Rocm91Z2hCZWhhdmlvcixcbiAgICAgIGludGVncmF0aW9uUmVzcG9uc2VzOiBvcHRpb25zLmludGVncmF0aW9uUmVzcG9uc2VzLFxuICAgICAgY29ubmVjdGlvblR5cGU6IG9wdGlvbnMuY29ubmVjdGlvblR5cGUsXG4gICAgICBjb25uZWN0aW9uSWQ6IG9wdGlvbnMudnBjTGluayA/IG9wdGlvbnMudnBjTGluay52cGNMaW5rSWQgOiB1bmRlZmluZWQsXG4gICAgICBjcmVkZW50aWFscyxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJNZXRob2RSZXNwb25zZXMobWV0aG9kUmVzcG9uc2VzOiBNZXRob2RSZXNwb25zZVtdIHwgdW5kZWZpbmVkKTogQ2ZuTWV0aG9kLk1ldGhvZFJlc3BvbnNlUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFtZXRob2RSZXNwb25zZXMpIHtcbiAgICAgIC8vIEZhbGwgYmFjayB0byBub3RoaW5nXG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIHJldHVybiBtZXRob2RSZXNwb25zZXMubWFwKG1yID0+IHtcbiAgICAgIGxldCByZXNwb25zZU1vZGVsczoge1tjb250ZW50VHlwZTogc3RyaW5nXTogc3RyaW5nfSB8IHVuZGVmaW5lZDtcblxuICAgICAgaWYgKG1yLnJlc3BvbnNlTW9kZWxzKSB7XG4gICAgICAgIHJlc3BvbnNlTW9kZWxzID0ge307XG4gICAgICAgIGZvciAoY29uc3QgY29udGVudFR5cGUgaW4gbXIucmVzcG9uc2VNb2RlbHMpIHtcbiAgICAgICAgICBpZiAobXIucmVzcG9uc2VNb2RlbHMuaGFzT3duUHJvcGVydHkoY29udGVudFR5cGUpKSB7XG4gICAgICAgICAgICByZXNwb25zZU1vZGVsc1tjb250ZW50VHlwZV0gPSBtci5yZXNwb25zZU1vZGVsc1tjb250ZW50VHlwZV0ubW9kZWxJZDtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgY29uc3QgbWV0aG9kUmVzcG9uc2VQcm9wID0ge1xuICAgICAgICBzdGF0dXNDb2RlOiBtci5zdGF0dXNDb2RlLFxuICAgICAgICByZXNwb25zZVBhcmFtZXRlcnM6IG1yLnJlc3BvbnNlUGFyYW1ldGVycyxcbiAgICAgICAgcmVzcG9uc2VNb2RlbHMsXG4gICAgICB9O1xuXG4gICAgICByZXR1cm4gbWV0aG9kUmVzcG9uc2VQcm9wO1xuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJSZXF1ZXN0TW9kZWxzKHJlcXVlc3RNb2RlbHM6IHsgW3BhcmFtOiBzdHJpbmddOiBJTW9kZWwgfSB8IHVuZGVmaW5lZCk6IHsgW3BhcmFtOiBzdHJpbmddOiBzdHJpbmcgfSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKCFyZXF1ZXN0TW9kZWxzKSB7XG4gICAgICAvLyBGYWxsIGJhY2sgdG8gbm90aGluZ1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCBtb2RlbHM6IHtbcGFyYW06IHN0cmluZ106IHN0cmluZ30gPSB7fTtcbiAgICBmb3IgKGNvbnN0IGNvbnRlbnRUeXBlIGluIHJlcXVlc3RNb2RlbHMpIHtcbiAgICAgIGlmIChyZXF1ZXN0TW9kZWxzLmhhc093blByb3BlcnR5KGNvbnRlbnRUeXBlKSkge1xuICAgICAgICBtb2RlbHNbY29udGVudFR5cGVdID0gcmVxdWVzdE1vZGVsc1tjb250ZW50VHlwZV0ubW9kZWxJZDtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gbW9kZWxzO1xuICB9XG5cbiAgcHJpdmF0ZSByZXF1ZXN0VmFsaWRhdG9ySWQob3B0aW9uczogTWV0aG9kT3B0aW9ucyk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKG9wdGlvbnMucmVxdWVzdFZhbGlkYXRvciAmJiBvcHRpb25zLnJlcXVlc3RWYWxpZGF0b3JPcHRpb25zKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgb25lIG9mIFxcJ3JlcXVlc3RWYWxpZGF0b3JcXCcgb3IgXFwncmVxdWVzdFZhbGlkYXRvck9wdGlvbnNcXCcgbXVzdCBiZSBzcGVjaWZpZWQuJyk7XG4gICAgfVxuXG4gICAgaWYgKG9wdGlvbnMucmVxdWVzdFZhbGlkYXRvck9wdGlvbnMpIHtcbiAgICAgIGNvbnN0IHZhbGlkYXRvciA9IHRoaXMucmVzdEFwaS5hZGRSZXF1ZXN0VmFsaWRhdG9yKCd2YWxpZGF0b3InLCBvcHRpb25zLnJlcXVlc3RWYWxpZGF0b3JPcHRpb25zKTtcbiAgICAgIHJldHVybiB2YWxpZGF0b3IucmVxdWVzdFZhbGlkYXRvcklkO1xuICAgIH1cblxuICAgIC8vIEZvciBiYWNrd2FyZCBjb21wYXRpYmlsaXR5XG4gICAgcmV0dXJuIG9wdGlvbnMucmVxdWVzdFZhbGlkYXRvcj8ucmVxdWVzdFZhbGlkYXRvcklkO1xuICB9XG59XG5cbmV4cG9ydCBlbnVtIEF1dGhvcml6YXRpb25UeXBlIHtcbiAgLyoqXG4gICAqIE9wZW4gYWNjZXNzLlxuICAgKi9cbiAgTk9ORSA9ICdOT05FJyxcblxuICAvKipcbiAgICogVXNlIEFXUyBJQU0gcGVybWlzc2lvbnMuXG4gICAqL1xuICBJQU0gPSAnQVdTX0lBTScsXG5cbiAgLyoqXG4gICAqIFVzZSBhIGN1c3RvbSBhdXRob3JpemVyLlxuICAgKi9cbiAgQ1VTVE9NID0gJ0NVU1RPTScsXG5cbiAgLyoqXG4gICAqIFVzZSBhbiBBV1MgQ29nbml0byB1c2VyIHBvb2wuXG4gICAqL1xuICBDT0dOSVRPID0gJ0NPR05JVE9fVVNFUl9QT09MUycsXG59XG5cbmZ1bmN0aW9uIHBhdGhGb3JBcm4ocGF0aDogc3RyaW5nKTogc3RyaW5nIHtcbiAgcmV0dXJuIHBhdGgucmVwbGFjZSgvXFx7W15cXH1dKlxcfS9nLCAnKicpOyAvLyByZXBsYWNlIHBhdGggcGFyYW1ldGVycyAobGlrZSAne2Jvb2tJZH0nKSB3aXRoIGFzdGVyaXNrXG59XG4iXX0=