"use strict";
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const path = require("path");
const apigw = require("../lib");
function defineRestApi(definition) {
    const stack = new cdk.Stack();
    return new apigw.SpecRestApi(stack, 'API', {
        apiDefinition: definition,
    });
}
module.exports = {
    'apigateway.ApiDefinition.fromJson': {
        'happy case'(test) {
            const stack = new cdk.Stack();
            const definition = {
                key1: 'val1',
            };
            const config = apigw.ApiDefinition.fromInline(definition).bind(stack);
            test.deepEqual(config.inlineDefinition, definition);
            test.ok(config.s3Location === undefined);
            test.done();
        },
        'fails if Json definition is empty'(test) {
            test.throws(() => defineRestApi(apigw.ApiDefinition.fromInline({})), /cannot be empty/);
            test.done();
        },
        'fails if definition is not an object'(test) {
            test.throws(() => defineRestApi(apigw.ApiDefinition.fromInline('not-json')), /should be of type object/);
            test.done();
        },
    },
    'apigateway.ApiDefinition.fromAsset': {
        'happy case'(test) {
            const stack = new cdk.Stack();
            const config = apigw.ApiDefinition.fromAsset(path.join(__dirname, 'sample-definition.yaml')).bind(stack);
            test.ok(config.inlineDefinition === undefined);
            test.ok(config.s3Location !== undefined);
            test.deepEqual(stack.resolve(config.s3Location.bucket), {
                Ref: 'AssetParameters68497ac876de4e963fc8f7b5f1b28844c18ecc95e3f7c6e9e0bf250e03c037fbS3Bucket42039E29',
            });
            test.done();
        },
        'fails if a directory is given for an asset'(test) {
            // GIVEN
            const fileAsset = apigw.ApiDefinition.fromAsset(path.join(__dirname, 'authorizers'));
            // THEN
            test.throws(() => defineRestApi(fileAsset), /Asset cannot be a \.zip file or a directory/);
            test.done();
        },
        'only one Asset object gets created even if multiple functions use the same AssetApiDefinition'(test) {
            // GIVEN
            const app = new cdk.App();
            const stack = new cdk.Stack(app, 'MyStack');
            const directoryAsset = apigw.ApiDefinition.fromAsset(path.join(__dirname, 'sample-definition.yaml'));
            // WHEN
            new apigw.SpecRestApi(stack, 'API1', {
                apiDefinition: directoryAsset,
            });
            new apigw.SpecRestApi(stack, 'API2', {
                apiDefinition: directoryAsset,
            });
            // THEN
            const assembly = app.synth();
            const synthesized = assembly.stacks[0];
            // API1 has an asset, API2 does not
            test.deepEqual(synthesized.assets.length, 1);
            test.done();
        },
    },
    'apigateway.ApiDefinition.fromBucket': {
        'happy case'(test) {
            const stack = new cdk.Stack();
            const bucket = new s3.Bucket(stack, 'my-bucket');
            const config = apigw.ApiDefinition.fromBucket(bucket, 'my-key', 'my-version').bind(stack);
            test.ok(config.inlineDefinition === undefined);
            test.ok(config.s3Location !== undefined);
            test.deepEqual(stack.resolve(config.s3Location.bucket), {
                Ref: 'mybucket15D133BF',
            });
            test.equals(config.s3Location.key, 'my-key');
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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