"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointType = exports.ApiKeySourceType = exports.RestApi = exports.SpecRestApi = exports.RestApiBase = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const api_key_1 = require("./api-key");
const apigateway_generated_1 = require("./apigateway.generated");
const deployment_1 = require("./deployment");
const domain_name_1 = require("./domain-name");
const gateway_response_1 = require("./gateway-response");
const model_1 = require("./model");
const requestvalidator_1 = require("./requestvalidator");
const resource_1 = require("./resource");
const stage_1 = require("./stage");
const usage_plan_1 = require("./usage-plan");
const RESTAPI_SYMBOL = Symbol.for('@aws-cdk/aws-apigateway.RestApiBase');
/**
 * Base implementation that are common to various implementations of IRestApi.
 *
 * @stability stable
 */
class RestApiBase extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        this.restApiName = (_a = props.restApiName) !== null && _a !== void 0 ? _a : id;
        Object.defineProperty(this, RESTAPI_SYMBOL, { value: true });
    }
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x) {
        return x !== null && typeof (x) === 'object' && RESTAPI_SYMBOL in x;
    }
    /**
     * API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     *
     * @stability stable
     */
    get latestDeployment() {
        return this._latestDeployment;
    }
    /**
     * The first domain name mapped to this API, if defined through the `domainName` configuration prop, or added via `addDomainName`.
     *
     * @stability stable
     */
    get domainName() {
        return this._domainName;
    }
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     *
     * @stability stable
     */
    urlForPath(path = '/') {
        if (!this.deploymentStage) {
            throw new Error('Cannot determine deployment stage for API from "deploymentStage". Use "deploy" or explicitly set "deploymentStage"');
        }
        return this.deploymentStage.urlForPath(path);
    }
    /**
     * Defines an API Gateway domain name and maps it to this API.
     *
     * @param id The construct id.
     * @param options custom domain options.
     * @stability stable
     */
    addDomainName(id, options) {
        const domainName = new domain_name_1.DomainName(this, id, {
            ...options,
            mapping: this,
        });
        if (!this._domainName) {
            this._domainName = domainName;
        }
        return domainName;
    }
    /**
     * Adds a usage plan.
     *
     * @stability stable
     */
    addUsagePlan(id, props = {}) {
        return new usage_plan_1.UsagePlan(this, id, props);
    }
    /**
     * Gets the "execute-api" ARN.
     *
     * @stability stable
     */
    arnForExecuteApi(method = '*', path = '/*', stage = '*') {
        if (!path.startsWith('/')) {
            throw new Error(`"path" must begin with a "/": '${path}'`);
        }
        if (method.toUpperCase() === 'ANY') {
            method = '*';
        }
        return core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: this.restApiId,
            sep: '/',
            resourceName: `${stage}/${method}${path}`,
        });
    }
    /**
     * Adds a new gateway response.
     *
     * @stability stable
     */
    addGatewayResponse(id, options) {
        return new gateway_response_1.GatewayResponse(this, id, {
            restApi: this,
            ...options,
        });
    }
    /**
     * Add an ApiKey.
     *
     * @stability stable
     */
    addApiKey(id, options) {
        return new api_key_1.ApiKey(this, id, {
            resources: [this],
            ...options,
        });
    }
    /**
     * Returns the given named metric for this API.
     *
     * @stability stable
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApiGateway',
            metricName,
            dimensions: { ApiName: this.restApiName },
            ...props,
        });
    }
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricClientError(props) {
        return this.metric('4XXError', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricServerError(props) {
        return this.metric('5XXError', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricCacheHitCount(props) {
        return this.metric('CacheHitCount', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the number of requests served from the backend in a given period, when API caching is enabled.
     *
     * @default - sum over 5 minutes
     * @stability stable
     */
    metricCacheMissCount(props) {
        return this.metric('CacheMissCount', { statistic: 'Sum', ...props });
    }
    /**
     * Metric for the total number API requests in a given period.
     *
     * @default - SampleCount over 5 minutes
     * @stability stable
     */
    metricCount(props) {
        return this.metric('Count', { statistic: 'SampleCount', ...props });
    }
    /**
     * Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * @default - no statistic
     * @stability stable
     */
    metricIntegrationLatency(props) {
        return this.metric('IntegrationLatency', props);
    }
    /**
     * The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * @default - no statistic
     * @stability stable
     */
    metricLatency(props) {
        return this.metric('Latency', props);
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        ignore(method);
    }
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        ignore(deployment);
    }
    /**
     * @stability stable
     */
    configureCloudWatchRole(apiResource) {
        const role = new iam.Role(this, 'CloudWatchRole', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs')],
        });
        const resource = new apigateway_generated_1.CfnAccount(this, 'Account', {
            cloudWatchRoleArn: role.roleArn,
        });
        resource.node.addDependency(apiResource);
    }
    /**
     * @stability stable
     */
    configureDeployment(props) {
        const deploy = props.deploy === undefined ? true : props.deploy;
        if (deploy) {
            this._latestDeployment = new deployment_1.Deployment(this, 'Deployment', {
                description: 'Automatically created by the RestApi construct',
                api: this,
                retainDeployments: props.retainDeployments,
            });
            // encode the stage name into the construct id, so if we change the stage name, it will recreate a new stage.
            // stage name is part of the endpoint, so that makes sense.
            const stageName = (props.deployOptions && props.deployOptions.stageName) || 'prod';
            this.deploymentStage = new stage_1.Stage(this, `DeploymentStage.${stageName}`, {
                deployment: this._latestDeployment,
                ...props.deployOptions,
            });
            new core_1.CfnOutput(this, 'Endpoint', { exportName: props.endpointExportName, value: this.urlForPath() });
        }
        else {
            if (props.deployOptions) {
                throw new Error('Cannot set \'deployOptions\' if \'deploy\' is disabled');
            }
        }
    }
    /**
     * @internal
     */
    _configureEndpoints(props) {
        var _a, _b;
        if (props.endpointTypes && props.endpointConfiguration) {
            throw new Error('Only one of the RestApi props, endpointTypes or endpointConfiguration, is allowed');
        }
        if (props.endpointConfiguration) {
            return {
                types: props.endpointConfiguration.types,
                vpcEndpointIds: (_b = (_a = props.endpointConfiguration) === null || _a === void 0 ? void 0 : _a.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.map(vpcEndpoint => vpcEndpoint.vpcEndpointId),
            };
        }
        if (props.endpointTypes) {
            return { types: props.endpointTypes };
        }
        return undefined;
    }
}
exports.RestApiBase = RestApiBase;
/**
 * (experimental) Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @experimental
 * @resource AWS::ApiGateway::RestApi
 */
class SpecRestApi extends RestApiBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
        const apiDefConfig = props.apiDefinition.bind(this);
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            body: apiDefConfig.inlineDefinition ? apiDefConfig.inlineDefinition : undefined,
            bodyS3Location: apiDefConfig.inlineDefinition ? undefined : apiDefConfig.s3Location,
            endpointConfiguration: this._configureEndpoints(props),
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        this.restApiRootResourceId = resource.attrRootResourceId;
        this.root = new RootResource(this, {}, this.restApiRootResourceId);
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
    }
}
exports.SpecRestApi = SpecRestApi;
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @stability stable
 */
class RestApi extends RestApiBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        /**
         * The list of methods bound to this RestApi.
         *
         * @stability stable
         */
        this.methods = new Array();
        /**
         * This list of deployments bound to this RestApi
         */
        this.deployments = new Array();
        const resource = new apigateway_generated_1.CfnRestApi(this, 'Resource', {
            name: this.restApiName,
            description: props.description,
            policy: props.policy,
            failOnWarnings: props.failOnWarnings,
            minimumCompressionSize: props.minimumCompressionSize,
            binaryMediaTypes: props.binaryMediaTypes,
            endpointConfiguration: this._configureEndpoints(props),
            apiKeySourceType: props.apiKeySourceType,
            cloneFrom: props.cloneFrom ? props.cloneFrom.restApiId : undefined,
            parameters: props.parameters,
        });
        this.node.defaultChild = resource;
        this.restApiId = resource.ref;
        const cloudWatchRole = props.cloudWatchRole !== undefined ? props.cloudWatchRole : true;
        if (cloudWatchRole) {
            this.configureCloudWatchRole(resource);
        }
        this.configureDeployment(props);
        if (props.domainName) {
            this.addDomainName('CustomDomain', props.domainName);
        }
        this.root = new RootResource(this, props, resource.attrRootResourceId);
        this.restApiRootResourceId = resource.attrRootResourceId;
    }
    /**
     * Import an existing RestApi.
     *
     * @stability stable
     */
    static fromRestApiId(scope, id, restApiId) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = restApiId;
            }
            get root() {
                throw new Error('root is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
            get restApiRootResourceId() {
                throw new Error('restApiRootResourceId is not configured when imported using `fromRestApiId()`. Use `fromRestApiAttributes()` API instead.');
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing RestApi that can be configured with additional Methods and Resources.
     *
     * @experimental
     */
    static fromRestApiAttributes(scope, id, attrs) {
        class Import extends RestApiBase {
            constructor() {
                super(...arguments);
                this.restApiId = attrs.restApiId;
                this.restApiRootResourceId = attrs.rootResourceId;
                this.root = new RootResource(this, {}, this.restApiRootResourceId);
            }
        }
        return new Import(scope, id);
    }
    /**
     * The deployed root URL of this REST API.
     *
     * @stability stable
     */
    get url() {
        return this.urlForPath();
    }
    /**
     * Adds a new model.
     *
     * @stability stable
     */
    addModel(id, props) {
        return new model_1.Model(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Adds a new request validator.
     *
     * @stability stable
     */
    addRequestValidator(id, props) {
        return new requestvalidator_1.RequestValidator(this, id, {
            ...props,
            restApi: this,
        });
    }
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method) {
        this.methods.push(method);
        // add this method as a dependency to all deployments defined for this api
        // when additional deployments are added, _attachDeployment is called and
        // this method will be added there.
        for (const dep of this.deployments) {
            dep._addMethodDependency(method);
        }
    }
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment) {
        this.deployments.push(deployment);
        // add all methods that were already defined as dependencies of this
        // deployment when additional methods are added, _attachMethod is called and
        // it will be added as a dependency to this deployment.
        for (const method of this.methods) {
            deployment._addMethodDependency(method);
        }
    }
    /**
     * Performs validation of the REST API.
     *
     * @stability stable
     */
    validate() {
        if (this.methods.length === 0) {
            return ["The REST API doesn't contain any methods"];
        }
        return [];
    }
}
exports.RestApi = RestApi;
/**
 * @stability stable
 */
var ApiKeySourceType;
(function (ApiKeySourceType) {
    ApiKeySourceType["HEADER"] = "HEADER";
    ApiKeySourceType["AUTHORIZER"] = "AUTHORIZER";
})(ApiKeySourceType = exports.ApiKeySourceType || (exports.ApiKeySourceType = {}));
/**
 * @stability stable
 */
var EndpointType;
(function (EndpointType) {
    EndpointType["EDGE"] = "EDGE";
    EndpointType["REGIONAL"] = "REGIONAL";
    EndpointType["PRIVATE"] = "PRIVATE";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
class RootResource extends resource_1.ResourceBase {
    constructor(api, props, resourceId) {
        super(api, 'Default');
        this.parentResource = undefined;
        this.defaultIntegration = props.defaultIntegration;
        this.defaultMethodOptions = props.defaultMethodOptions;
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions;
        this.api = api;
        this.resourceId = resourceId;
        this.path = '/';
        if (api instanceof RestApi) {
            this._restApi = api;
        }
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * Get the RestApi associated with this Resource.
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this._restApi) {
            throw new Error('RestApi is not available on Resource not connected to an instance of RestApi. Use `api` instead');
        }
        return this._restApi;
    }
}
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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