# Copyright © 2018 Broadcom. All rights reserved.
# The term "Broadcom" refers to Broadcom Inc. and/or its subsidiaries.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may also obtain a copy of the License at
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""

:mod:`pyfos_auth` - PyFOS module to facilitate \
        establishing a session to a FOS switch.
***********************************************************************************************
The :mod:`pyfos_auth` module provides functions to establish,
modify, and terminate a session to a FOS switch for management.
Information from the session is referenced by each subsequent call.
Providing invalid session information will result in error.

.. note::
   The session idle timeout is set to 2 hours on a FOS switch.

.. note::
   Only a limited number of sessions are available. Thus, proper
   termination of a session is recommended after usage.

.. note::
   A session defaults to the Virtual Fabric ID (VFID) of -1 at the time
   of login. It must be changed using :func:`vfid_set` to
   direct the subsequent requests to a given VFID.

.. note::

    * If using HTTP, None should passed to pyfos_auth.login() \
            for the is_https parameter. Typically, there is no need \
            to pass the "--secured" option for utility scripts \
            provided by Brocade.
    * If using HTTPS with a self-signed certificate generated \
            in FOS using <b>seccertmgmt generate -cert https</b>, \
            "self" is passed to pyfos_auth.login(). For utility \
            scripts provided by Brocade, "--secured=self" needs \
            to be provided.
    * If using HTTPS with a certificate generated by a CA or generated \
            locally within your own org, "CA" is passed to \
            pyfos_auth.login(). For utility scripts provided by \
            Brocade, "--secured=CA" must be provided. Also, a \
            root certificate of CA or of the local issuer must \
            be added to your platform. For example, on Red Hat, the \
            steps to add the root certificate are:

               1. sudo cp <root cert file name, such as xxxx.pem> \
                        /usr/local/share/ca-certificates/xxxx.crt

               2. sudo update-ca-certificates

            Please reference pyfos/docs/CA_Setup_Reference_Examples.doc \
                    for additional details.

            Please reference pyfos/docs/CA_Setup_Reference_Examples.doc \
                    for additional details

Example usage of the module::

    session = pyfos_auth.login("myname", "mypassword", "10.10.10.10", 1)

    if pyfos_auth.is_failed_login(session):
        print("login failed because",
              session.get(pyfos_auth.CREDENTIAL_KEY)[pyfos.auth.LOGIN_ERROR_KEY])
        sys.exit()

    pyfos_auth.logout(session)

.. _error-structure-reference:

PyFOS functions generally return a generic error structure upon failure.

Generic error structure:

    +----------------+-------------------------+------------------------------+
    |Dictionary Key  |Description              |Example                       |
    +================+=========================+==============================+
    |error-app-tag   |Application error tag    |Error, Info, or Warning       |
    +----------------+-------------------------+------------------------------+
    |error-info	     |Array of "error-code"    |-20			      |
    +----------------+-------------------------+------------------------------+
    |error-message   |Error message in string  |Invalid speed ...	      |
    +----------------+-------------------------+------------------------------+
    |error-path	     |URI path		       |/fibrechannel/name/0/0/speed  |
    +----------------+-------------------------+------------------------------+
    |error-tag	     |Mapped to HTTP error     |Operation-failed	      |
    +----------------+-------------------------+------------------------------+
    |error-type	     |Error type	       |Application or protocol       |
    +----------------+-------------------------+------------------------------+

"""

import errno
import time
from socket import error as socket_error
from pyfos import pyfos_login
from pyfos import pyfos_util
import pyfos.pyfos_brocade_fibrechannel_switch as pyfos_switch
from pyfos.pyfos_version import fosversion
from pyfos.pyfos_auth_token import auth_token, auth_token_manager

LOGIN_ERROR_KEY = 'login-error'
CREDENTIAL_KEY = 'credential'
DEFAULT_THROTTLE_DELAY = 4


def login(username, password, ip_addr, is_https,
          throttle_delay=DEFAULT_THROTTLE_DELAY, verbose=0,
          tokenArg=None, sessionless=False, nocred=False):
    """Establish a session to the FOS switch based on a valid login credential.

    :param username: User name to establish a session.
    :param password: Password to establish a session.
    :param ip_addr: IP address of the FOS switch with which to \
            establish a session.
    :param is_https: None to indicate HTTP, "self" to indicate \
            HTTPS with a self-signed certificate, and "CA" to indicate \
            HTTPS with a CA-signed certificate.
    :param throttle_delay: Delay in seconds, to account for the maximum \
            number of requests allowed on FOS RESTCONF. The maximum number \
            of request per period is configurable on FOS.
    :param verbose: 1 for verbose session.
    :param tokenArg: The "AuthToken" input argument, The values supported are \
            a string "AuthToken" or an "AuthTokenManager" instance. The utils \
            will use TokenManager instance internally for its use. The custom \
            tokenManager implementation may pass the string "AuthToken".
    :param sessionless: The session-less authentication for REST request. \
            The value passed is False by default. If the value is True then \
            login session is not used, however existing infra can continue to \
            use the login and logout infra as no-operation in this scenario .
    :param nocred: There is no credential needed for the REST request. \
            The support is limited to some known requests only and is not\
            supported for all REST requests.\
    :rtype: Dictionary of the error description.
        :ref:`Generic error structure <error-structure-reference>` \
                or session description.
    """
    if nocred is True:
        return {CREDENTIAL_KEY: None,
                "user": None,
                "ip_addr": ip_addr,
                "vfid": -1, "ishttps": is_https, "debug": 0,
                "throttle_delay": throttle_delay,
                "auth_token": tokenArg,
                "sessionless": sessionless,
                "nocredential": nocred,
                "version": None}
    credential = None
    authtoken = None
    if tokenArg is not None:
        if password is not None:
            # print("Error: password not needed for authtoken based session.")
            return {CREDENTIAL_KEY:
                    {LOGIN_ERROR_KEY: ip_addr + " Password not needed for "
                     "authtoken based REST session."},
                    "user": username,
                    "ip_addr": ip_addr,
                    "vfid": -1, "ishttps": is_https, "debug": 0,
                    "version": None,
                    "throttle_delay": throttle_delay,
                    "auth_token": tokenArg,
                    "sessionless": sessionless}
        else:
            if isinstance(tokenArg, auth_token_manager):
                authtoken = tokenArg.findToken(ip_addr, username)
                if authtoken is None or authtoken.auth_token is None:
                    return {CREDENTIAL_KEY:
                            {LOGIN_ERROR_KEY: ip_addr + " No Auth Token"},
                            "user": username,
                            "ip_addr": ip_addr,
                            "vfid": -1, "ishttps": is_https, "debug": 0,
                            "version": None,
                            "throttle_delay": throttle_delay,
                            "auth_token": tokenArg,
                            "sessionless": sessionless}
            elif isinstance(tokenArg, str):
                authtoken = auth_token()
                authtoken.setValue(ip_addr, username, tokenArg)

    try:
        credential, contentdict = pyfos_login.login(username,
                                                    password,
                                                    ip_addr,
                                                    is_https,
                                                    throttle_delay,
                                                    authtoken,
                                                    sessionless)
    except socket_error as serr:
        if serr.errno == errno.ECONNREFUSED:
            return {CREDENTIAL_KEY:
                    {LOGIN_ERROR_KEY: ip_addr + " refused connection"},
                    "user": username,
                    "ip_addr": ip_addr,
                    "vfid": -1, "ishttps": is_https, "debug": 0,
                    "version": None,
                    "throttle_delay": throttle_delay,
                    "auth_token": tokenArg,
                    "sessionless": sessionless}
        elif serr.errno == errno.EHOSTUNREACH:
            return {CREDENTIAL_KEY:
                    {LOGIN_ERROR_KEY: ip_addr + " not reachable"},
                    "user": username,
                    "ip_addr": ip_addr,
                    "vfid": -1, "ishttps": is_https, "debug": 0,
                    "version": None,
                    "throttle_delay": throttle_delay,
                    "auth_token": tokenArg,
                    "sessionless": sessionless}
        else:
            return {CREDENTIAL_KEY:
                    {LOGIN_ERROR_KEY: ip_addr + " unknown error: " +
                     str(serr.errno)},
                    "user": username,
                    "ip_addr": ip_addr,
                    "vfid": -1, "ishttps": is_https, "debug": 0,
                    "version": None,
                    "throttle_delay": throttle_delay,
                    "auth_token": tokenArg,
                    "sessionless": sessionless}

    if isinstance(credential, dict) and LOGIN_ERROR_KEY in credential.keys():
        return {CREDENTIAL_KEY:
                {LOGIN_ERROR_KEY: ip_addr + " " + credential[LOGIN_ERROR_KEY]},
                "ip_addr": ip_addr,
                "user": username,
                "vfid": -1, "ishttps": is_https, "debug": 0,
                "version": None,
                "throttle_delay": throttle_delay,
                "auth_token": tokenArg,
                "sessionless": sessionless}

    if isinstance(credential, list):
        return credential
    else:
        # default version
        defversion = fosversion("8.2.1b")
        defcredver = fosversion("1.3.0")
        credver = None
        session = {CREDENTIAL_KEY: credential, "ip_addr": ip_addr,
                   "user": username,
                   "vfid": -1, "ishttps": is_https,
                   "debug": 0, "version": None,
                   "throttle_delay": throttle_delay,
                   "auth_token": tokenArg,
                   "sessionless": sessionless}
        session.update(contentdict)
        if contentdict['content-version'] is not None:
            credverstr = str(contentdict['content-version'])
            credverlist = credverstr.split("=")
            credver = fosversion(credverlist[1])

        if credver is not None and credver >= defcredver:
            session["version"] = defversion

        switchobj = pyfos_switch.fibrechannel_switch.get(session)
        if isinstance(switchobj, pyfos_switch.fibrechannel_switch):
            fwversion = fosversion(switchobj.peek_firmware_version())
            if fwversion == defversion and fwversion.release == "*":
                fwversion = defversion
            try:
                fwversion.to_string()
            except AttributeError as error:
                switchobj.dbg_print(4, "Incorrect FOS version retrieved :",
                                    switchobj.peek_firmware_version(), error)
                logout(session)
                return {CREDENTIAL_KEY:
                        {LOGIN_ERROR_KEY: ip_addr +
                         " Incorrect FOS version string :" +
                         switchobj.peek_firmware_version()},
                        "ip_addr": ip_addr,
                        "user": username,
                        "vfid": -1, "ishttps": is_https, "debug": 0,
                        "version": switchobj.peek_firmware_version(),
                        "throttle_delay": throttle_delay,
                        "auth_token": tokenArg,
                        "sessionless": sessionless}
            session["version"] = fwversion

        debug_set(session, verbose)

        session["username"] = username
        session["password"] = password

        return session


def logout(session):
    """Terminate a session to FOS.

    :param session: Dictionary of the session returned by :func:`login`.
    :rtype: None.
    """
    sessionless = session.get('sessionless', False)
    nocred = session.get('nocredential', False)
    if sessionless is True or nocred is True:
        return {'logout-success': 'Success'}

    delay = pyfos_util.getthrottledelay(session)
    try:
        ret = pyfos_login.logout(session.get(CREDENTIAL_KEY),
                                 session.get("ip_addr"),
                                 session.get("ishttps"),
                                 delay)
    except socket_error as serr:
        ret = {'logout-error': serr}
        # print(ret)

    return ret


def throttle_delay_set(session, delay):
    """Assign a new throttle delay to a session. The throttle delay
    value, in seconds, is added to each request to avoid hitting
    RESTCONF throttling limit.

    :param session: Dictionary of session returned by :func:`login`.
    :param delay: New delay to be assigned to the session, in seconds.
    :rtype: None.
    """
    session['throttle_delay'] = delay

    return "Success"


def vfid_set(session, vfid):
    """Assign a new VFID to a session.

    :param session: Dictionary of session returned by :func:`login`.
    :param vfid: New VFID to be assigned to the session.
    :rtype: None.
    """
    session['vfid'] = vfid

    return "Success"


def is_failed_login(session):
    nocred = session.get("nocredential", False)
    if not nocred and LOGIN_ERROR_KEY in session.get(CREDENTIAL_KEY).keys():
        return True

    return False


def debug_set(session, debug):
    session['debug'] = debug

    return "Success"
