"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Archiver = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class Archiver extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.logGroupKmsKey = this.createLogGroupKey();
        this.logGroup = this.createLogGroup();
        const topic = new aws_cdk_lib_1.aws_sns.Topic(this, 'notifications', {
            displayName: 'archiver-notifications',
        });
        this.bucket = this.createArchiveBucket();
        this.bucket.addEventNotification(aws_cdk_lib_1.aws_s3.EventType.OBJECT_CREATED, new aws_cdk_lib_1.aws_s3_notifications.SnsDestination(topic));
        this.createProjects();
        new aws_cdk_lib_1.CfnOutput(this, 's3-bucket-arn', {
            description: 'ARN of the S3 bucket storing the repositories.',
            value: this.bucket.bucketArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-arn', {
            description: 'ARN of the Cloudwatch Log group storing the code build logs.',
            value: this.logGroup.logGroupArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'log-group-key', {
            description: 'ARN of the KMS key used to encrypt the Cloudwatch logs.',
            value: this.logGroupKmsKey.keyArn,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'sns-topic-arn', {
            description: 'ARN of the SNS topic.',
            value: topic.topicArn,
        });
    }
    createLogGroup() {
        const loggroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'loggroup', {
            encryptionKey: this.logGroupKmsKey,
            retention: this.props.retentionDays
                ? this.props.retentionDays
                : aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        loggroup.node.addDependency(this.logGroupKmsKey);
        return loggroup;
    }
    /**
     *Create the S3 bucket that will later store the repositories.
     *
     * @private
     * @return {*}
     * @memberof Archiver
     */
    createArchiveBucket() {
        return new aws_cdk_lib_1.aws_s3.Bucket(this, 'destination', {
            blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.RETAIN,
            lifecycleRules: [
                {
                    expiration: aws_cdk_lib_1.Duration.days(360),
                    transitions: [
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.GLACIER,
                            transitionAfter: aws_cdk_lib_1.Duration.days(90),
                        },
                        {
                            storageClass: aws_cdk_lib_1.aws_s3.StorageClass.DEEP_ARCHIVE,
                            transitionAfter: aws_cdk_lib_1.Duration.days(180),
                        },
                    ],
                },
            ],
            publicReadAccess: false,
            versioned: true,
        });
    }
    createLogGroupKey() {
        const key = new aws_cdk_lib_1.aws_kms.Key(this, 'loggroupKey', {
            description: 'Repository Archiver',
            enableKeyRotation: true,
            pendingWindow: aws_cdk_lib_1.Duration.days(7),
            keyUsage: aws_cdk_lib_1.aws_kms.KeyUsage.ENCRYPT_DECRYPT,
            keySpec: aws_cdk_lib_1.aws_kms.KeySpec.SYMMETRIC_DEFAULT,
            alias: 'archiver-loggroup-key',
        });
        key.grantEncryptDecrypt(new aws_cdk_lib_1.aws_iam.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`));
        return key;
    }
    /**
     * Creates for each backup configuration a separate CodeBuild project.
     *
     * @private
     * @memberof Archiver
     */
    createProjects() {
        this.props.backupConfigurations.forEach((element) => {
            const project = this.createProject(element);
            project.enableBatchBuilds();
            this.bucket.grantReadWrite(project);
        });
    }
    /**
     * Create a CodeBuild project.
     *
     * @private
     * @param {BackupConfiguration} element
     * @return {*}
     * @memberof Archiver
     */
    createProject(element) {
        return new aws_cdk_lib_1.aws_codebuild.Project(this, 'archiver-' + element.organizationName + '-' + element.projectName, {
            timeout: aws_cdk_lib_1.Duration.hours(5),
            projectName: 'AzureDevOpsGitBackup' +
                '-' +
                element.organizationName +
                '-' +
                element.projectName,
            description: 'Backup Azure DevOps git repositories to an S3 bucket.',
            checkSecretsInPlainTextEnvVariables: true,
            concurrentBuildLimit: 90,
            environment: {
                environmentVariables: {
                    TOKEN: {
                        value: element.secretArn + ':pat',
                        type: aws_cdk_lib_1.aws_codebuild.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    },
                    ORGANIZATION: { value: element.organizationName },
                    PROJECT: { value: element.projectName },
                },
                buildImage: aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0,
            },
            logging: {
                cloudWatch: {
                    enabled: true,
                    logGroup: this.logGroup,
                },
            },
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject({
                version: 0.2,
                batch: {
                    'fail-fast': false,
                    'build-list': this.createBatchConfiguration(element.repositoryNames),
                },
                phases: {
                    build: {
                        commands: [
                            'git clone --mirror "https://${TOKEN}@dev.azure.com/${ORGANIZATION}/${PROJECT}/_git/${REPOSITORY}"',
                            'tar czf ${REPOSITORY}.tgz ./${REPOSITORY}.git',
                            'aws s3 cp ./${REPOSITORY}.tgz ' +
                                this.bucket.s3UrlForObject() +
                                '/${ORGANIZATION}/${PROJECT}/${REPOSITORY}.tgz',
                        ],
                    },
                },
            }),
        });
    }
    createBatchConfiguration(repositoryNames) {
        const output = [];
        repositoryNames.forEach((element) => {
            output.push({
                identifier: 'build_' + element.replace(/-/g, '_'),
                env: {
                    variables: {
                        REPOSITORY: element,
                    },
                },
            });
        });
        return output;
    }
}
exports.Archiver = Archiver;
_a = JSII_RTTI_SYMBOL_1;
Archiver[_a] = { fqn: "azure-devops-repository-archiver.Archiver", version: "0.0.9" };
//# sourceMappingURL=data:application/json;base64,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