"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const clients = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const client_lambda_shared_1 = require("./client.lambda-shared");
const constants_1 = require("./constants");
const s3 = clients.s3();
const sqs = clients.sqs();
const lambda = clients.lambda();
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event) {
    var _a;
    console.log(`Event: ${JSON.stringify(event)}`);
    const client = await client_lambda_shared_1.DenyListClient.newClient();
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setDimensions();
        const ruleCount = Object.keys(client.map).length;
        metrics.putMetric("DenyListRuleCount" /* DENY_LIST_RULE_COUNT */, ruleCount, aws_embedded_metrics_1.Unit.Count);
    })();
    const packageData = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_PACKAGE_DATA_BUCKET_NAME);
    const pruneQueue = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_QUEUE_URL);
    const keyPrefix = env_lambda_shared_1.requireEnv(constants_1.ENV_PRUNE_PACKAGE_DATA_KEY_PREFIX);
    const objectsFound = new Array();
    for (const nameVersion of Object.keys(client.map)) {
        const prefix = `${keyPrefix}${nameVersion}/`;
        console.log(`Querying bucket ${packageData} for all objects with prefix ${prefix}`);
        // for each prefix, query the package data bucket for all objects with that
        // prefix and delete them
        let continuation = undefined;
        do {
            const req = {
                Bucket: packageData,
                Prefix: prefix,
                ContinuationToken: continuation,
            };
            const result = await s3.listObjectsV2(req).promise();
            continuation = result.NextContinuationToken;
            // queue all objects for deletion
            for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
                if (!object.Key) {
                    continue;
                }
                const sendMessageRequest = {
                    QueueUrl: pruneQueue,
                    MessageBody: object.Key,
                };
                console.log(JSON.stringify({ sendMessageRequest }));
                const sendMessageResponse = await sqs.sendMessage(sendMessageRequest).promise();
                console.log(JSON.stringify({ sendMessageResponse }));
                objectsFound.push(object.Key);
            }
        } while (continuation);
        // trigger the "on change" handler objects were found and we have a handler
        const onChangeFunctionName = process.env[constants_1.ENV_PRUNE_ON_CHANGE_FUNCTION_NAME];
        if (onChangeFunctionName && objectsFound.length > 0) {
            console.log(`Triggering a on-change handler: ${onChangeFunctionName}`);
            const onChangeCallbackRequest = {
                FunctionName: onChangeFunctionName,
                InvocationType: 'Event',
            };
            console.log(JSON.stringify({ onChangeCallbackRequest }));
            const onChangeCallbackResponse = await lambda.invoke(onChangeCallbackRequest).promise();
            console.log(JSON.stringify({ onChangeCallbackResponse }));
        }
    }
}
exports.handler = handler;
//# sourceMappingURL=data:application/json;base64,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