"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = exports.ServerlessLaravel = exports.ServerlessApi = void 0;
const cdk = require("@aws-cdk/core");
const apigateway = require("@aws-cdk/aws-apigatewayv2");
const lambda = require("@aws-cdk/aws-lambda");
const rds = require("@aws-cdk/aws-rds");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
/**
 * Use `ServerlessApi` to create the serverless API resource
 */
class ServerlessApi extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id);
        const DEFAULT_LAMBDA_ASSET_PATH = path.join(__dirname, '../composer/laravel58-bref');
        const DEFAULT_DB_MASTER_USER = 'admin';
        this.vpc = (_a = props.vpc) !== null && _a !== void 0 ? _a : new aws_ec2_1.Vpc(this, 'Vpc', { maxAzs: 3, natGateways: 1 });
        this.handler = (_b = props.handler) !== null && _b !== void 0 ? _b : new lambda.Function(this, 'handler', {
            runtime: lambda.Runtime.PROVIDED,
            handler: 'public/index.php',
            layers: [
                lambda.LayerVersion.fromLayerVersionArn(this, 'BrefPHPLayer', props.brefLayerVersion),
            ],
            code: lambda.Code.fromAsset((_c = props === null || props === void 0 ? void 0 : props.lambdaCodePath) !== null && _c !== void 0 ? _c : DEFAULT_LAMBDA_ASSET_PATH),
            environment: {
                APP_STORAGE: '/tmp',
                DB_WRITER: (_e = (_d = props.databaseConfig) === null || _d === void 0 ? void 0 : _d.writerEndpoint) !== null && _e !== void 0 ? _e : '',
                DB_READER: (_j = (_g = (_f = props.databaseConfig) === null || _f === void 0 ? void 0 : _f.readerEndpoint) !== null && _g !== void 0 ? _g : (_h = props.databaseConfig) === null || _h === void 0 ? void 0 : _h.writerEndpoint) !== null && _j !== void 0 ? _j : '',
                DB_USER: (_l = (_k = props.databaseConfig) === null || _k === void 0 ? void 0 : _k.masterUserName) !== null && _l !== void 0 ? _l : DEFAULT_DB_MASTER_USER,
            },
            timeout: cdk.Duration.seconds(120),
            vpc: this.vpc,
        });
        // allow lambda execution role to connect to RDS proxy
        if (props.rdsProxy) {
            this.handler.addToRolePolicy(new iam.PolicyStatement({
                actions: ['rds-db:connect'],
                resources: [props.rdsProxy.dbProxyArn],
            }));
        }
        const endpoint = new apigateway.HttpApi(this, 'apiservice', {
            defaultIntegration: new apigateway.LambdaProxyIntegration({
                handler: this.handler,
            }),
        });
        new cdk.CfnOutput(this, 'EndpointURL', { value: endpoint.url });
    }
}
exports.ServerlessApi = ServerlessApi;
/**
 * Use `ServerlessLaravel` to create the serverless Laravel resource
 */
class ServerlessLaravel extends ServerlessApi {
    constructor(scope, id, props) {
        super(scope, id, {
            lambdaCodePath: props.laravelPath,
            brefLayerVersion: props.brefLayerVersion,
            handler: props.handler,
            vpc: props.vpc,
            databaseConfig: props.databaseConfig,
            rdsProxy: props.rdsProxy,
        });
    }
}
exports.ServerlessLaravel = ServerlessLaravel;
class DatabaseCluster extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this._runBefore = [];
        this.masterUser = (_a = props.masterUserName) !== null && _a !== void 0 ? _a : 'admin';
        // generate and store password for masterUser in the secrets manager
        const masterUserSecret = new secretsmanager.Secret(this, 'DbMasterSecret', {
            secretName: `${cdk.Stack.of(this).stackName}-DbMasterSecret`,
            generateSecretString: {
                secretStringTemplate: JSON.stringify({
                    username: this.masterUser,
                }),
                passwordLength: 12,
                excludePunctuation: true,
                includeSpace: false,
                generateStringKey: 'password',
            },
        });
        this.masterPassword = masterUserSecret;
        const dbConnectionGroup = new aws_ec2_1.SecurityGroup(this, 'DB Secuirty Group', {
            vpc: props.vpc,
        });
        dbConnectionGroup.connections.allowInternally(aws_ec2_1.Port.tcp(3306));
        if (props.singleInstanceOnly === true) {
            this.dbInstance = new rds.DatabaseInstance(this, 'DBInstance', {
                engine: rds.DatabaseInstanceEngine.MYSQL,
                masterUsername: masterUserSecret.secretValueFromJson('username').toString(),
                masterUserPassword: masterUserSecret.secretValueFromJson('password'),
                vpc: props.vpc,
                securityGroups: [dbConnectionGroup],
                instanceType: props.instanceType,
                deletionProtection: false,
            });
        }
        else {
            this.dbCluster = new rds.DatabaseCluster(this, 'DBCluster', {
                engine: rds.DatabaseClusterEngine.auroraMysql({
                    version: rds.AuroraMysqlEngineVersion.VER_2_08_1,
                }),
                instanceProps: {
                    vpc: props.vpc,
                    instanceType: (_b = props.instanceType) !== null && _b !== void 0 ? _b : new aws_ec2_1.InstanceType('t3.medium'),
                    securityGroups: [dbConnectionGroup],
                },
                masterUser: {
                    username: masterUserSecret.secretValueFromJson('username').toString(),
                    password: masterUserSecret.secretValueFromJson('password'),
                },
                instances: 1,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            // Workaround for bug where TargetGroupName is not set but required
            let cfnDbInstance = this.dbCluster.node.children.find((child) => {
                return child instanceof rds.CfnDBInstance;
            });
            this._runBefore.push(cfnDbInstance);
        }
        // enable the RDS proxy by default
        if (!props.rdsProxy === false) {
            // create iam role for RDS proxy
            const rdsProxyRole = new iam.Role(this, 'RdsProxyRole', {
                assumedBy: new iam.ServicePrincipal('rds.amazonaws.com'),
            });
            // see: https://aws.amazon.com/tw/blogs/compute/using-amazon-rds-proxy-with-aws-lambda/
            rdsProxyRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'secretsmanager:GetResourcePolicy',
                    'secretsmanager:GetSecretValue',
                    'secretsmanager:DescribeSecret',
                    'secretsmanager:ListSecretVersionIds',
                ],
                resources: [masterUserSecret.secretArn],
            }));
            const proxyOptions = {
                vpc: props.vpc,
                secrets: [masterUserSecret],
                iamAuth: true,
                dbProxyName: `${cdk.Stack.of(this).stackName}-RDSProxy`,
                securityGroups: [dbConnectionGroup],
                role: rdsProxyRole,
            };
            // create the RDS proxy
            if (this.dbCluster) {
                this.rdsProxy = this.dbCluster.addProxy('RDSProxy', proxyOptions);
            }
            if (this.dbInstance) {
                this.rdsProxy = this.dbInstance.addProxy('RDSProxy', proxyOptions);
            }
            // ensure DB instance is ready before creating the proxy
            (_c = this.rdsProxy) === null || _c === void 0 ? void 0 : _c.node.addDependency(...this._runBefore);
        }
    }
}
exports.DatabaseCluster = DatabaseCluster;
//# sourceMappingURL=data:application/json;base64,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