"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Lambda = exports.LambdaInvocationType = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * The type of invocation to use for a Lambda Action.
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * The function will be invoked asynchronously.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * The function will be invoked sychronously. Use RequestResponse only when
     * you want to make a mail flow decision, such as whether to stop the receipt
     * rule or the receipt rule set.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
/**
 * Calls an AWS Lambda function, and optionally, publishes a notification to
 * Amazon SNS.
 */
class Lambda {
    constructor(props) {
        this.props = props;
    }
    bind(rule) {
        // Allow SES to invoke Lambda function
        // See https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-permissions.html#receiving-email-permissions-lambda
        const permissionId = 'AllowSes';
        if (!this.props.function.permissionsNode.tryFindChild(permissionId)) {
            this.props.function.addPermission(permissionId, {
                action: 'lambda:InvokeFunction',
                principal: new iam.ServicePrincipal('ses.amazonaws.com'),
                sourceAccount: cdk.Aws.ACCOUNT_ID,
            });
        }
        // Ensure permission is deployed before rule
        const permission = this.props.function.permissionsNode.tryFindChild(permissionId);
        if (permission) { // The Lambda could be imported
            rule.node.addDependency(permission);
        }
        else {
            // eslint-disable-next-line max-len
            rule.node.addWarning('This rule is using a Lambda action with an imported function. Ensure permission is given to SES to invoke that function.');
        }
        return {
            lambdaAction: {
                functionArn: this.props.function.functionArn,
                invocationType: this.props.invocationType,
                topicArn: this.props.topic ? this.props.topic.topicArn : undefined,
            },
        };
    }
}
exports.Lambda = Lambda;
//# sourceMappingURL=data:application/json;base64,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