"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const kms = require("@aws-cdk/aws-kms");
const lambda = require("@aws-cdk/aws-lambda");
const s3 = require("@aws-cdk/aws-s3");
const ses = require("@aws-cdk/aws-ses");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
const actions = require("../lib");
let stack;
let rule;
let topic;
beforeEach(() => {
    stack = new core_1.Stack();
    const ruleSet = new ses.ReceiptRuleSet(stack, 'RuleSet');
    rule = ruleSet.addRule('Rule');
    topic = new sns.Topic(stack, 'Topic');
});
test('add header action', () => {
    rule.addAction(new actions.AddHeader({
        name: 'X-My-Header',
        value: 'value',
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Rule: {
            Actions: [
                {
                    AddHeaderAction: {
                        HeaderName: 'X-My-Header',
                        HeaderValue: 'value',
                    },
                },
            ],
            Enabled: true,
        },
    });
});
test('add header action with invalid header name', () => {
    expect(() => rule.addAction(new actions.AddHeader({
        name: 'He@der',
        value: 'value',
    }))).toThrow(/`name`/);
});
test('add header action with invalid header value', () => {
    expect(() => rule.addAction(new actions.AddHeader({
        name: 'Header',
        value: `va
    lu`,
    }))).toThrow(/`value`/);
});
test('add bounce action', () => {
    rule.addAction(new actions.Bounce({
        sender: 'noreply@aws.com',
        template: actions.BounceTemplate.MESSAGE_CONTENT_REJECTED,
        topic,
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Rule: {
            Actions: [
                {
                    BounceAction: {
                        Message: 'Message content rejected',
                        Sender: 'noreply@aws.com',
                        SmtpReplyCode: '500',
                        TopicArn: {
                            Ref: 'TopicBFC7AF6E',
                        },
                        StatusCode: '5.6.1',
                    },
                },
            ],
            Enabled: true,
        },
    });
});
test('add lambda action', () => {
    const fn = new lambda.Function(stack, 'Function', {
        code: lambda.Code.fromInline('boom'),
        handler: 'index.handler',
        runtime: lambda.Runtime.NODEJS_10_X,
    });
    rule.addAction(new actions.Lambda({
        function: fn,
        invocationType: actions.LambdaInvocationType.REQUEST_RESPONSE,
        topic,
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Properties: {
            Rule: {
                Actions: [
                    {
                        LambdaAction: {
                            FunctionArn: {
                                'Fn::GetAtt': [
                                    'Function76856677',
                                    'Arn',
                                ],
                            },
                            InvocationType: 'RequestResponse',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E',
                            },
                        },
                    },
                ],
                Enabled: true,
            },
            RuleSetName: {
                Ref: 'RuleSetE30C6C48',
            },
        },
        DependsOn: [
            'FunctionAllowSes1829904A',
        ],
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: 'lambda:InvokeFunction',
        FunctionName: {
            'Fn::GetAtt': [
                'Function76856677',
                'Arn',
            ],
        },
        Principal: 'ses.amazonaws.com',
        SourceAccount: {
            Ref: 'AWS::AccountId',
        },
    });
});
test('add s3 action', () => {
    const bucket = new s3.Bucket(stack, 'Bucket');
    const kmsKey = new kms.Key(stack, 'Key');
    rule.addAction(new actions.S3({
        bucket,
        kmsKey,
        objectKeyPrefix: 'emails/',
        topic,
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Properties: {
            Rule: {
                Actions: [
                    {
                        S3Action: {
                            BucketName: {
                                Ref: 'Bucket83908E77',
                            },
                            KmsKeyArn: {
                                'Fn::GetAtt': [
                                    'Key961B73FD',
                                    'Arn',
                                ],
                            },
                            ObjectKeyPrefix: 'emails/',
                            TopicArn: {
                                Ref: 'TopicBFC7AF6E',
                            },
                        },
                    },
                ],
                Enabled: true,
            },
            RuleSetName: {
                Ref: 'RuleSetE30C6C48',
            },
        },
        DependsOn: [
            'BucketPolicyE9A3008A',
        ],
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::S3::BucketPolicy', {
        Bucket: {
            Ref: 'Bucket83908E77',
        },
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:PutObject',
                    Condition: {
                        StringEquals: {
                            'aws:Referer': {
                                Ref: 'AWS::AccountId',
                            },
                        },
                    },
                    Effect: 'Allow',
                    Principal: {
                        Service: 'ses.amazonaws.com',
                    },
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                {
                                    'Fn::GetAtt': [
                                        'Bucket83908E77',
                                        'Arn',
                                    ],
                                },
                                '/emails/*',
                            ],
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
    });
    expect(stack).toHaveResource('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: [
                        'kms:Create*',
                        'kms:Describe*',
                        'kms:Enable*',
                        'kms:List*',
                        'kms:Put*',
                        'kms:Update*',
                        'kms:Revoke*',
                        'kms:Disable*',
                        'kms:Get*',
                        'kms:Delete*',
                        'kms:ScheduleKeyDeletion',
                        'kms:CancelKeyDeletion',
                        'kms:GenerateDataKey',
                        'kms:TagResource',
                        'kms:UntagResource',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
                {
                    Action: [
                        'km:Encrypt',
                        'kms:GenerateDataKey',
                    ],
                    Condition: {
                        Null: {
                            'kms:EncryptionContext:aws:ses:rule-name': 'false',
                            'kms:EncryptionContext:aws:ses:message-id': 'false',
                        },
                        StringEquals: {
                            'kms:EncryptionContext:aws:ses:source-account': {
                                Ref: 'AWS::AccountId',
                            },
                        },
                    },
                    Effect: 'Allow',
                    Principal: {
                        Service: 'ses.amazonaws.com',
                    },
                    Resource: '*',
                },
            ],
            Version: '2012-10-17',
        },
    });
});
test('add sns action', () => {
    rule.addAction(new actions.Sns({
        encoding: actions.EmailEncoding.BASE64,
        topic,
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Rule: {
            Actions: [
                {
                    SNSAction: {
                        Encoding: 'Base64',
                        TopicArn: {
                            Ref: 'TopicBFC7AF6E',
                        },
                    },
                },
            ],
            Enabled: true,
        },
    });
});
test('add stop action', () => {
    rule.addAction(new actions.Stop({
        topic,
    }));
    expect(stack).toHaveResource('AWS::SES::ReceiptRule', {
        Rule: {
            Actions: [
                {
                    StopAction: {
                        Scope: 'RuleSet',
                        TopicArn: {
                            Ref: 'TopicBFC7AF6E',
                        },
                    },
                },
            ],
            Enabled: true,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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