"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("@aws-cdk/aws-s3-assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
/**
 * Represents an OpenAPI definition asset.
 *
 * @stability stable
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket.
     *
     * @stability stable
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object.
     *
     * The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @stability stable
     * @example
     *
     *   apigateway.ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     *
     * @stability stable
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
    /**
     * Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindAfterCreate(_scope, _restApi) {
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IRestApi(_restApi);
        return;
    }
}
exports.ApiDefinition = ApiDefinition;
_a = JSII_RTTI_SYMBOL_1;
ApiDefinition[_a] = { fqn: "@aws-cdk/aws-apigateway.ApiDefinition", version: "1.139.0" };
/**
 * OpenAPI specification from an S3 archive.
 *
 * @stability stable
 */
class S3ApiDefinition extends ApiDefinition {
    /**
     * @stability stable
     */
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
_b = JSII_RTTI_SYMBOL_1;
S3ApiDefinition[_b] = { fqn: "@aws-cdk/aws-apigateway.S3ApiDefinition", version: "1.139.0" };
/**
 * OpenAPI specification from an inline JSON object.
 *
 * @stability stable
 */
class InlineApiDefinition extends ApiDefinition {
    /**
     * @stability stable
     */
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
_c = JSII_RTTI_SYMBOL_1;
InlineApiDefinition[_c] = { fqn: "@aws-cdk/aws-apigateway.InlineApiDefinition", version: "1.139.0" };
/**
 * OpenAPI specification from a local file.
 *
 * @stability stable
 */
class AssetApiDefinition extends ApiDefinition {
    /**
     * @stability stable
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    /**
     * Called when the specification is initialized to allow this object to bind to the stack, add resources and have fun.
     *
     * @stability stable
     */
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Called after the CFN RestApi resource has been created to allow the Api Definition to bind to it.
     *
     * Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     *
     * @stability stable
     */
    bindAfterCreate(scope, restApi) {
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IRestApi(restApi);
        if (!scope.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const child = constructs_1.Node.of(restApi).defaultChild;
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, this.asset.assetPath);
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'BodyS3Location');
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
_d = JSII_RTTI_SYMBOL_1;
AssetApiDefinition[_d] = { fqn: "@aws-cdk/aws-apigateway.AssetApiDefinition", version: "1.139.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGktZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFDQSxvREFBb0Q7QUFFcEQsaUdBQWlHO0FBQ2pHLDhEQUE4RDtBQUM5RCx5Q0FBeUM7QUFDekMsMkNBQWtDOzs7Ozs7QUFNbEMsTUFBc0IsYUFBYTs7Ozs7O0lBRTFCLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0IsRUFBRSxHQUFXLEVBQUUsYUFBc0I7UUFDOUUsT0FBTyxJQUFJLGVBQWUsQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ3hEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7SUFHTSxNQUFNLENBQUMsVUFBVSxDQUFDLFVBQWU7UUFDdEMsT0FBTyxJQUFJLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0tBQzVDOzs7Ozs7SUFHTSxNQUFNLENBQUMsU0FBUyxDQUFDLElBQVksRUFBRSxPQUFnQztRQUNwRSxPQUFPLElBQUksa0JBQWtCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQzlDOzs7Ozs7Ozs7SUFNTSxlQUFlLENBQUMsTUFBaUIsRUFBRSxRQUFrQjs7UUFDMUQsT0FBTztLQUNSOztBQXRCSCxzQ0F1QkM7Ozs7Ozs7O0FBc0JELE1BQWEsZUFBZ0IsU0FBUSxhQUFhOzs7O0lBR2hELFlBQVksTUFBa0IsRUFBVSxHQUFXLEVBQVUsYUFBc0I7UUFDakYsS0FBSyxFQUFFLENBQUM7UUFEOEIsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUFVLGtCQUFhLEdBQWIsYUFBYSxDQUFTO1FBR2pGLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztTQUNwRTtRQUVELElBQUksQ0FBQyxVQUFVLEdBQUcsTUFBTSxDQUFDLFVBQVUsQ0FBQztLQUNyQzs7Ozs7O0lBRU0sSUFBSSxDQUFDLE1BQWlCO1FBQzNCLE9BQU87WUFDTCxVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLElBQUksQ0FBQyxVQUFVO2dCQUN2QixHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7Z0JBQ2IsT0FBTyxFQUFFLElBQUksQ0FBQyxhQUFhO2FBQzVCO1NBQ0YsQ0FBQztLQUNIOztBQXJCSCwwQ0FzQkM7Ozs7Ozs7O0FBR0QsTUFBYSxtQkFBb0IsU0FBUSxhQUFhOzs7O0lBQ3BELFlBQW9CLFVBQWU7UUFDakMsS0FBSyxFQUFFLENBQUM7UUFEVSxlQUFVLEdBQVYsVUFBVSxDQUFLO1FBR2pDLElBQUksT0FBTSxDQUFDLFVBQVUsQ0FBQyxLQUFLLFFBQVEsRUFBRTtZQUNuQyxNQUFNLElBQUksS0FBSyxDQUFDLHFDQUFxQyxDQUFDLENBQUM7U0FDeEQ7UUFFRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN4QyxNQUFNLElBQUksS0FBSyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7U0FDcEQ7S0FDRjs7Ozs7O0lBRU0sSUFBSSxDQUFDLE1BQWlCO1FBQzNCLE9BQU87WUFDTCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsVUFBVTtTQUNsQyxDQUFDO0tBQ0g7O0FBakJILGtEQWtCQzs7Ozs7Ozs7QUFHRCxNQUFhLGtCQUFtQixTQUFRLGFBQWE7Ozs7SUFHbkQsWUFBNkIsSUFBWSxFQUFtQixVQUFrQyxFQUFHO1FBQy9GLEtBQUssRUFBRSxDQUFDO1FBRG1CLFNBQUksR0FBSixJQUFJLENBQVE7UUFBbUIsWUFBTyxHQUFQLE9BQU8sQ0FBOEI7S0FFaEc7Ozs7OztJQUVNLElBQUksQ0FBQyxLQUFnQjtRQUMxQiw4RkFBOEY7UUFDOUYsSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUM1QixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFO2dCQUN2RCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsR0FBRyxJQUFJLENBQUMsT0FBTzthQUNoQixDQUFDLENBQUM7U0FDSjtRQUVELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7U0FDOUU7UUFFRCxPQUFPO1lBQ0wsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVk7Z0JBQy9CLEdBQUcsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFdBQVc7YUFDNUI7U0FDRixDQUFDO0tBQ0g7Ozs7Ozs7OztJQUVNLGVBQWUsQ0FBQyxLQUFnQixFQUFFLE9BQWlCOztRQUN4RCxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLHVDQUF1QyxDQUFDLEVBQUU7WUFDNUUsT0FBTyxDQUFDLGNBQWM7U0FDdkI7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNmLE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLENBQUMsQ0FBQztTQUNqRTtRQUVELE1BQU0sS0FBSyxHQUFHLGlCQUFJLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFlBQTBCLENBQUM7UUFDMUQsS0FBSyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoRixLQUFLLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxvQ0FBb0MsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0tBQ2pGOztBQXhDSCxnREF5Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBzMyBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0ICogYXMgczNfYXNzZXRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMy1hc3NldHMnO1xuXG4vLyBrZWVwIHRoaXMgaW1wb3J0IHNlcGFyYXRlIGZyb20gb3RoZXIgaW1wb3J0cyB0byByZWR1Y2UgY2hhbmNlIGZvciBtZXJnZSBjb25mbGljdHMgd2l0aCB2Mi1tYWluXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tZHVwbGljYXRlLWltcG9ydHMsIGltcG9ydC9vcmRlclxuaW1wb3J0ICogYXMgY3hhcGkgZnJvbSAnQGF3cy1jZGsvY3gtYXBpJztcbmltcG9ydCB7IE5vZGUgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmblJlc3RBcGkgfSBmcm9tICcuL2FwaWdhdGV3YXkuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElSZXN0QXBpIH0gZnJvbSAnLi9yZXN0YXBpJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEFwaURlZmluaXRpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUJ1Y2tldChidWNrZXQ6IHMzLklCdWNrZXQsIGtleTogc3RyaW5nLCBvYmplY3RWZXJzaW9uPzogc3RyaW5nKTogUzNBcGlEZWZpbml0aW9uIHtcbiAgICByZXR1cm4gbmV3IFMzQXBpRGVmaW5pdGlvbihidWNrZXQsIGtleSwgb2JqZWN0VmVyc2lvbik7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUlubGluZShkZWZpbml0aW9uOiBhbnkpOiBJbmxpbmVBcGlEZWZpbml0aW9uIHtcbiAgICByZXR1cm4gbmV3IElubGluZUFwaURlZmluaXRpb24oZGVmaW5pdGlvbik7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUFzc2V0KGZpbGU6IHN0cmluZywgb3B0aW9ucz86IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMpOiBBc3NldEFwaURlZmluaXRpb24ge1xuICAgIHJldHVybiBuZXcgQXNzZXRBcGlEZWZpbml0aW9uKGZpbGUsIG9wdGlvbnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIGFic3RyYWN0IGJpbmQoc2NvcGU6IENvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBiaW5kQWZ0ZXJDcmVhdGUoX3Njb3BlOiBDb25zdHJ1Y3QsIF9yZXN0QXBpOiBJUmVzdEFwaSkge1xuICAgIHJldHVybjtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwaURlZmluaXRpb25TM0xvY2F0aW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYnVja2V0OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGtleTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB2ZXJzaW9uPzogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHMzTG9jYXRpb24/OiBBcGlEZWZpbml0aW9uUzNMb2NhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGlubGluZURlZmluaXRpb24/OiBhbnk7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBTM0FwaURlZmluaXRpb24gZXh0ZW5kcyBBcGlEZWZpbml0aW9uIHtcbiAgcHJpdmF0ZSBidWNrZXROYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3IoYnVja2V0OiBzMy5JQnVja2V0LCBwcml2YXRlIGtleTogc3RyaW5nLCBwcml2YXRlIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuXG4gICAgaWYgKCFidWNrZXQuYnVja2V0TmFtZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdidWNrZXROYW1lIGlzIHVuZGVmaW5lZCBmb3IgdGhlIHByb3ZpZGVkIGJ1Y2tldCcpO1xuICAgIH1cblxuICAgIHRoaXMuYnVja2V0TmFtZSA9IGJ1Y2tldC5idWNrZXROYW1lO1xuICB9XG5cbiAgcHVibGljIGJpbmQoX3Njb3BlOiBDb25zdHJ1Y3QpOiBBcGlEZWZpbml0aW9uQ29uZmlnIHtcbiAgICByZXR1cm4ge1xuICAgICAgczNMb2NhdGlvbjoge1xuICAgICAgICBidWNrZXQ6IHRoaXMuYnVja2V0TmFtZSxcbiAgICAgICAga2V5OiB0aGlzLmtleSxcbiAgICAgICAgdmVyc2lvbjogdGhpcy5vYmplY3RWZXJzaW9uLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIElubGluZUFwaURlZmluaXRpb24gZXh0ZW5kcyBBcGlEZWZpbml0aW9uIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSBkZWZpbml0aW9uOiBhbnkpIHtcbiAgICBzdXBlcigpO1xuXG4gICAgaWYgKHR5cGVvZihkZWZpbml0aW9uKSAhPT0gJ29iamVjdCcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignZGVmaW5pdGlvbiBzaG91bGQgYmUgb2YgdHlwZSBvYmplY3QnKTtcbiAgICB9XG5cbiAgICBpZiAoT2JqZWN0LmtleXMoZGVmaW5pdGlvbikubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0pTT04gZGVmaW5pdGlvbiBjYW5ub3QgYmUgZW1wdHknKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgIHJldHVybiB7XG4gICAgICBpbmxpbmVEZWZpbml0aW9uOiB0aGlzLmRlZmluaXRpb24sXG4gICAgfTtcbiAgfVxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBBc3NldEFwaURlZmluaXRpb24gZXh0ZW5kcyBBcGlEZWZpbml0aW9uIHtcbiAgcHJpdmF0ZSBhc3NldD86IHMzX2Fzc2V0cy5Bc3NldDtcblxuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHBhdGg6IHN0cmluZywgcHJpdmF0ZSByZWFkb25seSBvcHRpb25zOiBzM19hc3NldHMuQXNzZXRPcHRpb25zID0geyB9KSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QpOiBBcGlEZWZpbml0aW9uQ29uZmlnIHtcbiAgICAvLyBJZiB0aGUgc2FtZSBBc3NldEFQSURlZmluaXRpb24gaXMgdXNlZCBtdWx0aXBsZSB0aW1lcywgcmV0YWluIG9ubHkgdGhlIGZpcnN0IGluc3RhbnRpYXRpb24uXG4gICAgaWYgKHRoaXMuYXNzZXQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5hc3NldCA9IG5ldyBzM19hc3NldHMuQXNzZXQoc2NvcGUsICdBUElEZWZpbml0aW9uJywge1xuICAgICAgICBwYXRoOiB0aGlzLnBhdGgsXG4gICAgICAgIC4uLnRoaXMub3B0aW9ucyxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmFzc2V0LmlzWmlwQXJjaGl2ZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBBc3NldCBjYW5ub3QgYmUgYSAuemlwIGZpbGUgb3IgYSBkaXJlY3RvcnkgKCR7dGhpcy5wYXRofSlgKTtcbiAgICB9XG5cbiAgICByZXR1cm4ge1xuICAgICAgczNMb2NhdGlvbjoge1xuICAgICAgICBidWNrZXQ6IHRoaXMuYXNzZXQuczNCdWNrZXROYW1lLFxuICAgICAgICBrZXk6IHRoaXMuYXNzZXQuczNPYmplY3RLZXksXG4gICAgICB9LFxuICAgIH07XG4gIH1cblxuICBwdWJsaWMgYmluZEFmdGVyQ3JlYXRlKHNjb3BlOiBDb25zdHJ1Y3QsIHJlc3RBcGk6IElSZXN0QXBpKSB7XG4gICAgaWYgKCFzY29wZS5ub2RlLnRyeUdldENvbnRleHQoY3hhcGkuQVNTRVRfUkVTT1VSQ0VfTUVUQURBVEFfRU5BQkxFRF9DT05URVhUKSkge1xuICAgICAgcmV0dXJuOyAvLyBub3QgZW5hYmxlZFxuICAgIH1cblxuICAgIGlmICghdGhpcy5hc3NldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdiaW5kVG9SZXNvdXJjZSgpIG11c3QgYmUgY2FsbGVkIGFmdGVyIGJpbmQoKScpO1xuICAgIH1cblxuICAgIGNvbnN0IGNoaWxkID0gTm9kZS5vZihyZXN0QXBpKS5kZWZhdWx0Q2hpbGQgYXMgQ2ZuUmVzdEFwaTtcbiAgICBjaGlsZC5hZGRNZXRhZGF0YShjeGFwaS5BU1NFVF9SRVNPVVJDRV9NRVRBREFUQV9QQVRIX0tFWSwgdGhpcy5hc3NldC5hc3NldFBhdGgpO1xuICAgIGNoaWxkLmFkZE1ldGFkYXRhKGN4YXBpLkFTU0VUX1JFU09VUkNFX01FVEFEQVRBX1BST1BFUlRZX0tFWSwgJ0JvZHlTM0xvY2F0aW9uJyk7XG4gIH1cbn0iXX0=