import * as cloudwatch from '@aws-cdk/aws-cloudwatch';
import { IVpcEndpoint } from '@aws-cdk/aws-ec2';
import * as iam from '@aws-cdk/aws-iam';
import { IResource as IResourceBase, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { ApiDefinition } from './api-definition';
import { ApiKeyOptions, IApiKey } from './api-key';
import { CfnAccount, CfnRestApi } from './apigateway.generated';
import { Deployment } from './deployment';
import { DomainName, DomainNameOptions } from './domain-name';
import { GatewayResponse, GatewayResponseOptions } from './gateway-response';
import { Method } from './method';
import { Model, ModelOptions } from './model';
import { RequestValidator, RequestValidatorOptions } from './requestvalidator';
import { IResource, ResourceOptions } from './resource';
import { Stage, StageOptions } from './stage';
import { UsagePlan, UsagePlanProps } from './usage-plan';
/**
 * @stability stable
 */
export interface IRestApi extends IResourceBase {
    /**
     * The ID of this API Gateway RestApi.
     *
     * @stability stable
     * @attribute true
     */
    readonly restApiId: string;
    /**
     * The resource ID of the root resource.
     *
     * @stability stable
     * @attribute true
     */
    readonly restApiRootResourceId: string;
    /**
     * API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * `undefined` when no deployment is configured.
     *
     * @stability stable
     */
    readonly latestDeployment?: Deployment;
    /**
     * API Gateway stage that points to the latest deployment (if defined).
     *
     * @stability stable
     */
    deploymentStage: Stage;
    /**
     * Represents the root resource ("/") of this API. Use it to define the API model:.
     *
     * api.root.addMethod('ANY', redirectToHomePage); // "ANY /"
     *     api.root.addResource('friends').addMethod('GET', getFriendsHandler); // "GET /friends"
     *
     * @stability stable
     */
    readonly root: IResource;
    /**
     * Gets the "execute-api" ARN.
     *
     * @param method The method (default `*`).
     * @param path The resource path.
     * @param stage The stage (default `*`).
     * @returns The "execute-api" ARN.
     * @default "*" returns the execute API ARN for all methods/resources in
     * this API.
     * @stability stable
     */
    arnForExecuteApi(method?: string, path?: string, stage?: string): string;
}
/**
 * Represents the props that all Rest APIs share.
 *
 * @stability stable
 */
export interface RestApiBaseProps {
    /**
     * Indicates if a Deployment should be automatically created for this API, and recreated when the API model (resources, methods) changes.
     *
     * Since API Gateway deployments are immutable, When this option is enabled
     * (by default), an AWS::ApiGateway::Deployment resource will automatically
     * created with a logical ID that hashes the API model (methods, resources
     * and options). This means that when the model changes, the logical ID of
     * this CloudFormation resource will change, and a new deployment will be
     * created.
     *
     * If this is set, `latestDeployment` will refer to the `Deployment` object
     * and `deploymentStage` will refer to a `Stage` that points to this
     * deployment. To customize the stage options, use the `deployOptions`
     * property.
     *
     * A CloudFormation Output will also be defined with the root URL endpoint
     * of this REST API.
     *
     * @default true
     * @stability stable
     */
    readonly deploy?: boolean;
    /**
     * Options for the API Gateway stage that will always point to the latest deployment when `deploy` is enabled.
     *
     * If `deploy` is disabled,
     * this value cannot be set.
     *
     * @default - Based on defaults of `StageOptions`.
     * @stability stable
     */
    readonly deployOptions?: StageOptions;
    /**
     * Retains old deployment resources when the API changes.
     *
     * This allows
     * manually reverting stages to point to old deployments via the AWS
     * Console.
     *
     * @default false
     * @stability stable
     */
    readonly retainDeployments?: boolean;
    /**
     * A name for the API Gateway RestApi resource.
     *
     * @default - ID of the RestApi construct.
     * @stability stable
     */
    readonly restApiName?: string;
    /**
     * Custom header parameters for the request.
     *
     * @default - No parameters.
     * @see https://docs.aws.amazon.com/cli/latest/reference/apigateway/import-rest-api.html
     * @stability stable
     */
    readonly parameters?: {
        [key: string]: string;
    };
    /**
     * A policy document that contains the permissions for this RestApi.
     *
     * @default - No policy.
     * @stability stable
     */
    readonly policy?: iam.PolicyDocument;
    /**
     * Indicates whether to roll back the resource if a warning occurs while API Gateway is creating the RestApi resource.
     *
     * @default false
     * @stability stable
     */
    readonly failOnWarnings?: boolean;
    /**
     * Configure a custom domain name and map it to this API.
     *
     * @default - no domain name is defined, use `addDomainName` or directly define a `DomainName`.
     * @stability stable
     */
    readonly domainName?: DomainNameOptions;
    /**
     * Automatically configure an AWS CloudWatch role for API Gateway.
     *
     * @default true
     * @stability stable
     */
    readonly cloudWatchRole?: boolean;
    /**
     * Export name for the CfnOutput containing the API endpoint.
     *
     * @default - when no export name is given, output will be created without export
     * @stability stable
     */
    readonly endpointExportName?: string;
    /**
     * A list of the endpoint types of the API.
     *
     * Use this property when creating
     * an API.
     *
     * @default EndpointType.EDGE
     * @stability stable
     */
    readonly endpointTypes?: EndpointType[];
    /**
     * Specifies whether clients can invoke the API using the default execute-api endpoint.
     *
     * To require that clients use a custom domain name to invoke the
     * API, disable the default endpoint.
     *
     * @default false
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-apigateway-restapi.html
     * @stability stable
     */
    readonly disableExecuteApiEndpoint?: boolean;
}
/**
 * (deprecated) Represents the props that all Rest APIs share.
 *
 * @deprecated - superseded by `RestApiBaseProps`
 */
export interface RestApiOptions extends RestApiBaseProps, ResourceOptions {
}
/**
 * Props to create a new instance of RestApi.
 *
 * @stability stable
 */
export interface RestApiProps extends RestApiOptions {
    /**
     * A description of the purpose of this API Gateway RestApi resource.
     *
     * @default - No description.
     * @stability stable
     */
    readonly description?: string;
    /**
     * The list of binary media mime-types that are supported by the RestApi resource, such as "image/png" or "application/octet-stream".
     *
     * @default - RestApi supports only UTF-8-encoded text payloads.
     * @stability stable
     */
    readonly binaryMediaTypes?: string[];
    /**
     * A nullable integer that is used to enable compression (with non-negative between 0 and 10485760 (10M) bytes, inclusive) or disable compression (when undefined) on an API.
     *
     * When compression is enabled, compression or
     * decompression is not applied on the payload if the payload size is
     * smaller than this value. Setting it to zero allows compression for any
     * payload size.
     *
     * @default - Compression is disabled.
     * @stability stable
     */
    readonly minimumCompressionSize?: number;
    /**
     * The ID of the API Gateway RestApi resource that you want to clone.
     *
     * @default - None.
     * @stability stable
     */
    readonly cloneFrom?: IRestApi;
    /**
     * The source of the API key for metering requests according to a usage plan.
     *
     * @default - Metering is disabled.
     * @stability stable
     */
    readonly apiKeySourceType?: ApiKeySourceType;
    /**
     * The EndpointConfiguration property type specifies the endpoint types of a REST API.
     *
     * @default EndpointType.EDGE
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-apigateway-restapi-endpointconfiguration.html
     * @stability stable
     */
    readonly endpointConfiguration?: EndpointConfiguration;
}
/**
 * Props to instantiate a new SpecRestApi.
 *
 * @stability stable
 */
export interface SpecRestApiProps extends RestApiBaseProps {
    /**
     * An OpenAPI definition compatible with API Gateway.
     *
     * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-import-api.html
     * @stability stable
     */
    readonly apiDefinition: ApiDefinition;
}
/**
 * Base implementation that are common to various implementations of IRestApi.
 *
 * @stability stable
 */
export declare abstract class RestApiBase extends Resource implements IRestApi {
    /**
     * Checks if the given object is an instance of RestApiBase.
     * @internal
     */
    static _isRestApiBase(x: any): x is RestApiBase;
    /**
     * API Gateway deployment that represents the latest changes of the API.
     *
     * This resource will be automatically updated every time the REST API model changes.
     * This will be undefined if `deploy` is false.
     *
     * @stability stable
     */
    get latestDeployment(): Deployment | undefined;
    /**
     * The first domain name mapped to this API, if defined through the `domainName` configuration prop, or added via `addDomainName`.
     *
     * @stability stable
     */
    get domainName(): DomainName | undefined;
    /**
     * The ID of this API Gateway RestApi.
     *
     * @stability stable
     */
    abstract readonly restApiId: string;
    /**
     * The resource ID of the root resource.
     *
     * @stability stable
     * @attribute true
     */
    abstract readonly restApiRootResourceId: string;
    /**
     * Represents the root resource of this API endpoint ('/').
     *
     * Resources and Methods are added to this resource.
     *
     * @stability stable
     */
    abstract readonly root: IResource;
    /**
     * API Gateway stage that points to the latest deployment (if defined).
     *
     * If `deploy` is disabled, you will need to explicitly assign this value in order to
     * set up integrations.
     *
     * @stability stable
     */
    deploymentStage: Stage;
    /**
     * A human friendly name for this Rest API.
     *
     * Note that this is different from `restApiId`.
     *
     * @stability stable
     * @attribute true
     */
    readonly restApiName: string;
    private _latestDeployment?;
    private _domainName?;
    /**
     * @stability stable
     */
    protected cloudWatchAccount?: CfnAccount;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: RestApiBaseProps);
    /**
     * Returns the URL for an HTTP path.
     *
     * Fails if `deploymentStage` is not set either by `deploy` or explicitly.
     *
     * @stability stable
     */
    urlForPath(path?: string): string;
    /**
     * Defines an API Gateway domain name and maps it to this API.
     *
     * @param id The construct id.
     * @param options custom domain options.
     * @stability stable
     */
    addDomainName(id: string, options: DomainNameOptions): DomainName;
    /**
     * Adds a usage plan.
     *
     * @stability stable
     */
    addUsagePlan(id: string, props?: UsagePlanProps): UsagePlan;
    /**
     * Gets the "execute-api" ARN.
     *
     * @stability stable
     */
    arnForExecuteApi(method?: string, path?: string, stage?: string): string;
    /**
     * Adds a new gateway response.
     *
     * @stability stable
     */
    addGatewayResponse(id: string, options: GatewayResponseOptions): GatewayResponse;
    /**
     * Add an ApiKey.
     *
     * @stability stable
     */
    addApiKey(id: string, options?: ApiKeyOptions): IApiKey;
    /**
     * Returns the given named metric for this API.
     *
     * @stability stable
     */
    metric(metricName: string, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of client-side errors captured in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricClientError(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of server-side errors captured in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricServerError(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of requests served from the API cache in a given period.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricCacheHitCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the number of requests served from the backend in a given period, when API caching is enabled.
     *
     * Default: sum over 5 minutes
     *
     * @stability stable
     */
    metricCacheMissCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the total number API requests in a given period.
     *
     * Default: sample count over 5 minutes
     *
     * @stability stable
     */
    metricCount(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Metric for the time between when API Gateway relays a request to the backend and when it receives a response from the backend.
     *
     * Default: average over 5 minutes.
     *
     * @stability stable
     */
    metricIntegrationLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * The time between when API Gateway receives a request from a client and when it returns a response to the client.
     *
     * The latency includes the integration latency and other API Gateway overhead.
     *
     * Default: average over 5 minutes.
     *
     * @stability stable
     */
    metricLatency(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method: Method): void;
    /**
     * Associates a Deployment resource with this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment: Deployment): void;
    /**
     * Associates a Stage with this REST API
     *
     * @internal
     */
    _attachStage(stage: Stage): void;
    /**
     * @internal
     */
    protected _configureCloudWatchRole(apiResource: CfnRestApi): void;
    /**
     * @deprecated This method will be made internal. No replacement
     */
    protected configureCloudWatchRole(apiResource: CfnRestApi): void;
    /**
     * @deprecated This method will be made internal. No replacement
     */
    protected configureDeployment(props: RestApiBaseProps): void;
    /**
     * @internal
     */
    protected _configureDeployment(props: RestApiBaseProps): void;
    /**
     * @internal
     */
    protected _configureEndpoints(props: RestApiProps): CfnRestApi.EndpointConfigurationProperty | undefined;
    private cannedMetric;
}
/**
 * Represents a REST API in Amazon API Gateway, created with an OpenAPI specification.
 *
 * Some properties normally accessible on @see {@link RestApi} - such as the description -
 * must be declared in the specification. All Resources and Methods need to be defined as
 * part of the OpenAPI specification file, and cannot be added via the CDK.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @stability stable
 * @resource AWS::ApiGateway::RestApi
 */
export declare class SpecRestApi extends RestApiBase {
    /**
     * The ID of this API Gateway RestApi.
     *
     * @stability stable
     */
    readonly restApiId: string;
    /**
     * The resource ID of the root resource.
     *
     * @stability stable
     * @attribute true
     */
    readonly restApiRootResourceId: string;
    /**
     * Represents the root resource of this API endpoint ('/').
     *
     * Resources and Methods are added to this resource.
     *
     * @stability stable
     */
    readonly root: IResource;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: SpecRestApiProps);
}
/**
 * Attributes that can be specified when importing a RestApi.
 *
 * @stability stable
 */
export interface RestApiAttributes {
    /**
     * The ID of the API Gateway RestApi.
     *
     * @stability stable
     */
    readonly restApiId: string;
    /**
     * The resource ID of the root resource.
     *
     * @stability stable
     */
    readonly rootResourceId: string;
}
/**
 * Represents a REST API in Amazon API Gateway.
 *
 * Use `addResource` and `addMethod` to configure the API model.
 *
 * By default, the API will automatically be deployed and accessible from a
 * public endpoint.
 *
 * @stability stable
 */
export declare class RestApi extends RestApiBase {
    /**
     * Import an existing RestApi.
     *
     * @stability stable
     */
    static fromRestApiId(scope: Construct, id: string, restApiId: string): IRestApi;
    /**
     * Import an existing RestApi that can be configured with additional Methods and Resources.
     *
     * @stability stable
     */
    static fromRestApiAttributes(scope: Construct, id: string, attrs: RestApiAttributes): IRestApi;
    /**
     * The ID of this API Gateway RestApi.
     *
     * @stability stable
     */
    readonly restApiId: string;
    /**
     * Represents the root resource of this API endpoint ('/').
     *
     * Resources and Methods are added to this resource.
     *
     * @stability stable
     */
    readonly root: IResource;
    /**
     * The resource ID of the root resource.
     *
     * @stability stable
     */
    readonly restApiRootResourceId: string;
    /**
     * The list of methods bound to this RestApi.
     *
     * @stability stable
     */
    readonly methods: Method[];
    /**
     * This list of deployments bound to this RestApi
     */
    private readonly deployments;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: RestApiProps);
    /**
     * The deployed root URL of this REST API.
     *
     * @stability stable
     */
    get url(): string;
    /**
     * Adds a new model.
     *
     * @stability stable
     */
    addModel(id: string, props: ModelOptions): Model;
    /**
     * Adds a new request validator.
     *
     * @stability stable
     */
    addRequestValidator(id: string, props: RequestValidatorOptions): RequestValidator;
    /**
     * Internal API used by `Method` to keep an inventory of methods at the API
     * level for validation purposes.
     *
     * @internal
     */
    _attachMethod(method: Method): void;
    /**
     * Attaches a deployment to this REST API.
     *
     * @internal
     */
    _attachDeployment(deployment: Deployment): void;
    /**
     * Performs validation of the REST API.
     *
     * @stability stable
     */
    protected validate(): string[];
}
/**
 * The endpoint configuration of a REST API, including VPCs and endpoint types.
 *
 * EndpointConfiguration is a property of the AWS::ApiGateway::RestApi resource.
 *
 * @stability stable
 */
export interface EndpointConfiguration {
    /**
     * A list of endpoint types of an API or its custom domain name.
     *
     * @default EndpointType.EDGE
     * @stability stable
     */
    readonly types: EndpointType[];
    /**
     * A list of VPC Endpoints against which to create Route53 ALIASes.
     *
     * @default - no ALIASes are created for the endpoint.
     * @stability stable
     */
    readonly vpcEndpoints?: IVpcEndpoint[];
}
/**
 * @stability stable
 */
export declare enum ApiKeySourceType {
    /**
     * To read the API key from the `X-API-Key` header of a request.
     *
     * @stability stable
     */
    HEADER = "HEADER",
    /**
     * To read the API key from the `UsageIdentifierKey` from a custom authorizer.
     *
     * @stability stable
     */
    AUTHORIZER = "AUTHORIZER"
}
/**
 * @stability stable
 */
export declare enum EndpointType {
    /**
     * For an edge-optimized API and its custom domain name.
     *
     * @stability stable
     */
    EDGE = "EDGE",
    /**
     * For a regional API and its custom domain name.
     *
     * @stability stable
     */
    REGIONAL = "REGIONAL",
    /**
     * For a private API and its custom domain name.
     *
     * @stability stable
     */
    PRIVATE = "PRIVATE"
}
