/**
 * Implementation of the classifier
 * for the Cambioni et al. (2019) model.
 *
 * Copyright (c) 2019 Arizona Board of Regents
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @file
 * @author Alexandre Emsenhuber
 * @author Saverio Cambioni
 */

/* Include Files */

#include <stdlib.h>

#include "collision_classifier.h"

/**
 * Classifier of collision outcome for a giant impact.
 *
 * @param X predictors. Array of 4 values:
 *          - log10( mass of the target / Earth mass )
 *          - ratio mass projectile / mass target
 *          - impact angle [degree]
 *          - ration impact velocity / escape velocity
 * @param label Collision label:
 *          - 1 => Hit-and-run
 *          - 0 => Accretion
 *          - -1 => Disruption
 * @param score SVM scores for each class
 */
void collision_classifier( const double X[4], int* label, double score[3] ) {
	static const double c0[ 20 ] = {
		-0.99999999999999911, -0.3748588133292457, -0.99999999999999911, -0.99999999999999911, -0.23515929829608595, -0.21885102306190557, -0.61832588942015076, -0.99999999999999911, -0.59944312696661817, -0.99999999999999911,
		-0.99999999999999911,  0.88651421931625418, 0.44946956985132969,  0.99999999999999911,  0.99999999999999911,  0.99999999999999911,  0.99999999999999911,  0.99999999999999911,  0.71065436190642228,  0.99999999999999911
	};

	static const double d0[ 80 ] = {
		-0.96812029973325509,   -0.41005860951897372,   -0.94467388146643227,    0.53583033933168323, -0.96812029973325509,   0.989415275665051,   -0.94467388146643227,    0.19701007863568945, -0.0029336978779782525, -0.68995338655577865,
		-0.94467388146643227,    0.53583033933168323,   -0.0029336978779782525, -0.68995338655577865, -0.34149690721110715,   0.53583033933168323, -0.0029336978779782525, -0.68995338655577865,  0.062416245192012422,   0.19701007863568945,
		-0.0029336978779782525, -0.68995338655577865,    1.4491847351093896,    -0.48063044275629813, -0.0029336978779782525, 0.989415275665051,   -0.94467388146643227,    0.19701007863568945, -0.0029336978779782525,  0.989415275665051,
		-0.34149690721110715,    1.2134708607236708,     0.96225290397729857,   -0.41005860951897372, -0.94467388146643227,   0.19701007863568945,  0.96225290397729857,   -0.41005860951897372, -0.54345348341266686,    0.19701007863568945,
		 0.96225290397729857,    0.989415275665051,     -0.94467388146643227,    0.19701007863568945, -0.96812029973325509,  -0.41005860951897372, -0.94467388146643227,    0.87465060002767692, -0.96812029973325509,    0.989415275665051,
		-0.94467388146643227,    0.87465060002767692,   -0.0029336978779782525, -0.68995338655577865, -0.13954033100954735,   0.53583033933168323, -0.0029336978779782525, -0.68995338655577865,  0.062416245192012422,   0.53583033933168323,
		-0.0029336978779782525,  0.0097835560362337145, -0.94467388146643227,    0.53583033933168323, -0.0029336978779782525, 0.989415275665051,   -0.94467388146643227,    0.53583033933168323,  0.96225290397729857,   -0.41005860951897372,
		-0.94467388146643227,    0.53583033933168323,    0.96225290397729857,   -0.41005860951897372, -0.13954033100954735,   0.53583033933168323,  0.96225290397729857,    0.989415275665051,   -0.94467388146643227,    0.53583033933168323
	};

	static const double c1[ 46 ] = {
		-1.000000000000002,   -0.76670342934144986, -1.000000000000002,   -1.000000000000002,     -0.70344151259629351,  -0.018471782343521263, -0.080110544146148446, -1.000000000000002,   -0.078907221146861134, -0.60814908181881111,
		-1.000000000000002,   -1.000000000000002,   -1.000000000000002,   -1.000000000000002,     -1.000000000000002,    -0.23916359608618118,  -1.000000000000002,    -1.000000000000002,   -0.11648904039659738,  -1.000000000000002,
		-1.000000000000002,   -1.000000000000002,   -0.276775816564861,   -1.000000000000002,     -0.011912253422085212,  0.46772353672116118,   0.99999999999999956,   0.99999999999999956,  0.99999999999999956,   0.99999999999999956,
		 0.99999999999999956,  0.99999999999999956,  0.85223145726220983,  0.99999999999999956,    0.99999999999999956,   0.99999999999999956,   0.99999999999999956,   0.99999999999999956,  0.99999999999999956,   0.99999999999999956,
		 0.28919147275244123,  0.99999999999999956,  0.99999999999999956,  0.0055619032835237184,  0.28541590784350507,   0.99999999999999956
	};

  static const double d1[ 184 ] = {
		-2.13440421594161, -1.1856779806440236, -2.2745417110389918, 1.0363299530020105, -2.13440421594161, -1.1856779806440236, -1.7997405823943717, 1.0363299530020105, -2.13440421594161, -1.1856779806440236,
		-0.37533719646051239, 2.5066981135857094, -2.13440421594161, 1.9010899712947762, -3.21781328661297, 1.77151403329386, -2.13440421594161, 1.9010899712947762, -1.7997405823943717, 2.5066981135857094,
		-2.13440421594161, 1.9010899712947762, 1.5238673181179667, -1.7573695521070178, 2.8435976452744592E-15, -1.8030315710317839, -0.85013832510513221, -1.0221854718151682, 2.8435976452744592E-15, -1.8030315710317839,
		-0.37533719646051239, 0.30114587271016091, 2.8435976452744592E-15, -1.8030315710317839, 0.57426506082872719, -1.7573695521070178, 2.8435976452744592E-15, -1.8030315710317839, 0.57426506082872719, 2.5066981135857094,
		2.8435976452744592E-15, -1.1856779806440236, -2.2745417110389918, 1.0363299530020105, 2.8435976452744592E-15, -1.1856779806440236, -1.7997405823943717, 0.30114587271016091, 2.8435976452744592E-15, -1.1856779806440236,
		-1.7997405823943717, 1.0363299530020105, 2.8435976452744592E-15, -1.1856779806440236, -1.3249394537497519, 1.0363299530020105, 2.8435976452744592E-15, -1.1856779806440236, -0.37533719646051239, 2.5066981135857094,
		2.8435976452744592E-15, -0.25964759506238377, -2.2745417110389918, 1.0363299530020105, 2.8435976452744592E-15, -0.25964759506238377, -1.7997405823943717, 1.77151403329386, 2.8435976452744592E-15, 1.9010899712947762,
		-3.21781328661297, 1.77151403329386, 2.8435976452744592E-15, 1.9010899712947762, -2.2745417110389918, 1.0363299530020105, 2.8435976452744592E-15, 1.9010899712947762, -2.2745417110389918, 1.77151403329386,
		2.1344042159416157, -1.1856779806440236, -1.7997405823943717, 0.30114587271016091, 2.1344042159416157, -1.1856779806440236, -0.85013832510513221, 1.77151403329386, 2.1344042159416157, -1.1856779806440236,
		-0.37533719646051239, 2.5066981135857094, 2.1344042159416157, 1.9010899712947762, -2.2745417110389918, 1.0363299530020105, 2.1344042159416157, 1.9010899712947762, -0.37533719646051239, -1.6103327360486479,
		-2.13440421594161, -1.1856779806440236, -1.7997405823943717, 1.77151403329386, -2.13440421594161, -1.1856779806440236, -1.3249394537497519, 1.77151403329386, -2.13440421594161, -1.1856779806440236,
		-0.85013832510513221, 2.5066981135857094, -2.13440421594161, 1.9010899712947762, -3.21781328661297, 1.0363299530020105, -2.13440421594161, 1.9010899712947762, -2.2745417110389918, 2.5066981135857094,
		2.8435976452744592E-15, -1.8030315710317839, -1.3249394537497519, 0.30114587271016091, 2.8435976452744592E-15, -1.8030315710317839, -0.85013832510513221, 0.30114587271016091, 2.8435976452744592E-15, -1.8030315710317839,
		-0.85013832510513221, 1.0363299530020105, 2.8435976452744592E-15, -1.8030315710317839, -0.37533719646051239, 1.0363299530020105, 2.8435976452744592E-15, -1.1856779806440236, -1.3249394537497519, 1.77151403329386,
		2.8435976452744592E-15, -1.1856779806440236, -0.85013832510513221, 1.77151403329386, 2.8435976452744592E-15, -0.25964759506238377, -3.21781328661297, 0.30114587271016091, 2.8435976452744592E-15, -0.25964759506238377,
		-2.2745417110389918, 1.77151403329386, 2.8435976452744592E-15, -0.25964759506238377, -1.3249394537497519, 2.5066981135857094, 2.8435976452744592E-15, 1.9010899712947762, -3.21781328661297, 0.30114587271016091,
		2.8435976452744592E-15, 1.9010899712947762, -3.21781328661297, 1.0363299530020105, 2.1344042159416157, -1.1856779806440236, -1.3249394537497519, 0.30114587271016091, 2.1344042159416157, -1.1856779806440236,
		-1.3249394537497519, 1.0363299530020105, 2.1344042159416157, 1.9010899712947762, -3.21781328661297, 0.30114587271016091, 2.1344042159416157, 1.9010899712947762, -2.2745417110389918, 2.5066981135857094,
		2.1344042159416157, 1.9010899712947762, -1.7997405823943717, 2.5066981135857094
	};

	static const double c2[ 136 ] = {
		-1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,   -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,
		-1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -0.619643797572181,    -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,
		-1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,   -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,
		-1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,   -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,
		-0.48152498010844691, -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,   -1.0000000000000018,  -0.2854038914275413,  -0.8306810612092973,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,
		-1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -0.072582757747631843, -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -0.81360251156629881, -1.0000000000000018,  -1.0000000000000018,
		-1.0000000000000018,  -0.21445097628068693, -1.0000000000000018,  -1.0000000000000018,   -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,  -1.0000000000000018,   0.74719315629539418,  0.99999999999999944,
		 0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.66951450230080178,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.19604255721711347,
		 0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,
		 0.99999999999999944,  0.59789874476577376,  0.99999999999999944,  0.99999999999999944,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,
		 0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,
		 0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.32332351039587859,  0.99999999999999944,
		 0.99999999999999944,  0.99999999999999944,  0.98752629370117551,  0.99999999999999944,   0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,  0.99999999999999944,
		 0.52707858329864354,  0.99999999999999944,  0.99999999999999944,  0.26931262793741517,   0.99999999999999944,  0.99999999999999944
	};

	static const double d2[ 544 ] = {
		-4.7673073854411134, -2.7399058149021052, -1.7265799742132044, -1.0064218285397697, -4.7673073854411134, -2.7399058149021052, -0.64803792665534987, -1.793050384180074, -4.7673073854411134, -2.7399058149021052,
		0.430504120902505, -2.1863646620002264, -4.7673073854411134, -2.7399058149021052, 1.5090461684603598, -2.7763360787304547, -4.7673073854411134, -2.7399058149021052, 1.5090461684603598, -2.3830218009103032,
		-4.7673073854411134, -2.7399058149021052, 4.7446723111339244, -2.5796789398203783, -4.7673073854411134, -2.7399058149021052, 5.7513115555212551, -2.5796789398203783, -4.7673073854411134, 4.2810349087266912,
		-7.1049096513683736, 6.8598637278632753, -4.7673073854411134, 4.2810349087266912, -2.8051220217710595, -1.0064218285397697, -4.7673073854411134, 4.2810349087266912, -1.7265799742132044, -1.3997361063599223,
		-4.7673073854411134, 4.2810349087266912, -0.64803792665534987, -1.793050384180074, -4.7673073854411134, 4.2810349087266912, 0.430504120902505, -2.1863646620002264, -4.7673073854411134, 4.2810349087266912,
		3.6661302635760693, -2.5796789398203783, -4.7673073854411134, 4.2810349087266912, 5.7513115555212551, -2.3830218009103032, 0.067355991317064087, -4.144093959627865, -0.64803792665534987, -1.793050384180074,
		0.067355991317064087, -4.144093959627865, 0.430504120902505, -2.3830218009103032, 0.067355991317064087, -4.144093959627865, 0.430504120902505, -2.1863646620002264, 0.067355991317064087, -4.144093959627865,
		1.5090461684603598, -2.5796789398203783, 0.067355991317064087, -4.144093959627865, 1.5090461684603598, -2.3830218009103032, 0.067355991317064087, -4.144093959627865, 3.6661302635760693, -2.7763360787304547,
		0.067355991317064087, -4.144093959627865, 4.7446723111339244, -2.7763360787304547, 0.067355991317064087, -4.144093959627865, 5.7513115555212551, -2.7763360787304547, 0.067355991317064087, -4.144093959627865,
		5.7513115555212551, -2.5796789398203783, 0.067355991317064087, -2.7399058149021052, -4.9622061168867688, 4.8932923387625147, 0.067355991317064087, -2.7399058149021052, -1.7265799742132044, -1.3997361063599223,
		0.067355991317064087, -2.7399058149021052, -0.64803792665534987, -1.793050384180074, 0.067355991317064087, -2.7399058149021052, 0.430504120902505, -2.3830218009103032, 0.067355991317064087, -2.7399058149021052,
		0.430504120902505, -2.1863646620002264, 0.067355991317064087, -2.7399058149021052, 1.5090461684603598, -2.5796789398203783, 0.067355991317064087, -2.7399058149021052, 1.5090461684603598, -2.3830218009103032,
		0.067355991317064087, -2.7399058149021052, 5.7513115555212551, -2.5796789398203783, 0.067355991317064087, -0.63362359781346644, -2.8051220217710595, -0.6131075507196172, 0.067355991317064087, -0.63362359781346644,
		-1.7265799742132044, -1.3997361063599223, 0.067355991317064087, -0.63362359781346644, -0.64803792665534987, -1.793050384180074, 0.067355991317064087, -0.63362359781346644, 0.430504120902505, -2.3830218009103032,
		0.067355991317064087, -0.63362359781346644, 0.430504120902505, -2.1863646620002264, 0.067355991317064087, -0.63362359781346644, 1.5090461684603598, -2.3830218009103032, 0.067355991317064087, -0.63362359781346644,
		3.6661302635760693, -2.5796789398203783, 0.067355991317064087, -0.63362359781346644, 4.7446723111339244, -2.5796789398203783, 0.067355991317064087, -0.63362359781346644, 5.7513115555212551, -2.5796789398203783,
		0.067355991317064087, -0.63362359781346644, 5.7513115555212551, -2.3830218009103032, 0.067355991317064087, 4.2810349087266912, -7.1049096513683736, 6.8598637278632753, 0.067355991317064087, 4.2810349087266912,
		-3.8836640693289142, -0.21979327289946551, 0.067355991317064087, 4.2810349087266912, -2.8051220217710595, -1.0064218285397697, 0.067355991317064087, 4.2810349087266912, -1.7265799742132044, -1.793050384180074,
		0.067355991317064087, 4.2810349087266912, 0.430504120902505, -2.1863646620002264, 0.067355991317064087, 4.2810349087266912, 1.5090461684603598, -2.3830218009103032, 0.067355991317064087, 4.2810349087266912,
		3.6661302635760693, -2.5796789398203783, 0.067355991317064087, 4.2810349087266912, 4.7446723111339244, -2.5796789398203783, 0.067355991317064087, 4.2810349087266912, 5.7513115555212551, -2.5796789398203783,
		4.9020193680752406, -2.7399058149021052, -1.7265799742132044, -1.3997361063599223, 4.9020193680752406, -2.7399058149021052, -0.64803792665534987, -2.1863646620002264, 4.9020193680752406, -2.7399058149021052,
		-0.64803792665534987, -1.793050384180074, 4.9020193680752406, -2.7399058149021052, -0.64803792665534987, 8.8264351169640367, 4.9020193680752406, -2.7399058149021052, 0.430504120902505, -2.3830218009103032,
		4.9020193680752406, -2.7399058149021052, 0.430504120902505, -2.1863646620002264, 4.9020193680752406, -2.7399058149021052, 1.5090461684603598, -2.5796789398203783, 4.9020193680752406, -2.7399058149021052,
		3.6661302635760693, -2.5796789398203783, 4.9020193680752406, -2.7399058149021052, 4.7446723111339244, -2.5796789398203783, 4.9020193680752406, -2.7399058149021052, 5.7513115555212551, -2.5796789398203783,
		4.9020193680752406, -2.7399058149021052, 5.7513115555212551, -2.3830218009103032, 4.9020193680752406, 4.2810349087266912, -2.8051220217710595, -1.0064218285397697, 4.9020193680752406, 4.2810349087266912,
		-1.7265799742132044, -1.793050384180074, 4.9020193680752406, 4.2810349087266912, -0.64803792665534987, -2.1863646620002264, 4.9020193680752406, 4.2810349087266912, 0.430504120902505, -2.3830218009103032,
		4.9020193680752406, 4.2810349087266912, 3.6661302635760693, -2.5796789398203783, 4.9020193680752406, 4.2810349087266912, 4.7446723111339244, -2.5796789398203783, 4.9020193680752406, 4.2810349087266912,
		5.7513115555212551, -2.3830218009103032, -4.7673073854411134, -2.7399058149021052, -3.8836640693289142, 0.17352100492068709, -4.7673073854411134, -2.7399058149021052, -1.7265799742132044, -1.3997361063599223,
		-4.7673073854411134, -2.7399058149021052, 0.430504120902505, -2.3830218009103032, -4.7673073854411134, -2.7399058149021052, 1.5090461684603598, -2.1863646620002264, -4.7673073854411134, -2.7399058149021052,
		3.6661302635760693, -2.7763360787304547, -4.7673073854411134, -2.7399058149021052, 4.7446723111339244, -2.9729932176405311, -4.7673073854411134, -2.7399058149021052, 4.7446723111339244, -2.7763360787304547,
		-4.7673073854411134, -2.7399058149021052, 5.7513115555212551, -2.7763360787304547, -4.7673073854411134, 4.2810349087266912, -4.9622061168867688, 0.96014956056099143, -4.7673073854411134, 4.2810349087266912,
		-3.8836640693289142, -0.6131075507196172, -4.7673073854411134, 4.2810349087266912, -3.8836640693289142, -0.21979327289946551, -4.7673073854411134, 4.2810349087266912, -1.7265799742132044, -1.793050384180074,
		-4.7673073854411134, 4.2810349087266912, 0.430504120902505, -2.3830218009103032, -4.7673073854411134, 4.2810349087266912, 1.5090461684603598, -2.5796789398203783, -4.7673073854411134, 4.2810349087266912,
		4.7446723111339244, -2.7763360787304547, 0.067355991317064087, -4.144093959627865, -3.8836640693289142, 2.9267209496617528, 0.067355991317064087, -4.144093959627865, -2.8051220217710595, -0.6131075507196172,
		0.067355991317064087, -4.144093959627865, -1.7265799742132044, -1.3997361063599223, 0.067355991317064087, -4.144093959627865, -1.7265799742132044, 0.96014956056099143, 0.067355991317064087, -4.144093959627865,
		0.430504120902505, -2.5796789398203783, 0.067355991317064087, -4.144093959627865, 1.5090461684603598, -2.7763360787304547, 0.067355991317064087, -4.144093959627865, 3.6661302635760693, -2.9729932176405311,
		0.067355991317064087, -4.144093959627865, 4.7446723111339244, -2.9729932176405311, 0.067355991317064087, -4.144093959627865, 5.6794087523507324, -2.9729932176405311, 0.067355991317064087, -2.7399058149021052,
		-2.8051220217710595, -0.6131075507196172, 0.067355991317064087, -2.7399058149021052, -0.64803792665534987, -2.1863646620002264, 0.067355991317064087, -2.7399058149021052, 1.5090461684603598, -2.7763360787304547,
		0.067355991317064087, -2.7399058149021052, 3.6661302635760693, -2.9729932176405311, 0.067355991317064087, -2.7399058149021052, 3.6661302635760693, -2.7763360787304547, 0.067355991317064087, -2.7399058149021052,
		4.7446723111339244, -2.9729932176405311, 0.067355991317064087, -2.7399058149021052, 4.7446723111339244, -2.7763360787304547, 0.067355991317064087, -2.7399058149021052, 5.7513115555212551, -2.7763360787304547,
		0.067355991317064087, -0.63362359781346644, -3.8836640693289142, 0.17352100492068709, 0.067355991317064087, -0.63362359781346644, -2.8051220217710595, -1.0064218285397697, 0.067355991317064087, -0.63362359781346644,
		-0.64803792665534987, -2.1863646620002264, 0.067355991317064087, -0.63362359781346644, 1.5090461684603598, -2.7763360787304547, 0.067355991317064087, -0.63362359781346644, 1.5090461684603598, -2.5796789398203783,
		0.067355991317064087, -0.63362359781346644, 3.6661302635760693, -2.9729932176405311, 0.067355991317064087, -0.63362359781346644, 3.6661302635760693, -2.7763360787304547, 0.067355991317064087, -0.63362359781346644,
		4.7446723111339244, -2.7763360787304547, 0.067355991317064087, -0.63362359781346644, 5.7513115555212551, -2.7763360787304547, 0.067355991317064087, 4.2810349087266912, -4.9622061168867688, 0.96014956056099143,
		0.067355991317064087, 4.2810349087266912, -3.8836640693289142, -0.6131075507196172, 0.067355991317064087, 4.2810349087266912, -3.8836640693289142, 6.8598637278632753, 0.067355991317064087, 4.2810349087266912,
		-1.7265799742132044, 4.8932923387625147, 0.067355991317064087, 4.2810349087266912, 1.5090461684603598, -2.7763360787304547, 0.067355991317064087, 4.2810349087266912, 1.5090461684603598, -2.5796789398203783,
		0.067355991317064087, 4.2810349087266912, 1.5090461684603598, -0.21979327289946551, 0.067355991317064087, 4.2810349087266912, 3.6661302635760693, -2.7763360787304547, 0.067355991317064087, 4.2810349087266912,
		4.7446723111339244, -2.7763360787304547, 4.9020193680752406, -2.7399058149021052, -2.8051220217710595, -1.0064218285397697, 4.9020193680752406, -2.7399058149021052, -2.8051220217710595, -0.6131075507196172,
		4.9020193680752406, -2.7399058149021052, -1.7265799742132044, -1.793050384180074, 4.9020193680752406, -2.7399058149021052, 0.430504120902505, -2.5796789398203783, 4.9020193680752406, -2.7399058149021052,
		1.5090461684603598, -2.9729932176405311, 4.9020193680752406, -2.7399058149021052, 1.5090461684603598, -2.7763360787304547, 4.9020193680752406, -2.7399058149021052, 3.6661302635760693, -2.9729932176405311,
		4.9020193680752406, -2.7399058149021052, 3.6661302635760693, -2.7763360787304547, 4.9020193680752406, -2.7399058149021052, 4.7446723111339244, -2.9729932176405311, 4.9020193680752406, -2.7399058149021052,
		4.7446723111339244, -2.7763360787304547, 4.9020193680752406, -2.7399058149021052, 5.7513115555212551, -2.7763360787304547, 4.9020193680752406, 4.2810349087266912, -0.64803792665534987, -2.3830218009103032,
		4.9020193680752406, 4.2810349087266912, 0.430504120902505, -2.7763360787304547, 4.9020193680752406, 4.2810349087266912, 0.430504120902505, -2.5796789398203783, 4.9020193680752406, 4.2810349087266912,
		1.5090461684603598, -2.7763360787304547, 4.9020193680752406, 4.2810349087266912, 3.6661302635760693, -2.9729932176405311, 4.9020193680752406, 4.2810349087266912, 3.6661302635760693, -2.7763360787304547,
		4.9020193680752406, 4.2810349087266912, 4.7446723111339244, -2.7763360787304547
	};

	const double* ci[ 3 ] = { c0, c1, c2 };
	const double* di[ 3 ] = { d0, d1, d2 };

	const double oi[ 3 ][ 4 ] = {
		{ -0.99696048632218981, 0.34650455927051926, 35.182066869301195, 1.7092705167173332 },
		{ -1.0000000000000013, 0.39205816554809358, 50.928859060402196, 2.2951901565995461 },
		{ -1.0139318885448916, 0.39512383900928133, 49.506346749225287, 1.7558823529411685 }
	};

	const double mi[ 3 ][ 4 ] = {
		{ 0.68297448548548256, 0.23551630001045257, 24.480429228819204, 0.97278394958655334 },
		{ 0.71184721492684289, 0.2461101255913701, 24.000096118927456, 1.0333260317898867 },
		{ 0.7097052701984341, 0.24435372788243789, 23.859890900103604, 0.87237770699226624 }
	};

	const int li[ 3 ] = {
		20,
		46,
		136
	};

	const double ai[ 3 ] = {
		1.5169953015819087,
		0.65816766137128868,
		0.29144440937686777
	};

	const double bi[ 3 ] = {
		-1.6512373495903629,
		-6.3007673496917675,
		-7.548107436699695
	};

	static const signed char iv[ 9 ] = { 1, -1, 0, 1, 0, -1, 0, 1, -1 };
	static const double def[ 3 ] = { 0.091420534458509145, 0.37130801687763715, 0.53727144866385368 };

	double pscore[ 3 ];

	for ( int n = 0; n < 3; n++ ) {
		double mu[ 4 ];

		for ( int i = 0; i < 4; i++ ) {
			mu[ i ] = ( X[ i ] - oi[ n ][ i ] ) / ( mi[ n ][ i ] * ai[ n ] );
		}

		pscore[ n ] = 0.;

		for ( int i = 0; i < li[ n ]; i++ ) {
			double f = 0.;
			for ( int j = 0; j < 4; j++ ) {
				f += mu[ j ] * di[ n ][ j + 4 * i ];
			}

			pscore[ n ] += ( f + 1. ) * ( f + 1. ) * ci[ n ][ i ];
		}

		pscore[ n ] += bi[ n ];
	}

	/* Compute the score for each class */

	int good = 0; /* Flag indicating if at least one value is not NaN */
	for ( int i = 0; i < 3; i++ ) {
		double tot = 0.;
		for ( int n = 0; n < 3; n++ ) {
			if ( iv[ i + 3 * n ] != 0 ) {
				double val = 1. - (double)( iv[ i + 3 * n ] ) * pscore[ n ];
				if ( val > 0. ) tot += val;
			}
		}

		score[ i ] = -tot / 6.;

		if ( score[ i ] == score[ i ] ) {
			good = 1;
		}
	}

	/* Find the index of the largest value in "search" */

	const double* search = good ? score : def;

	double val = strtod( "-inf", NULL );
	*label = 0; /* Default index if all values are NaN, which should not happen anyway */

	for ( int i = 0; i < 3; i++ ) {
		if ( search[ i ] > val ) {
			val = search[ i ];
			*label = i + 1;
		}
	}

	*label = *label - 2;
}
