"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputValidator = void 0;
const cdk = require("@aws-cdk/core");
/**
 * Class to validate that inputs match requirements.
 */
class InputValidator {
    /**
     * Validates length is between allowed min and max lengths.
     */
    static validateLength(resourceName, inputName, minLength, maxLength, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && (inputString.length < minLength || inputString.length > maxLength)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must have length between ${minLength} and ${maxLength}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the allowed regex pattern.
     */
    static validateRegex(resourceName, inputName, regexp, inputString) {
        if (!cdk.Token.isUnresolved(inputString) && inputString !== undefined && !regexp.test(inputString)) {
            throw new Error(`Invalid ${inputName} for resource ${resourceName}, must match regex pattern ${regexp}, got: '${this.truncateString(inputString, 100)}'`);
        }
    }
    /**
     * Validates string matches the valid URL regex pattern.
     */
    static validateUrl(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^https?:\/\/.*/, inputString);
    }
    /**
    * Validates string matches the valid email regex pattern.
    */
    static validateEmail(resourceName, inputName, inputString) {
        this.validateRegex(resourceName, inputName, /^[\w\d.%+\-]+@[a-z\d.\-]+\.[a-z]{2,4}$/i, inputString);
    }
    static truncateString(string, maxLength) {
        if (string.length > maxLength) {
            return string.substring(0, maxLength) + '[truncated]';
        }
        return string;
    }
}
exports.InputValidator = InputValidator;
//# sourceMappingURL=data:application/json;base64,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