"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Channel = exports.ChannelType = exports.LatencyMode = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core = require("aws-cdk-lib");
const aws_ivs_1 = require("aws-cdk-lib/aws-ivs");
const stream_key_1 = require("./stream-key");
/**
 * Reference to a new or existing IVS Channel
 */
class ChannelBase extends core.Resource {
    addStreamKey(id) {
        return new stream_key_1.StreamKey(this, id, {
            channel: this,
        });
    }
}
/**
  Channel latency mode
*/
var LatencyMode;
(function (LatencyMode) {
    /**
     * Use LOW to minimize broadcaster-to-viewer latency for interactive broadcasts.
     */
    LatencyMode["LOW"] = "LOW";
    /**
     * Use NORMAL for broadcasts that do not require viewer interaction.
     */
    LatencyMode["NORMAL"] = "NORMAL";
})(LatencyMode = exports.LatencyMode || (exports.LatencyMode = {}));
/**
  * The channel type, which determines the allowable resolution and bitrate.
  * If you exceed the allowable resolution or bitrate, the stream probably will disconnect immediately.
*/
var ChannelType;
(function (ChannelType) {
    /**
     * Multiple qualities are generated from the original input, to automatically give viewers the best experience for
     * their devices and network conditions.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ivs-channel.html
     */
    ChannelType["STANDARD"] = "STANDARD";
    /**
     * delivers the original input to viewers. The viewer’s video-quality choice is limited to the original input.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ivs-channel.html
     */
    ChannelType["BASIC"] = "BASIC";
})(ChannelType = exports.ChannelType || (exports.ChannelType = {}));
/**
  A new IVS channel
*/
class Channel extends ChannelBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.name,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ivs_alpha_ChannelProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Channel);
            }
            throw error;
        }
        if (props.name && !core.Token.isUnresolved(props.name) && !/^[a-zA-Z0-9-_]*$/.test(props.name)) {
            throw new Error(`name must contain only numbers, letters, hyphens and underscores, got: '${props.name}'`);
        }
        const resource = new aws_ivs_1.CfnChannel(this, 'Resource', {
            authorized: props.authorized,
            latencyMode: props.latencyMode,
            name: props.name,
            type: props.type,
        });
        this.channelArn = resource.attrArn;
        this.channelIngestEndpoint = resource.attrIngestEndpoint;
        this.channelPlaybackUrl = resource.attrPlaybackUrl;
    }
    /**
     * Import an existing channel
     */
    static fromChannelArn(scope, id, channelArn) {
        // This will throw an error if the arn cannot be parsed
        let arnComponents = core.Arn.split(channelArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!core.Token.isUnresolved(arnComponents.service) && arnComponents.service !== 'ivs') {
            throw new Error(`Invalid service, expected 'ivs', got '${arnComponents.service}'`);
        }
        if (!core.Token.isUnresolved(arnComponents.resource) && arnComponents.resource !== 'channel') {
            throw new Error(`Invalid resource, expected 'channel', got '${arnComponents.resource}'`);
        }
        class Import extends ChannelBase {
            constructor() {
                super(...arguments);
                this.channelArn = channelArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Channel = Channel;
_a = JSII_RTTI_SYMBOL_1;
Channel[_a] = { fqn: "@aws-cdk/aws-ivs-alpha.Channel", version: "2.70.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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