from dataclasses import InitVar, dataclass
from typing import ClassVar, Dict, List, Optional

from avionix.kube.core import EnvVar
from cryptography.fernet import Fernet


def _create_fernet_key():
    return Fernet.generate_key().decode("utf-8")


@dataclass
class SmtpNotificationOptions:
    """
    Class for setting airflow email options. These options will be set in the
    environment and also in the kubernetes pod environment to ensure the email
    configuration is available on all pods.

    :param smtp_host: Host name for the smtp server
    :param smtp_starttls: Use tls if true
    :param smtp_ssl: Use ssl if true
    :param smtp_user: Email user name
    :param smtp_password: Email password
    :param smtp_port: Port to use
    :param smtp_mail_from: Email to send email notification from
    """

    smtp_host: str
    smtp_starttls: bool
    smtp_ssl: bool
    smtp_user: str
    smtp_password: str
    smtp_port: int
    smtp_mail_from: str
    _prefixes: ClassVar[List[str]] = [
        "AIRFLOW__SMTP__",
    ]

    def to_dict(self):
        added_env_vars = {}
        for prefix in self._prefixes:
            for key in self.__dict__:
                if "password" not in key:
                    added_env_vars[f"{prefix}{key.upper()}"] = str(self.__dict__[key])
        return added_env_vars


@dataclass
class AirflowOptions:
    """
    Class for storing airflow options. Note that for storage specification the
    default unit is bytes, and you can use Mi or Gi to specify larger units.

    :param dag_sync_image: The image to use for syncing dags
    :param dag_sync_command: The commands to use for syncing dags
    :param dag_sync_schedule: Cron format schedule for how often to sync dags
    :param dag_storage: How much persistent storage to allocate for dags
    :param logs_storage: How much persistent storage to allocate for dags
    :param external_storage: How much external storage to allocate for dags
    :param default_executor_cpu: Default CPU to allocate to the executor \
        This is specified in number of vCPUs. You can fractions of CPUs as well or \
        milliCPUs (eg. "500m" for 500 milliCPUs)
    :param default_executor_memory: Default memory to allocate to the executor \
        This can be specified in bytes (default) or using standard prefix (eg. "20Mi")
    :param access_modes: Used to set access modes on the dag and logs volumes
    :param default_timezone: The default timezone
    :param core_executor: The core executor to use, supported executors are \
        "KubernetesExecutor" and "CeleryExecutor"
    :param namespace: The kubernetes namespace to use for the installation
    :param domain_name: The domain name to place on the ingress controller
    :param additional_vars: Any additional environment variables to place on the \
        airflow nodes. Note that you can add additional airflow configuration using \
        this setting. For more info on this see the \
        `airflow config documentation <https://airflow.apache.org/docs/stable/howto
        /set-config.html>`__
    :param fernet_key: The fernet key to use for airflow, this is autogenerated by \
        default, so unless you need to use an old fernet key, just leave this
    :param dags_paused_at_creation: Whether or not dags should be paused on creation
    :param worker_image: The docker image to use as the airflow worker
    :param worker_image_tag: The docker tag of the airflow worker image
    :param open_node_ports: Whether to expose the node ports
    :param local_mode: Whether or not to run in local mode
    :param image_pull_policy: Policy for pulling docker image. Can be one of "Never",
        "Always", "IfNotPresent"
    :param master_image_tag: The docker tag to use for the master image
    :param delete_pods_on_failure: Whether or not to terminate pods if
        KubernetesExecutor task fails
    """

    dag_sync_image: str
    dag_sync_command: List[str]
    dag_sync_schedule: str
    dag_storage: str = "50Mi"
    log_storage: str = "50Mi"
    external_storage: str = "50Mi"
    default_executor_cpu: int = 5
    default_executor_memory: int = 2
    access_modes: InitVar[Optional[List[str]]] = None
    default_timezone: str = "utc"
    core_executor: str = "CeleryExecutor"
    namespace: str = "airflow"
    domain_name: Optional[str] = "www.avionix-airflow.com"
    additional_vars: InitVar[Optional[Dict[str, str]]] = None
    fernet_key: InitVar[str] = ""
    dags_paused_at_creation: bool = True
    worker_image: str = ""
    worker_image_tag: str = ""
    open_node_ports: bool = False
    local_mode: bool = False
    smtp_notification_options: Optional[SmtpNotificationOptions] = None
    git_ssh_key: Optional[str] = None
    image_pull_policy: str = "IfNotPresent"
    master_image: str = ""
    master_image_tag: str = ""
    delete_pods_on_failure: bool = False
    default_image: ClassVar[str] = "zachb1996/avionix_airflow"

    def __post_init__(
        self, access_modes, additional_vars, fernet_key: str,
    ):
        self.access_modes = self.__get_access_modes(access_modes)
        self.fernet_key = fernet_key if fernet_key else _create_fernet_key()
        self.worker_image = self._get_image_behavior(self.worker_image)
        self.master_image = self._get_image_behavior(self.master_image)
        self.worker_image_tag = self._get_tag(self.worker_image_tag)
        self.master_image_tag = self._get_tag(self.master_image_tag)

        self.__additional_vars = additional_vars if additional_vars is not None else {}
        if self.smtp_notification_options:
            self.__additional_vars.update(self.smtp_notification_options.to_dict())

    def _get_image_behavior(self, image: str):
        if not image and not self.local_mode:
            return self.default_image
        if not image and self.local_mode:
            return "airflow-image"
        return image

    @staticmethod
    def _get_tag(tag: str):
        if tag:
            return tag
        return "latest"

    @staticmethod
    def __get_access_modes(access_modes: Optional[List[str]]):
        if access_modes is None:
            return ["ReadWriteMany"]
        return access_modes

    @property
    def extra_env_vars(self):
        return [EnvVar(name, value) for name, value in self.__additional_vars.items()]

    @property
    def in_celery_mode(self):
        return self.core_executor == "CeleryExecutor"

    @property
    def in_kube_mode(self):
        return self.core_executor == "KubernetesExecutor"
