"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IngressV1Beta1Backend = exports.IngressV1Beta1 = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base_1 = require("./base");
const k8s = require("./imports/k8s");
/**
 * Ingress is a collection of rules that allow inbound connections to reach the endpoints defined by a backend.
 *
 * An Ingress can be configured to give services
 * externally-reachable urls, load balance traffic, terminate SSL, offer name
 * based virtual hosting etc.
 *
 * @stability stable
 */
class IngressV1Beta1 extends base_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _c;
        super(scope, id);
        this._rulesPerHost = {};
        this._tlsConfig = [];
        this.apiObject = new k8s.KubeIngressV1Beta1(this, 'Resource', {
            metadata: props.metadata,
            spec: {
                backend: cdk8s_1.Lazy.any({ produce: () => { var _c; return (_c = this._defaultBackend) === null || _c === void 0 ? void 0 : _c._toKube(); } }),
                rules: cdk8s_1.Lazy.any({ produce: () => this.synthRules() }),
                tls: cdk8s_1.Lazy.any({ produce: () => this.tlsConfig() }),
            },
        });
        if (props.defaultBackend) {
            this.addDefaultBackend(props.defaultBackend);
        }
        this.addRules(...(_c = props.rules) !== null && _c !== void 0 ? _c : []);
        if (props.tls) {
            this.addTls(props.tls);
        }
    }
    /**
     * (deprecated) Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        if (!this._defaultBackend && Object.keys(this._rulesPerHost).length === 0) {
            return ['ingress with no rules or default backend'];
        }
        return [];
    }
    /**
     * Defines the default backend for this ingress.
     *
     * A default backend capable of
     * servicing requests that don't match any rule.
     *
     * @param backend The backend to use for requests that do not match any rule.
     * @stability stable
     */
    addDefaultBackend(backend) {
        this.addRules({ backend });
    }
    /**
     * Specify a default backend for a specific host name.
     *
     * This backend will be used as a catch-all for requests
     * targeted to this host name (the `Host` header matches this value).
     *
     * @param host The host name to match.
     * @param backend The backend to route to.
     * @stability stable
     */
    addHostDefaultBackend(host, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend });
    }
    /**
     * Adds an ingress rule applied to requests to a specific host and a specific HTTP path (the `Host` header matches this value).
     *
     * @param host The host name.
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addHostRule(host, path, backend) {
        if (!host) {
            throw new Error('host must not be an empty string');
        }
        this.addRules({ host, backend, path });
    }
    /**
     * Adds an ingress rule applied to requests sent to a specific HTTP path.
     *
     * @param path The HTTP path.
     * @param backend The backend to route requests to.
     * @stability stable
     */
    addRule(path, backend) {
        this.addRules({ backend, path });
    }
    /**
     * Adds rules to this ingress.
     *
     * @param rules The rules to add.
     * @stability stable
     */
    addRules(...rules) {
        var _c, _d;
        for (const rule of rules) {
            // default backend is not really a rule
            if (!rule.host && !rule.path) {
                if (this._defaultBackend) {
                    throw new Error('a default backend is already defined for this ingress');
                }
                this._defaultBackend = rule.backend;
                continue;
            }
            const host = (_c = rule.host) !== null && _c !== void 0 ? _c : '';
            const backend = rule.backend;
            const path = rule.path;
            if (path && !path.startsWith('/')) {
                throw new Error(`ingress paths must begin with a "/": ${path}`);
            }
            const routes = this._rulesPerHost[host] = (_d = this._rulesPerHost[host]) !== null && _d !== void 0 ? _d : [];
            // check if we already have a rule for this host/path
            if (routes.find(r => r.path === path)) {
                throw new Error(`there is already an ingress rule for ${host}${path}`);
            }
            routes.push({ backend: backend._toKube(), path });
        }
    }
    synthRules() {
        const rules = new Array();
        for (const [host, paths] of Object.entries(this._rulesPerHost)) {
            rules.push({
                host: host ? host : undefined,
                http: { paths: paths.sort(sortByPath) },
            });
        }
        return rules.length > 0 ? rules : undefined;
    }
    /**
     * @stability stable
     */
    addTls(tls) {
        this._tlsConfig.push(...tls);
    }
    tlsConfig() {
        var _c;
        if (this._tlsConfig.length == 0) {
            return undefined;
        }
        const tls = new Array();
        for (const entry of this._tlsConfig) {
            tls.push({
                hosts: entry.hosts,
                secretName: (_c = entry.secret) === null || _c === void 0 ? void 0 : _c.name,
            });
        }
        return tls;
    }
}
exports.IngressV1Beta1 = IngressV1Beta1;
_a = JSII_RTTI_SYMBOL_1;
IngressV1Beta1[_a] = { fqn: "cdk8s-plus-17.IngressV1Beta1", version: "1.0.0-beta.77" };
/**
 * The backend for an ingress path.
 *
 * @stability stable
 */
class IngressV1Beta1Backend {
    constructor(backend) {
        this.backend = backend;
    }
    /**
     * A Kubernetes `Service` to use as the backend for this path.
     *
     * @param service The service object.
     * @stability stable
     */
    static fromService(service, options = {}) {
        if (service.ports.length === 0) {
            throw new Error('service does not expose any ports');
        }
        let servicePort;
        if (service.ports.length === 1) {
            servicePort = service.ports[0].port;
        }
        else {
            if (options.port !== undefined) {
                const found = service.ports.find(p => p.port === options.port);
                if (found) {
                    servicePort = found.port;
                }
                else {
                    throw new Error(`service exposes ports ${service.ports.map(p => p.port).join(',')} but backend is defined to use port ${options.port}`);
                }
            }
            else {
                throw new Error(`unable to determine service port since service exposes multiple ports: ${service.ports.map(x => x.port).join(',')}`);
            }
        }
        if (options.port !== undefined && servicePort !== options.port) {
            throw new Error(`backend defines port ${options.port} but service exposes port ${servicePort}`);
        }
        return new IngressV1Beta1Backend({
            serviceName: service.name,
            servicePort: k8s.IntOrString.fromNumber(servicePort),
        });
    }
    /**
     * @internal
     */
    _toKube() { return this.backend; }
}
exports.IngressV1Beta1Backend = IngressV1Beta1Backend;
_b = JSII_RTTI_SYMBOL_1;
IngressV1Beta1Backend[_b] = { fqn: "cdk8s-plus-17.IngressV1Beta1Backend", version: "1.0.0-beta.77" };
function sortByPath(lhs, rhs) {
    var _c, _d;
    const p1 = (_c = lhs.path) !== null && _c !== void 0 ? _c : '';
    const p2 = (_d = rhs.path) !== null && _d !== void 0 ? _d : '';
    return p1.localeCompare(p2);
}
//# sourceMappingURL=data:application/json;base64,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