# Project Introduction

<h2 align="center">
  FastAPI-Amis-Admin
</h2>
<p align="center">
    <em>fastapi-amis-admin is a fastapi management background framework with high performance, high efficiency and easy expansion</em><br/>
    <em>Inspired by Django-Admin, and has powerful functions not inferior to Django-Admin.</em>
</p>
<p align="center">
    <a href="https://github.com/amisadmin/fastapi_amis_admin/actions/workflows/pytest.yml" target="_blank">
        <img src="https://github.com/amisadmin/fastapi_amis_admin/actions/workflows/pytest.yml/badge.svg" alt="Pytest">
    </a>
    <a href="https://pypi.org/project/fastapi_amis_admin" target="_blank">
        <img src="https://badgen.net/pypi/v/fastapi-amis-admin?color=blue" alt="Package version">
    </a>
    <a href="https://pepy.tech/project/fastapi-amis-admin" target="_blank">
        <img src="https://pepy.tech/badge/fastapi-amis-admin" alt="Downloads">
    </a>
    <a href="https://gitter.im/amisadmin/fastapi-amis-admin">
        <img src="https://badges.gitter.im/amisadmin/fastapi-amis-admin.svg" alt="Chat on grid"/>
    </a>
    <a href="https://jq.qq.com/?_wv=1027&k=U4Dv6x8W" target="_blank">
        <img src="https://badgen.net/badge/qq%E7%BE%A4/229036692/orange" alt="229036692">
    </a>
</p>
<p align="center">
  <a href="https://github.com/amisadmin/fastapi_amis_admin" target="_blank">Source code</a>
  ·
  <a href="http://demo.amis.work/admin" target="_blank">Online demo</a>
  ·
  <a href="http://docs.amis.work" target="_blank">文档</a>
  ·
  <a href="http://docs.gh.amis.work" target="_blank">Can't open the document? </a>
</p>



------

`fastapi-amis-admin` is a high-performance and efficient `web-admin` framework developed based on `fastapi`+`amis`, using Python 3.7+ and based on standard Python type hints.
The original intention of `fastapi-amis-admin` is to improve the `fastapi` application ecology and quickly generate a visual management background for the `fastapi` web application.

## key features

- **High performance**: Based on [FastAPI](https://fastapi.tiangolo.com/zh/), you can enjoy all the advantages of `FastAPI`.

- **Faster efficiency**: Perfect coding type hints, code reusability is higher.

- **Support asynchronous and synchronous mixed writing**: `ORM` is based on `SQLModel`+`Sqlalchemy`, can customize the database type freely, supports synchronous and asynchronous mode, and has strong scalability.

- **Separation of front-end and back-end**: The front-end is rendered by `Amis`, the back-end interface is automatically generated by `fastapi-amis-admin`, and the interface can be reused.

- **Strong scalability**: The background page supports `Amis` page and ordinary `html` page, and developers can easily customize the interface freely.

- **Automatically generate API documentation**: The interface documentation is automatically generated by `FastAPI`, which is convenient for developers to debug and share the interface.

## core dependencies

- [FastAPI](https://fastapi.tiangolo.com) responsible for the web part
- [SQLModel](https://sqlmodel.tiangolo.com/) Responsible for ORM model mapping (
  Perfect combination of [SQLAlchemy](https://www.sqlalchemy.org/)+[Pydantic](https://pydantic-docs.helpmanual.io/), with `SQLAlchemy`
  and all features of `Pydantic`)
- [Amis](https://baidu.gitee.io/amis) Responsible for Admin background page display

## Project composition

`fastapi-amis-admin` consists of three core modules, of which `amis`, `fastapi-sqlmodel-crud` can be used as independent modules, and `amis_admin` is built based on the former.

- `amis`: `pydantic` data model building library based on `baidu amis` for fast generation/parsing of `amis` `json` data.
- `fastapi-sqlmodel-crud`: Based on `FastAPI`+`SQLModel`, it is used to quickly build Create, Read, Update, Delete common API interfaces.
- `admin`: Inspired by `Django-Admin`, combined with `amis`+`fastapi-sqlmodel-crud`, used to quickly build `Web Admin` management background.

## Install

```bash
pip install fastapi_amis_admin
```

## Simple example

```python
from fastapi import FastAPI
from fastapi_amis_admin.admin.settings import Settings
from fastapi_amis_admin.admin.site import AdminSite

# Create FastAPI application
app = FastAPI()

# Create AdminSite instance
site = AdminSite(settings=Settings(database_url_async='sqlite+aiosqlite:///amisadmin.db'))

# Mount the background management system
site.mount_app(app)

if __name__ == '__main__':
    import uvicorn

    uvicorn.run(app, debug=True)
```

## Model management example

```python
from fastapi import FastAPI
from sqlmodel import SQLModel
from fastapi_amis_admin.admin.settings import Settings
from fastapi_amis_admin.admin.site import AdminSite
from fastapi_amis_admin.admin import admin
from fastapi_amis_admin.models.fields import Field

# Create FastAPI application
app = FastAPI()

# Create AdminSite instance
site = AdminSite(settings=Settings(database_url_async='sqlite+aiosqlite:///amisadmin.db'))


# First create a SQLModel model, please refer to: https://sqlmodel.tiangolo.com/
class Category(SQLModel, table=True):
    id: int = Field(default=None, primary_key=True, nullable=False)
    name: str = Field(title='CategoryName')
    description: str = Field(default='', title='Description')


# Register ModelAdmin
@site.register_admin
class CategoryAdmin(admin.ModelAdmin):
    page_schema = 'Category Management'
    # Configuration management model
    model = Category


# Mount the background management system
site.mount_app(app)


# Create initialized database table
@app.on_event("startup")
async def startup():
    await site.db.async_run_sync(SQLModel.metadata.create_all, is_session=False)


if __name__ == '__main__':
    import uvicorn

    uvicorn.run(app, debug=True)
```

## Form management example

```python
from typing import Any
from fastapi import FastAPI
from pydantic import BaseModel
from starlette.requests import Request
from fastapi_amis_admin.amis.components import Form
from fastapi_amis_admin.admin import admin
from fastapi_amis_admin.admin.settings import Settings
from fastapi_amis_admin.admin.site import AdminSite
from fastapi_amis_admin.crud.schema import BaseApiOut
from fastapi_amis_admin.models.fields import Field

# Create FastAPI application
app = FastAPI()

# Create AdminSite instance
site = AdminSite(settings=Settings(database_url_async='sqlite+aiosqlite:///amisadmin.db'))


# Register FormAdmin
@site.register_admin
class UserLoginFormAdmin(admin.FormAdmin):
    page_schema = 'UserLoginForm'
    # Configure form information, can be omitted
    form = Form(title='This is a test login form', submitText='Login')

    # Create form data model
    class schema(BaseModel):
        username: str = Field(..., title='username', min_length=3, max_length=30)
        password: str = Field(..., title='password')

    # Process form submission data
    async def handle(self, request: Request, data: BaseModel, **kwargs) -> BaseApiOut[Any]:
        if data.username == 'amisadmin' and data.password == 'amisadmin':
            return BaseApiOut(msg='Login successful!', data={'token': 'xxxxxx'})
        return BaseApiOut(status=-1, msg='username or password is wrong!')


# Mount the background management system
site.mount_app(app)

if __name__ == '__main__':
    import uvicorn

    uvicorn.run(app, debug=True)
```

## Using the command line

```bash
# Install command line extensions
pip install fastapi_amis_admin[cli]

# view help
faa --help

# Initialize a `FastAPI-Amis-Admin` project
faa new project_name --init

# Initialize a `FastAPI-Amis-Admin` application
faa new app_name

# quickly run the project
It's true
```

## Interface preview

- Open `http://127.0.0.1:8000/admin/` in your browser:

![ModelAdmin](https://s2.loli.net/2022/03/20/ItgFYGUONm1jCz5.png)

- Open `http://127.0.0.1:8000/admin/docs` in your browser:

![Docs](https://s2.loli.net/2022/03/20/1GcCiPdmXayxrbH.png)

## Related Items

- [`Amis-Admin-Theme-Editor`](https://github.com/swelcker/amis-admin-theme-editor):Theme-Editor for the fastapi-amis-admin. 
  Allows to add custom css styles and to apply theme --vars change on the fly.
- [`FastAPI-User-Auth`](https://github.com/amisadmin/fastapi_user_auth): A simple and powerful `FastAPI` user `RBAC` authentication and authorization library.
- [`FastAPI-Scheduler`](https://github.com/amisadmin/fastapi_scheduler): A simple scheduled task management project based on `FastAPI`+`APScheduler`.
- [`FastAPI-Amis-Admin-Demo`](https://github.com/amisadmin/fastapi_amis_admin_demo): A sample `FastAPI-Amis-Admin` application.
- [`FastAPI-User-Auth-Demo`](https://github.com/amisadmin/fastapi_user_auth_demo): A sample `FastAPI-User-Auth` application.

## agreement

- `fastapi-amis-admin` is based on `Apache2.0` open source and free to use, and can be used for commercial purposes for free, but please clearly display the copyright information about FastAPI-Amis-Admin in the display interface.

## Thanks

Thanks to the following developers for their contributions to FastAPI-Amis-Admin:

<a href="https://github.com/amisadmin/fastapi_amis_admin/graphs/contributors">
  <img src="https://contrib.rocks/image?repo=amisadmin/fastapi_amis_admin" />
</a>