# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.base.exchange import Exchange
import json
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import AccountSuspended
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import BadResponse
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidAddress
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import NotSupported
from ccxt.base.errors import NetworkError
from ccxt.base.errors import DDoSProtection
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import InvalidNonce
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class digifinex(Exchange):

    def describe(self):
        return self.deep_extend(super(digifinex, self).describe(), {
            'id': 'digifinex',
            'name': 'DigiFinex',
            'countries': ['SG'],
            'version': 'v3',
            'rateLimit': 900,  # 300 for posts
            'has': {
                'CORS': None,
                'spot': True,
                'margin': True,
                'swap': True,
                'future': False,
                'option': False,
                'addMargin': False,
                'cancelOrder': True,
                'cancelOrders': True,
                'createOrder': True,
                'createPostOnlyOrder': True,
                'createReduceOnlyOrder': True,
                'createStopLimitOrder': False,
                'createStopMarketOrder': False,
                'createStopOrder': False,
                'fetchBalance': True,
                'fetchBorrowInterest': True,
                'fetchBorrowRate': True,
                'fetchBorrowRateHistories': False,
                'fetchBorrowRateHistory': False,
                'fetchBorrowRates': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchDepositWithdrawFee': 'emulated',
                'fetchDepositWithdrawFees': True,
                'fetchFundingHistory': False,
                'fetchFundingRate': True,
                'fetchFundingRateHistory': True,
                'fetchFundingRates': False,
                'fetchIndexOHLCV': False,
                'fetchLedger': True,
                'fetchLeverage': False,
                'fetchLeverageTiers': True,
                'fetchMarginMode': False,
                'fetchMarketLeverageTiers': True,
                'fetchMarkets': True,
                'fetchMarkOHLCV': False,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrders': True,
                'fetchPosition': True,
                'fetchPositionMode': False,
                'fetchPositions': True,
                'fetchPositionsRisk': False,
                'fetchPremiumIndexOHLCV': False,
                'fetchStatus': True,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': True,
                'fetchTradingFee': True,
                'fetchTradingFees': False,
                'fetchTransfers': True,
                'fetchWithdrawals': True,
                'reduceMargin': False,
                'setLeverage': True,
                'setMargin': False,
                'setMarginMode': False,
                'setPositionMode': False,
                'transfer': True,
                'withdraw': True,
            },
            'timeframes': {
                '1m': '1',
                '5m': '5',
                '15m': '15',
                '30m': '30',
                '1h': '60',
                '4h': '240',
                '12h': '720',
                '1d': '1D',
                '1w': '1W',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/51840849/87443315-01283a00-c5fe-11ea-8628-c2a0feaf07ac.jpg',
                'api': {
                    'rest': 'https://openapi.digifinex.com',
                },
                'www': 'https://www.digifinex.com',
                'doc': [
                    'https://docs.digifinex.com',
                ],
                'fees': 'https://digifinex.zendesk.com/hc/en-us/articles/360000328422-Fee-Structure-on-DigiFinex',
                'referral': 'https://www.digifinex.com/en-ww/from/DhOzBg?channelCode=ljaUPp',
            },
            'api': {
                'public': {
                    'spot': {
                        'get': [
                            '{market}/symbols',
                            'kline',
                            'margin/currencies',
                            'margin/symbols',
                            'markets',
                            'order_book',
                            'ping',
                            'spot/symbols',
                            'time',
                            'trades',
                            'trades/symbols',
                            'ticker',
                            'currencies',
                        ],
                    },
                    'swap': {
                        'get': [
                            'public/api_weight',
                            'public/candles',
                            'public/candles_history',
                            'public/depth',
                            'public/funding_rate',
                            'public/funding_rate_history',
                            'public/instrument',
                            'public/instruments',
                            'public/ticker',
                            'public/tickers',
                            'public/time',
                            'public/trades',
                        ],
                    },
                },
                'private': {
                    'spot': {
                        'get': [
                            '{market}/financelog',
                            '{market}/mytrades',
                            '{market}/order',
                            '{market}/order/detail',
                            '{market}/order/current',
                            '{market}/order/history',
                            'margin/assets',
                            'margin/financelog',
                            'margin/mytrades',
                            'margin/order',
                            'margin/order/current',
                            'margin/order/history',
                            'margin/positions',
                            'otc/financelog',
                            'spot/assets',
                            'spot/financelog',
                            'spot/mytrades',
                            'spot/order',
                            'spot/order/current',
                            'spot/order/history',
                            'deposit/address',
                            'deposit/history',
                            'withdraw/history',
                        ],
                        'post': [
                            '{market}/order/cancel',
                            '{market}/order/new',
                            '{market}/order/batch_new',
                            'margin/order/cancel',
                            'margin/order/new',
                            'margin/position/close',
                            'spot/order/cancel',
                            'spot/order/new',
                            'transfer',
                            'withdraw/new',
                            'withdraw/cancel',
                        ],
                    },
                    'swap': {
                        'get': [
                            'account/balance',
                            'account/positions',
                            'account/finance_record',
                            'account/trading_fee_rate',
                            'account/transfer_record',
                            'trade/history_orders',
                            'trade/history_trades',
                            'trade/open_orders',
                            'trade/order_info',
                        ],
                        'post': [
                            'account/leverage',
                            'trade/batch_cancel_order',
                            'trade/batch_order',
                            'trade/cancel_order',
                            'trade/order_place',
                        ],
                    },
                },
            },
            'fees': {
                'trading': {
                    'tierBased': True,
                    'percentage': True,
                    'maker': self.parse_number('0.002'),
                    'taker': self.parse_number('0.002'),
                },
            },
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    '10001': [BadRequest, "Wrong request method, please check it's a GET ot POST request"],
                    '10002': [AuthenticationError, 'Invalid ApiKey'],
                    '10003': [AuthenticationError, "Sign doesn't match"],
                    '10004': [BadRequest, 'Illegal request parameters'],
                    '10005': [DDoSProtection, 'Request frequency exceeds the limit'],
                    '10006': [PermissionDenied, 'Unauthorized to execute self request'],
                    '10007': [PermissionDenied, 'IP address Unauthorized'],
                    '10008': [InvalidNonce, 'Timestamp for self request is invalid, timestamp must within 1 minute'],
                    '10009': [NetworkError, 'Unexist endpoint, please check endpoint URL'],
                    '10011': [AccountSuspended, 'ApiKey expired. Please go to client side to re-create an ApiKey'],
                    '20001': [PermissionDenied, 'Trade is not open for self trading pair'],
                    '20002': [PermissionDenied, 'Trade of self trading pair is suspended'],
                    '20003': [InvalidOrder, 'Invalid price or amount'],
                    '20007': [InvalidOrder, 'Price precision error'],
                    '20008': [InvalidOrder, 'Amount precision error'],
                    '20009': [InvalidOrder, 'Amount is less than the minimum requirement'],
                    '20010': [InvalidOrder, 'Cash Amount is less than the minimum requirement'],
                    '20011': [InsufficientFunds, 'Insufficient balance'],
                    '20012': [BadRequest, 'Invalid trade type, valid value: buy/sell)'],
                    '20013': [InvalidOrder, 'No order info found'],
                    '20014': [BadRequest, 'Invalid date, Valid format: 2018-07-25)'],
                    '20015': [BadRequest, 'Date exceeds the limit'],
                    '20018': [PermissionDenied, 'Your trading rights have been banned by the system'],
                    '20019': [BadSymbol, 'Wrong trading pair symbol. Correct format:"usdt_btc". Quote asset is in the front'],
                    '20020': [DDoSProtection, "You have violated the API operation trading rules and temporarily forbid trading. At present, we have certain restrictions on the user's transaction rate and withdrawal rate."],
                    '50000': [ExchangeError, 'Exception error'],
                    '20021': [BadRequest, 'Invalid currency'],
                    '20022': [BadRequest, 'The ending timestamp must be larger than the starting timestamp'],
                    '20023': [BadRequest, 'Invalid transfer type'],
                    '20024': [BadRequest, 'Invalid amount'],
                    '20025': [BadRequest, 'This currency is not transferable at the moment'],
                    '20026': [InsufficientFunds, 'Transfer amount exceed your balance'],
                    '20027': [PermissionDenied, 'Abnormal account status'],
                    '20028': [PermissionDenied, 'Blacklist for transfer'],
                    '20029': [PermissionDenied, 'Transfer amount exceed your daily limit'],
                    '20030': [BadRequest, 'You have no position on self trading pair'],
                    '20032': [PermissionDenied, 'Withdrawal limited'],
                    '20033': [BadRequest, 'Wrong Withdrawal ID'],
                    '20034': [PermissionDenied, 'Withdrawal service of self crypto has been closed'],
                    '20035': [PermissionDenied, 'Withdrawal limit'],
                    '20036': [ExchangeError, 'Withdrawal cancellation failed'],
                    '20037': [InvalidAddress, 'The withdrawal address, Tag or chain type is not included in the withdrawal management list'],
                    '20038': [InvalidAddress, 'The withdrawal address is not on the white list'],
                    '20039': [ExchangeError, "Can't be canceled in current status"],
                    '20040': [RateLimitExceeded, 'Withdraw too frequently; limitation: 3 times a minute, 100 times a day'],
                    '20041': [PermissionDenied, 'Beyond the daily withdrawal limit'],
                    '20042': [BadSymbol, 'Current trading pair does not support API trading'],
                    '400002': [BadRequest, 'Invalid Parameter'],
                },
                'broad': {
                },
            },
            'options': {
                'defaultType': 'spot',
                'types': ['spot', 'margin', 'otc'],
                'accountsByType': {
                    'spot': '1',
                    'margin': '2',
                    'OTC': '3',
                },
                'networks': {
                    'ARBITRUM': 'Arbitrum',
                    'AVALANCEC': 'AVAX-CCHAIN',
                    'AVALANCEX': 'AVAX-XCHAIN',
                    'BEP20': 'BEP20',
                    'BSC': 'BEP20',
                    'CARDANO': 'Cardano',
                    'CELO': 'Celo',
                    'CHILIZ': 'Chiliz',
                    'COSMOS': 'COSMOS',
                    'CRC20': 'Crypto.com',
                    'CRONOS': 'Crypto.com',
                    'DOGECOIN': 'DogeChain',
                    'ERC20': 'ERC20',
                    'ETH': 'ERC20',
                    'ETHW': 'ETHW',
                    'IOTA': 'MIOTA',
                    'KLAYTN': 'KLAY',
                    'MATIC': 'Polygon',
                    'METIS': 'MetisDAO',
                    'MOONBEAM': 'GLMR',
                    'MOONRIVER': 'Moonriver',
                    'OPTIMISM': 'OPETH',
                    'POLYGON': 'Polygon',
                    'RIPPLE': 'XRP',
                    'SOLANA': 'SOL',  # SOL & SPL
                    'STELLAR': 'Stella',  # XLM
                    'TERRACLASSIC': 'TerraClassic',
                    'TERRA': 'Terra',
                    'TON': 'Ton',
                    'TRC20': 'TRC20',
                    'TRON': 'TRC20',
                    'TRX': 'TRC20',
                    'VECHAIN': 'Vechain',  # VET
                },
                'networksById': {
                    'Arbitrum': 'ARBITRUM',
                    'AVAX-CCHAIN': 'AVALANCEC',
                    'AVAX-XCHAIN': 'AVALANCEX',
                    'BEP20': 'BEP20',
                    'Cardano': 'CARDANO',
                    'Celo': 'CELO',
                    'Chiliz': 'CHILIZ',
                    'COSMOS': 'COSMOS',
                    'Crypto.com': 'CRC20',  # CRONOS
                    'DogeChain': 'DOGECOIN',
                    'ERC20': 'ERC20',
                    'ETHW': 'ETHW',
                    'MIOTA': 'IOTA',
                    'KLAY': 'KLAYTN',
                    'Polygon': 'POLYGON',
                    'MetisDAO': 'METIS',
                    'Moonriver': 'MOONRIVER',
                    'GLMR': 'MOONBEAM',
                    'OPETH': 'OPTIMISM',
                    'XRP': 'RIPPLE',
                    'SOL': 'SOLANA',
                    'Stella': 'STELLAR',
                    'Terra': 'TERRA',
                    'TerraClassic': 'TERRACLASSIC',
                    'Ton': 'TON',
                    'TRC20': 'TRC20',
                    'Vechain': 'VECHAIN',
                },
            },
            'commonCurrencies': {
                'BHT': 'Black House Test',
                'EPS': 'Epanus',
                'FREE': 'FreeRossDAO',
                'MBN': 'Mobilian Coin',
                'TEL': 'TEL666',
            },
        })

    def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = self.publicSpotGetCurrencies(params)
        #
        #     {
        #         "data":[
        #             {
        #                 "deposit_status":1,
        #                 "min_deposit_amount":10,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":10,
        #                 "min_withdraw_fee":5,
        #                 "currency":"USDT",
        #                 "withdraw_status":0,
        #                 "chain":"OMNI"
        #             },
        #             {
        #                 "deposit_status":1,
        #                 "min_deposit_amount":10,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":10,
        #                 "min_withdraw_fee":3,
        #                 "currency":"USDT",
        #                 "withdraw_status":1,
        #                 "chain":"ERC20"
        #             },
        #             {
        #                 "deposit_status":0,
        #                 "min_deposit_amount":0,
        #                 "withdraw_fee_rate":0,
        #                 "min_withdraw_amount":0,
        #                 "min_withdraw_fee":0,
        #                 "currency":"DGF13",
        #                 "withdraw_status":0,
        #                 "chain":""
        #             },
        #         ],
        #         "code":200
        #     }
        #
        data = self.safe_value(response, 'data', [])
        result = {}
        for i in range(0, len(data)):
            currency = data[i]
            id = self.safe_string(currency, 'currency')
            code = self.safe_currency_code(id)
            depositStatus = self.safe_integer(currency, 'deposit_status', 1)
            withdrawStatus = self.safe_integer(currency, 'withdraw_status', 1)
            deposit = depositStatus > 0
            withdraw = withdrawStatus > 0
            active = deposit and withdraw
            feeString = self.safe_string(currency, 'min_withdraw_fee')  # withdraw_fee_rate was zero for all currencies, so self was the worst case scenario
            minWithdrawString = self.safe_string(currency, 'min_withdraw_amount')
            minDepositString = self.safe_string(currency, 'min_deposit_amount')
            minDeposit = self.parse_number(minDepositString)
            minWithdraw = self.parse_number(minWithdrawString)
            fee = self.parse_number(feeString)
            # define precision with temporary way
            minFoundPrecision = Precise.string_min(feeString, Precise.string_min(minDepositString, minWithdrawString))
            precision = self.parse_number(minFoundPrecision)
            networkId = self.safe_string(currency, 'chain')
            networkCode = self.network_id_to_code(networkId)
            network = {
                'info': currency,
                'id': networkId,
                'network': networkCode,
                'active': active,
                'fee': fee,
                'precision': precision,
                'deposit': deposit,
                'withdraw': withdraw,
                'limits': {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'withdraw': {
                        'min': minWithdraw,
                        'max': None,
                    },
                    'deposit': {
                        'min': minDeposit,
                        'max': None,
                    },
                },
            }
            if code in result:
                if isinstance(result[code]['info'], list):
                    result[code]['info'].append(currency)
                else:
                    result[code]['info'] = [result[code]['info'], currency]
                if withdraw:
                    result[code]['withdraw'] = True
                    result[code]['limits']['withdraw']['min'] = min(result[code]['limits']['withdraw']['min'], minWithdraw)
                if deposit:
                    result[code]['deposit'] = True
                    result[code]['limits']['deposit']['min'] = min(result[code]['limits']['deposit']['min'], minDeposit)
                if active:
                    result[code]['active'] = True
            else:
                result[code] = {
                    'id': id,
                    'code': code,
                    'info': currency,
                    'type': None,
                    'name': None,
                    'active': active,
                    'deposit': deposit,
                    'withdraw': withdraw,
                    'fee': self.parse_number(feeString),
                    'precision': None,
                    'limits': {
                        'amount': {
                            'min': None,
                            'max': None,
                        },
                        'withdraw': {
                            'min': minWithdraw,
                            'max': None,
                        },
                        'deposit': {
                            'min': minDeposit,
                            'max': None,
                        },
                    },
                    'networks': {},
                }
            if networkId is not None:
                result[code]['networks'][networkId] = network
            else:
                result[code]['active'] = active
                result[code]['fee'] = self.parse_number(feeString)
                result[code]['deposit'] = deposit
                result[code]['withdraw'] = withdraw
                result[code]['limits'] = {
                    'amount': {
                        'min': None,
                        'max': None,
                    },
                    'withdraw': {
                        'min': minWithdraw,
                        'max': None,
                    },
                    'deposit': {
                        'min': minDeposit,
                        'max': None,
                    },
                }
            result[code]['precision'] = precision if (result[code]['precision'] is None) else max(result[code]['precision'], precision)
        return result

    def fetch_markets(self, params={}):
        """
        retrieves data on all markets for digifinex
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        options = self.safe_value(self.options, 'fetchMarkets', {})
        method = self.safe_string(options, 'method', 'fetch_markets_v2')
        return getattr(self, method)(params)

    def fetch_markets_v2(self, params={}):
        defaultType = self.safe_string(self.options, 'defaultType')
        marginMode, query = self.handle_margin_mode_and_params('fetchMarketsV2', params)
        method = 'publicSpotGetMarginSymbols' if (marginMode is not None) else 'publicSpotGetTradesSymbols'
        promises = [getattr(self, method)(query), self.publicSwapGetPublicInstruments(params)]
        spotMarkets = promises[0]
        swapMarkets = promises[1]
        #
        # spot and margin
        #
        #     {
        #         "symbol_list":[
        #             {
        #                 "order_types":["LIMIT","MARKET"],
        #                 "quote_asset":"USDT",
        #                 "minimum_value":2,
        #                 "amount_precision":4,
        #                 "status":"TRADING",
        #                 "minimum_amount":0.0001,
        #                 "symbol":"BTC_USDT",
        #                 "is_allow":1,
        #                 "zone":"MAIN",
        #                 "base_asset":"BTC",
        #                 "price_precision":2
        #             }
        #         ],
        #         "code":0
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "type": "REAL",
        #                 "contract_type": "PERPETUAL",
        #                 "base_currency": "BTC",
        #                 "quote_currency": "USDT",
        #                 "clear_currency": "USDT",
        #                 "contract_value": "0.001",
        #                 "contract_value_currency": "BTC",
        #                 "is_inverse": False,
        #                 "is_trading": True,
        #                 "status": "ONLINE",
        #                 "price_precision": 4,
        #                 "tick_size": "0.0001",
        #                 "min_order_amount": 1,
        #                 "open_max_limits": [
        #                     {
        #                         "leverage": "50",
        #                         "max_limit": "1000000"
        #                     }
        #                 ]
        #             },
        #         ]
        #     }
        #
        spotData = self.safe_value(spotMarkets, 'symbol_list', [])
        swapData = self.safe_value(swapMarkets, 'data', [])
        response = self.array_concat(spotData, swapData)
        result = []
        for i in range(0, len(response)):
            market = response[i]
            id = self.safe_string_2(market, 'symbol', 'instrument_id')
            baseId = self.safe_string_2(market, 'base_asset', 'base_currency')
            quoteId = self.safe_string_2(market, 'quote_asset', 'quote_currency')
            settleId = self.safe_string(market, 'clear_currency')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            settle = self.safe_currency_code(settleId)
            #
            # The status is documented in the exchange API docs as follows:
            # TRADING, HALT(delisted), BREAK(trading paused)
            # https://docs.digifinex.vip/en-ww/v3/#/public/spot/symbols
            # However, all spot markets actually have status == 'HALT'
            # despite that they appear to be active on the exchange website.
            # Apparently, we can't trust self status.
            # status = self.safe_string(market, 'status')
            # active = (status == 'TRADING')
            #
            isAllowed = self.safe_integer(market, 'is_allow', 1)
            type = 'margin' if (defaultType == 'margin') else 'spot'
            spot = settle is None
            swap = not spot
            margin = True if (marginMode is not None) else None
            symbol = base + '/' + quote
            isInverse = None
            isLinear = None
            if swap:
                type = 'swap'
                symbol = base + '/' + quote + ':' + settle
                isInverse = self.safe_value(market, 'is_inverse')
                isLinear = True if (not isInverse) else False
                isTrading = self.safe_value(market, 'isTrading')
                if isTrading:
                    isAllowed = True
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'settle': settle,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': settleId,
                'type': type,
                'spot': spot,
                'margin': margin,
                'swap': swap,
                'future': False,
                'option': False,
                'active': True if isAllowed else False,
                'contract': swap,
                'linear': isLinear,
                'inverse': isInverse,
                'contractSize': self.safe_number(market, 'contract_value'),
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.parse_number(self.parse_precision(self.safe_string(market, 'amount_precision'))),
                    'price': self.parse_number(self.parse_precision(self.safe_string(market, 'price_precision'))),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number_2(market, 'minimum_amount', 'min_order_amount'),
                        'max': None,
                    },
                    'price': {
                        'min': self.safe_number(market, 'tick_size'),
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'minimum_value'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def fetch_markets_v1(self, params={}):
        response = self.publicSpotGetMarkets(params)
        #
        #     {
        #         "data": [
        #             {
        #                 "volume_precision":4,
        #                 "price_precision":2,
        #                 "market":"btc_usdt",
        #                 "min_amount":2,
        #                 "min_volume":0.0001
        #             },
        #         ],
        #         "date":1564507456,
        #         "code":0
        #     }
        #
        markets = self.safe_value(response, 'data', [])
        result = []
        for i in range(0, len(markets)):
            market = markets[i]
            id = self.safe_string(market, 'market')
            baseId, quoteId = id.split('_')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            result.append({
                'id': id,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': 'spot',
                'spot': True,
                'margin': None,
                'swap': False,
                'future': False,
                'option': False,
                'active': None,
                'contract': False,
                'linear': None,
                'inverse': None,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'price': self.parse_number(self.parse_precision(self.safe_string(market, 'price_precision'))),
                    'amount': self.parse_number(self.parse_precision(self.safe_string(market, 'volume_precision'))),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'min_volume'),
                        'max': None,
                    },
                    'price': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'min_amount'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def parse_balance(self, response):
        #
        # spot and margin
        #
        #     {
        #         "currency": "BTC",
        #         "free": 4723846.89208129,
        #         "total": 0
        #     }
        #
        # swap
        #
        #     {
        #         "equity": "0",
        #         "currency": "BTC",
        #         "margin": "0",
        #         "frozen_margin": "0",
        #         "frozen_money": "0",
        #         "margin_ratio": "0",
        #         "realized_pnl": "0",
        #         "avail_balance": "0",
        #         "unrealized_pnl": "0",
        #         "time_stamp": 1661487402396
        #     }
        #
        result = {'info': response}
        for i in range(0, len(response)):
            balance = response[i]
            currencyId = self.safe_string(balance, 'currency')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            free = self.safe_string_2(balance, 'free', 'avail_balance')
            total = self.safe_string_2(balance, 'total', 'equity')
            account['free'] = free
            account['used'] = Precise.string_sub(total, free)
            account['total'] = total
            result[code] = account
        return self.safe_balance(result)

    def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#spot-account-assets
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#margin-assets
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#accountbalance
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        self.load_markets()
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchBalance', None, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotAssets',
            'margin': 'privateSpotGetMarginAssets',
            'swap': 'privateSwapGetAccountBalance',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchBalance', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginAssets'
            marketType = 'margin'
        response = getattr(self, method)(query)
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "list": [
        #             {
        #                 "currency": "BTC",
        #                 "free": 4723846.89208129,
        #                 "total": 0
        #             },
        #             ...
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "equity": "0",
        #                 "currency": "BTC",
        #                 "margin": "0",
        #                 "frozen_margin": "0",
        #                 "frozen_money": "0",
        #                 "margin_ratio": "0",
        #                 "realized_pnl": "0",
        #                 "avail_balance": "0",
        #                 "unrealized_pnl": "0",
        #                 "time_stamp": 1661487402396
        #             },
        #             ...
        #         ]
        #     }
        #
        balanceRequest = 'data' if (marketType == 'swap') else 'list'
        balances = self.safe_value(response, balanceRequest, [])
        return self.parse_balance(balances)

    def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-orderbook
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#orderbook
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        self.load_markets()
        market = self.market(symbol)
        marketType, query = self.handle_market_type_and_params('fetchOrderBook', market, params)
        request = {}
        method = None
        if marketType == 'swap':
            method = 'publicSwapGetPublicDepth'
            request['instrument_id'] = market['id']
        else:
            method = 'publicSpotGetOrderBook'
            request['symbol'] = market['id']
        if limit is not None:
            request['limit'] = limit
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot
        #
        #     {
        #         "bids": [
        #             [9605.77,0.0016],
        #             [9605.46,0.0003],
        #             [9602.04,0.0127],
        #         ],
        #         "asks": [
        #             [9627.22,0.025803],
        #             [9627.12,0.168543],
        #             [9626.52,0.0011529],
        #         ],
        #         "date":1564509499,
        #         "code":0
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "timestamp": 1667975290425,
        #             "asks": [
        #                 ["18384.7",3492],
        #                 ["18402.7",5000],
        #                 ["18406.7",5000],
        #             ],
        #             "bids": [
        #                 ["18366.2",4395],
        #                 ["18364.3",3070],
        #                 ["18359.4",5000],
        #             ]
        #         }
        #     }
        #
        timestamp = None
        orderBook = None
        if marketType == 'swap':
            orderBook = self.safe_value(response, 'data', {})
            timestamp = self.safe_integer(orderBook, 'timestamp')
        else:
            orderBook = response
            timestamp = self.safe_timestamp(response, 'date')
        return self.parse_order_book(orderBook, market['symbol'], timestamp)

    def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#ticker-price
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#tickers
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        symbols = self.market_symbols(symbols)
        first = self.safe_string(symbols, 0)
        market = None
        if first is not None:
            market = self.market(first)
        type = None
        type, params = self.handle_market_type_and_params('fetchTickers', market, params)
        method = 'publicSpotGetTicker'
        request = {}
        if type == 'swap':
            method = 'publicSwapGetPublicTickers'
        response = getattr(self, method)(self.extend(request, params))
        #
        # spot
        #
        #    {
        #        "ticker": [{
        #            "vol": 40717.4461,
        #            "change": -1.91,
        #            "base_vol": 392447999.65374,
        #            "sell": 9592.23,
        #            "last": 9592.22,
        #            "symbol": "btc_usdt",
        #            "low": 9476.24,
        #            "buy": 9592.03,
        #            "high": 9793.87
        #        }],
        #        "date": 1589874294,
        #        "code": 0
        #    }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "SUSHIUSDTPERP",
        #                 "index_price": "1.1297",
        #                 "mark_price": "1.1289",
        #                 "max_buy_price": "1.1856",
        #                 "min_sell_price": "1.0726",
        #                 "best_bid": "1.1278",
        #                 "best_bid_size": "500",
        #                 "best_ask": "1.1302",
        #                 "best_ask_size": "471",
        #                 "high_24h": "1.2064",
        #                 "open_24h": "1.1938",
        #                 "low_24h": "1.1239",
        #                 "last": "1.1302",
        #                 "last_qty": "29",
        #                 "volume_24h": "4946163",
        #                 "price_change_percent": "-0.053275255486681085",
        #                 "open_interest": "-",
        #                 "timestamp": 1663222782100
        #             },
        #             ...
        #         ]
        #     }
        #
        result = {}
        tickers = self.safe_value_2(response, 'ticker', 'data', [])
        date = self.safe_integer(response, 'date')
        for i in range(0, len(tickers)):
            rawTicker = self.extend({
                'date': date,
            }, tickers[i])
            ticker = self.parse_ticker(rawTicker)
            symbol = ticker['symbol']
            result[symbol] = ticker
        return self.filter_by_array(result, 'symbol', symbols)

    def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#ticker-price
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#ticker
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        method = 'publicSpotGetTicker'
        request = {}
        if market['swap']:
            method = 'publicSwapGetPublicTicker'
            request['instrument_id'] = market['id']
        else:
            request['symbol'] = market['id']
        response = getattr(self, method)(self.extend(request, params))
        #
        # spot
        #
        #    {
        #        "ticker": [{
        #            "vol": 40717.4461,
        #            "change": -1.91,
        #            "base_vol": 392447999.65374,
        #            "sell": 9592.23,
        #            "last": 9592.22,
        #            "symbol": "btc_usdt",
        #            "low": 9476.24,
        #            "buy": 9592.03,
        #            "high": 9793.87
        #        }],
        #        "date": 1589874294,
        #        "code": 0
        #    }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "index_price": "20141.9967",
        #             "mark_price": "20139.3404",
        #             "max_buy_price": "21146.4838",
        #             "min_sell_price": "19132.2725",
        #             "best_bid": "20140.0998",
        #             "best_bid_size": "3116",
        #             "best_ask": "20140.0999",
        #             "best_ask_size": "9004",
        #             "high_24h": "20410.6496",
        #             "open_24h": "20308.6998",
        #             "low_24h": "19600",
        #             "last": "20140.0999",
        #             "last_qty": "2",
        #             "volume_24h": "49382816",
        #             "price_change_percent": "-0.008301855936636448",
        #             "open_interest": "-",
        #             "timestamp": 1663221614998
        #         }
        #     }
        #
        date = self.safe_integer(response, 'date')
        tickers = self.safe_value(response, 'ticker', [])
        data = self.safe_value(response, 'data', {})
        firstTicker = self.safe_value(tickers, 0, {})
        result = None
        if market['swap']:
            result = data
        else:
            result = self.extend({'date': date}, firstTicker)
        return self.parse_ticker(result, market)

    def parse_ticker(self, ticker, market=None):
        #
        # spot: fetchTicker, fetchTickers
        #
        #     {
        #         "last":0.021957,
        #         "symbol": "btc_usdt",
        #         "base_vol":2249.3521732227,
        #         "change":-0.6,
        #         "vol":102443.5111,
        #         "sell":0.021978,
        #         "low":0.021791,
        #         "buy":0.021946,
        #         "high":0.022266,
        #         "date"1564518452,  # injected from fetchTicker/fetchTickers
        #     }
        #
        # swap: fetchTicker, fetchTickers
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "index_price": "20141.9967",
        #         "mark_price": "20139.3404",
        #         "max_buy_price": "21146.4838",
        #         "min_sell_price": "19132.2725",
        #         "best_bid": "20140.0998",
        #         "best_bid_size": "3116",
        #         "best_ask": "20140.0999",
        #         "best_ask_size": "9004",
        #         "high_24h": "20410.6496",
        #         "open_24h": "20308.6998",
        #         "low_24h": "19600",
        #         "last": "20140.0999",
        #         "last_qty": "2",
        #         "volume_24h": "49382816",
        #         "price_change_percent": "-0.008301855936636448",
        #         "open_interest": "-",
        #         "timestamp": 1663221614998
        #     }
        #
        marketId = self.safe_string_upper_2(ticker, 'symbol', 'instrument_id')
        symbol = self.safe_symbol(marketId, market)
        market = self.safe_market(marketId)
        timestamp = self.safe_timestamp(ticker, 'date')
        if market['swap']:
            timestamp = self.safe_integer(ticker, 'timestamp')
        last = self.safe_string(ticker, 'last')
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string_2(ticker, 'high', 'high_24h'),
            'low': self.safe_string_2(ticker, 'low', 'low_24h'),
            'bid': self.safe_string_2(ticker, 'buy', 'best_bid'),
            'bidVolume': self.safe_string(ticker, 'best_bid_size'),
            'ask': self.safe_string_2(ticker, 'sell', 'best_ask'),
            'askVolume': self.safe_string(ticker, 'best_ask_size'),
            'vwap': None,
            'open': self.safe_string(ticker, 'open_24h'),
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': self.safe_string_2(ticker, 'change', 'price_change_percent'),
            'average': None,
            'baseVolume': self.safe_string_2(ticker, 'vol', 'volume_24h'),
            'quoteVolume': self.safe_string(ticker, 'base_vol'),
            'info': ticker,
        }, market)

    def parse_trade(self, trade, market=None):
        #
        # spot: fetchTrades
        #
        #     {
        #         "date":1564520003,
        #         "id":1596149203,
        #         "amount":0.7073,
        #         "type":"buy",
        #         "price":0.02193,
        #     }
        #
        # swap: fetchTrades
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "trade_id": "1595190773677035521",
        #         "direction": "4",
        #         "volume": "4",
        #         "price": "16188.3",
        #         "trade_time": 1669158092314
        #     }
        #
        # spot: fetchMyTrades
        #
        #     {
        #         "symbol": "BTC_USDT",
        #         "order_id": "6707cbdcda0edfaa7f4ab509e4cbf966",
        #         "id": 28457,
        #         "price": 0.1,
        #         "amount": 0,
        #         "fee": 0.096,
        #         "fee_currency": "USDT",
        #         "timestamp": 1499865549,
        #         "side": "buy",  # or "side": "sell_market"
        #         "is_maker": True
        #     }
        #
        # swap: fetchMyTrades
        #
        #     {
        #         "trade_id": "1590136768424841218",
        #         "instrument_id": "BTCUSDTPERP",
        #         "order_id": "1590136768156405760",
        #         "type": 1,
        #         "order_type": 8,
        #         "price": "18514.5",
        #         "size": "1",
        #         "fee": "0.00925725",
        #         "close_profit": "0",
        #         "leverage": "20",
        #         "trade_type": 0,
        #         "match_role": 1,
        #         "trade_time": 1667953123562
        #     }
        #
        id = self.safe_string_2(trade, 'id', 'trade_id')
        orderId = self.safe_string(trade, 'order_id')
        priceString = self.safe_string(trade, 'price')
        amountString = self.safe_string_n(trade, ['amount', 'volume', 'size'])
        marketId = self.safe_string_upper_2(trade, 'symbol', 'instrument_id')
        symbol = self.safe_symbol(marketId, market)
        if market is None:
            market = self.safe_market(marketId)
        timestamp = self.safe_timestamp_2(trade, 'date', 'timestamp')
        side = self.safe_string_2(trade, 'type', 'side')
        type = None
        takerOrMaker = None
        if market['type'] == 'swap':
            timestamp = self.safe_integer(trade, 'trade_time')
            orderType = self.safe_string(trade, 'order_type')
            tradeRole = self.safe_string(trade, 'match_role')
            direction = self.safe_string(trade, 'direction')
            if orderType is not None:
                type = 'limit' if (orderType == '0') else None
            if tradeRole == '1':
                takerOrMaker = 'taker'
            elif tradeRole == '2':
                takerOrMaker = 'maker'
            else:
                takerOrMaker = None
            if (side == '1') or (direction == '1'):
                # side = 'open long'
                side = 'buy'
            elif (side == '2') or (direction == '2'):
                # side = 'open short'
                side = 'sell'
            elif (side == '3') or (direction == '3'):
                # side = 'close long'
                side = 'sell'
            elif (side == '4') or (direction == '4'):
                # side = 'close short'
                side = 'buy'
        else:
            parts = side.split('_')
            side = self.safe_string(parts, 0)
            type = self.safe_string(parts, 1)
            if type is None:
                type = 'limit'
            isMaker = self.safe_value(trade, 'is_maker')
            takerOrMaker = 'maker' if isMaker else 'taker'
        fee = None
        feeCostString = self.safe_string(trade, 'fee')
        if feeCostString is not None:
            feeCurrencyId = self.safe_string(trade, 'fee_currency')
            feeCurrencyCode = None
            if feeCurrencyId is not None:
                feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
            fee = {
                'cost': feeCostString,
                'currency': feeCurrencyCode,
            }
        return self.safe_trade({
            'id': id,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': type,
            'order': orderId,
            'side': side,
            'price': priceString,
            'amount': amountString,
            'cost': None,
            'takerOrMaker': takerOrMaker,
            'fee': fee,
        }, market)

    def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = self.publicSpotGetTime(params)
        #
        #     {
        #         "server_time": 1589873762,
        #         "code": 0
        #     }
        #
        return self.safe_timestamp(response, 'server_time')

    def fetch_status(self, params={}):
        """
        the latest known information on the availability of the exchange API
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `status structure <https://docs.ccxt.com/en/latest/manual.html#exchange-status-structure>`
        """
        response = self.publicSpotGetPing(params)
        #
        #     {
        #         "msg": "pong",
        #         "code": 0
        #     }
        #
        code = self.safe_integer(response, 'code')
        status = 'ok' if (code == 0) else 'maintenance'
        return {
            'status': status,
            'updated': None,
            'eta': None,
            'url': None,
            'info': response,
        }

    def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-recent-trades
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#recenttrades
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        self.load_markets()
        market = self.market(symbol)
        method = 'publicSpotGetTrades'
        request = {}
        if market['swap']:
            method = 'publicSwapGetPublicTrades'
            request['instrument_id'] = market['id']
        else:
            request['symbol'] = market['id']
        if limit is not None:
            request['limit'] = limit
        response = getattr(self, method)(self.extend(request, params))
        #
        # spot
        #
        #     {
        #         "data":[
        #             {
        #                 "date":1564520003,
        #                 "id":1596149203,
        #                 "amount":0.7073,
        #                 "type":"buy",
        #                 "price":0.02193,
        #             },
        #             {
        #                 "date":1564520002,
        #                 "id":1596149165,
        #                 "amount":0.3232,
        #                 "type":"sell",
        #                 "price":0.021927,
        #             },
        #         ],
        #         "code": 0,
        #         "date": 1564520003,
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "trade_id": "1595190773677035521",
        #                 "direction": "4",
        #                 "volume": "4",
        #                 "price": "16188.3",
        #                 "trade_time": 1669158092314
        #             },
        #             ...
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_trades(data, market, since, limit)

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     [
        #         1556712900,
        #         2205.899,
        #         0.029967,
        #         0.02997,
        #         0.029871,
        #         0.029927
        #     ]
        #
        if market['swap']:
            return [
                self.safe_integer(ohlcv, 0),
                self.safe_number(ohlcv, 1),  # open
                self.safe_number(ohlcv, 2),  # high
                self.safe_number(ohlcv, 3),  # low
                self.safe_number(ohlcv, 4),  # close
                self.safe_number(ohlcv, 5),  # volume
            ]
        else:
            return [
                self.safe_timestamp(ohlcv, 0),
                self.safe_number(ohlcv, 5),  # open
                self.safe_number(ohlcv, 3),  # high
                self.safe_number(ohlcv, 4),  # low
                self.safe_number(ohlcv, 2),  # close
                self.safe_number(ohlcv, 1),  # volume
            ]

    def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-candles-data
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#recentcandle
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        self.load_markets()
        market = self.market(symbol)
        method = 'publicSpotGetKline'
        request = {}
        if market['swap']:
            method = 'publicSwapGetPublicCandles'
            request['instrument_id'] = market['id']
            request['granularity'] = timeframe
            if limit is not None:
                request['limit'] = limit
        else:
            request['symbol'] = market['id']
            request['period'] = self.timeframes[timeframe]
            if since is not None:
                startTime = int(since / 1000)
                request['start_time'] = startTime
                if limit is not None:
                    duration = self.parse_timeframe(timeframe)
                    request['end_time'] = self.sum(startTime, limit * duration)
            elif limit is not None:
                endTime = self.seconds()
                duration = self.parse_timeframe(timeframe)
                request['start_time'] = self.sum(endTime, -limit * duration)
        response = getattr(self, method)(self.extend(request, params))
        #
        # spot
        #
        #     {
        #         "code":0,
        #         "data":[
        #             [1556712900,2205.899,0.029967,0.02997,0.029871,0.029927],
        #             [1556713800,1912.9174,0.029992,0.030014,0.029955,0.02996],
        #             [1556714700,1556.4795,0.029974,0.030019,0.029969,0.02999],
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "granularity": "1m",
        #             "candles": [
        #                 [1588089660000,"6900","6900","6900","6900","0","0"],
        #                 [1588089720000,"6900","6900","6900","6900","0","0"],
        #                 [1588089780000,"6900","6900","6900","6900","0","0"],
        #             ]
        #         }
        #     }
        #
        candles = None
        if market['swap']:
            data = self.safe_value(response, 'data', {})
            candles = self.safe_value(data, 'candles', [])
        else:
            candles = self.safe_value(response, 'data', [])
        return self.parse_ohlcvs(candles, market, timeframe, since, limit)

    def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#create-new-order
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#orderplace
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the digifinex api endpoint
        :param str params['timeInForce']: "GTC", "IOC", "FOK", or "PO"
        :param bool params['postOnly']: True or False
        :param bool params['reduceOnly']: True or False
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        symbol = market['symbol']
        marketType = None
        marginMode = None
        marketType, params = self.handle_market_type_and_params('createOrder', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotPostSpotOrderNew',
            'margin': 'privateSpotPostMarginOrderNew',
            'swap': 'privateSwapPostTradeOrderPlace',
        })
        marginMode, params = self.handle_margin_mode_and_params('createOrder', params)
        if marginMode is not None:
            method = 'privateSpotPostMarginOrderNew'
            marketType = 'margin'
        request = {}
        swap = (marketType == 'swap')
        isMarketOrder = (type == 'market')
        isLimitOrder = (type == 'limit')
        marketIdRequest = 'instrument_id' if swap else 'symbol'
        request[marketIdRequest] = market['id']
        postOnly = self.is_post_only(isMarketOrder, False, params)
        if swap:
            reduceOnly = self.safe_value(params, 'reduceOnly', False)
            timeInForce = self.safe_string(params, 'timeInForce')
            orderType = None
            if side == 'buy':
                requestType = 4 if (reduceOnly) else 1
                request['type'] = requestType
            else:
                requestType = 3 if (reduceOnly) else 2
                request['type'] = requestType
            if isLimitOrder:
                orderType = 0
            if timeInForce == 'FOK':
                orderType = 15 if isMarketOrder else 9
            elif timeInForce == 'IOC':
                orderType = 13 if isMarketOrder else 4
            elif (timeInForce == 'GTC') or (isMarketOrder):
                orderType = 14
            elif timeInForce == 'PO':
                postOnly = True
            if price is not None:
                request['price'] = self.price_to_precision(symbol, price)
            request['order_type'] = orderType
            request['size'] = amount  # swap orders require the amount to be the number of contracts
            params = self.omit(params, ['reduceOnly', 'timeInForce'])
        else:
            postOnly = 1 if (postOnly is True) else 2
            request['market'] = marketType
            suffix = ''
            if type == 'market':
                suffix = '_market'
            else:
                request['price'] = self.price_to_precision(symbol, price)
            request['type'] = side + suffix
            # limit orders require the amount in the base currency, market orders require the amount in the quote currency
            request['amount'] = self.amount_to_precision(symbol, amount)
        if postOnly:
            request['postOnly'] = postOnly
        query = self.omit(params, ['postOnly', 'post_only'])
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "order_id": "198361cecdc65f9c8c9bb2fa68faec40"
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": "1590873693003714560"
        #     }
        #
        result = self.parse_order(response, market)
        return self.extend(result, {
            'symbol': symbol,
            'type': type,
            'side': side,
            'amount': amount,
            'price': price,
        })

    def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#cancel-order
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#cancelorder
        :param str id: order id
        :param str|None symbol: not used by digifinex cancelOrder()
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        marketType = None
        marketType, params = self.handle_market_type_and_params('cancelOrder', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotPostSpotOrderCancel',
            'margin': 'privateSpotPostMarginOrderCancel',
            'swap': 'privateSwapPostTradeCancelOrder',
        })
        marginMode, query = self.handle_margin_mode_and_params('cancelOrder', params)
        if marginMode is not None:
            method = 'privateSpotPostMarginOrderCancel'
            marketType = 'margin'
        id = str(id)
        request = {
            'order_id': id,
        }
        if marketType == 'swap':
            self.check_required_symbol('cancelOrder', symbol)
            request['instrument_id'] = market['id']
        else:
            request['market'] = marketType
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "success": [
        #             "198361cecdc65f9c8c9bb2fa68faec40",
        #             "3fb0d98e51c18954f10d439a9cf57de0"
        #         ],
        #         "error": [
        #             "78a7104e3c65cc0c5a212a53e76d0205"
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": "1590923061186531328"
        #     }
        #
        if (marketType == 'spot') or (marketType == 'margin'):
            canceledOrders = self.safe_value(response, 'success', [])
            numCanceledOrders = len(canceledOrders)
            if numCanceledOrders != 1:
                raise OrderNotFound(self.id + ' cancelOrder() ' + id + ' not found')
        return response

    def cancel_orders(self, ids, symbol=None, params={}):
        """
        cancel multiple orders
        :param [str] ids: order ids
        :param str|None symbol: not used by digifinex cancelOrders()
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        defaultType = self.safe_string(self.options, 'defaultType', 'spot')
        orderType = self.safe_string(params, 'type', defaultType)
        params = self.omit(params, 'type')
        request = {
            'market': orderType,
            'order_id': ','.join(ids),
        }
        response = self.privateSpotPostCancelOrder(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "success": [
        #             "198361cecdc65f9c8c9bb2fa68faec40",
        #             "3fb0d98e51c18954f10d439a9cf57de0"
        #         ],
        #         "error": [
        #             "78a7104e3c65cc0c5a212a53e76d0205"
        #         ]
        #     }
        #
        canceledOrders = self.safe_value(response, 'success', [])
        numCanceledOrders = len(canceledOrders)
        if numCanceledOrders < 1:
            raise OrderNotFound(self.id + ' cancelOrders() error')
        return response

    def parse_order_status(self, status):
        statuses = {
            '0': 'open',
            '1': 'open',  # partially filled
            '2': 'closed',
            '3': 'canceled',
            '4': 'canceled',  # partially filled and canceled
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # spot: createOrder
        #
        #     {
        #         "code": 0,
        #         "order_id": "198361cecdc65f9c8c9bb2fa68faec40"
        #     }
        #
        # swap: createOrder
        #
        #     {
        #         "code": 0,
        #         "data": "1590873693003714560"
        #     }
        #
        # spot: fetchOrder, fetchOpenOrders, fetchOrders
        #
        #     {
        #         "symbol": "BTC_USDT",
        #         "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #         "created_date": 1562303547,
        #         "finished_date": 0,
        #         "price": 0.1,
        #         "amount": 1,
        #         "cash_amount": 1,
        #         "executed_amount": 0,
        #         "avg_price": 0,
        #         "status": 1,
        #         "type": "buy",
        #         "kind": "margin"
        #     }
        #
        # swap: fetchOrder, fetchOpenOrders, fetchOrders
        #
        #     {
        #         "order_id": "1590898207657824256",
        #         "instrument_id": "BTCUSDTPERP",
        #         "margin_mode": "crossed",
        #         "contract_val": "0.001",
        #         "type": 1,
        #         "order_type": 0,
        #         "price": "14000",
        #         "size": "6",
        #         "filled_qty": "0",
        #         "price_avg": "0",
        #         "fee": "0",
        #         "state": 0,
        #         "leverage": "20",
        #         "turnover": "0",
        #         "has_stop": 0,
        #         "insert_time": 1668134664828,
        #         "time_stamp": 1668134664828
        #     }
        #
        timestamp = None
        lastTradeTimestamp = None
        timeInForce = None
        type = None
        side = self.safe_string(order, 'type')
        marketId = self.safe_string_2(order, 'symbol', 'instrument_id')
        symbol = self.safe_symbol(marketId, market, '_')
        market = self.market(symbol)
        if market['type'] == 'swap':
            orderType = self.safe_integer(order, 'order_type')
            if (orderType == 9) or (orderType == 10) or (orderType == 11) or (orderType == 12) or (orderType == 15):
                timeInForce = 'FOK'
            elif (orderType == 1) or (orderType == 2) or (orderType == 3) or (orderType == 4) or (orderType == 13):
                timeInForce = 'IOC'
            elif (orderType == 6) or (orderType == 7) or (orderType == 8) or (orderType == 14):
                timeInForce = 'GTC'
            if (orderType == 0) or (orderType == 1) or (orderType == 4) or (orderType == 5) or (orderType == 9) or (orderType == 10):
                type = 'limit'
            else:
                type = 'market'
            if side == '1':
                side = 'open long'
            elif side == '2':
                side = 'open short'
            elif side == '3':
                side = 'close long'
            elif side == '4':
                side = 'close short'
            timestamp = self.safe_integer(order, 'insert_time')
            lastTradeTimestamp = self.safe_integer(order, 'time_stamp')
        else:
            timestamp = self.safe_timestamp(order, 'created_date')
            lastTradeTimestamp = self.safe_timestamp(order, 'finished_date')
            if side is not None:
                parts = side.split('_')
                numParts = len(parts)
                if numParts > 1:
                    side = parts[0]
                    type = parts[1]
                else:
                    type = 'limit'
        return self.safe_order({
            'info': order,
            'id': self.safe_string_2(order, 'order_id', 'data'),
            'clientOrderId': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': lastTradeTimestamp,
            'symbol': symbol,
            'type': type,
            'timeInForce': timeInForce,
            'postOnly': None,
            'side': side,
            'price': self.safe_number(order, 'price'),
            'stopPrice': None,
            'triggerPrice': None,
            'amount': self.safe_number_2(order, 'amount', 'size'),
            'filled': self.safe_number_2(order, 'executed_amount', 'filled_qty'),
            'remaining': None,
            'cost': None,
            'average': self.safe_number_2(order, 'avg_price', 'price_avg'),
            'status': self.parse_order_status(self.safe_string_2(order, 'status', 'state')),
            'fee': {
                'cost': self.safe_number(order, 'fee'),
            },
            'trades': None,
        }, market)

    def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#current-active-orders
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#openorder
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchOpenOrders', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotOrderCurrent',
            'margin': 'privateSpotGetMarginOrderCurrent',
            'swap': 'privateSwapGetTradeOpenOrders',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchOpenOrders', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginOrderCurrent'
            marketType = 'margin'
        request = {}
        swap = (marketType == 'swap')
        if swap:
            if since is not None:
                request['start_timestamp'] = since
            if limit is not None:
                request['limit'] = limit
        else:
            request['market'] = marketType
        if market is not None:
            marketIdRequest = 'instrument_id' if swap else 'symbol'
            request[marketIdRequest] = market['id']
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "order_id": "1590898207657824256",
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "margin_mode": "crossed",
        #                 "contract_val": "0.001",
        #                 "type": 1,
        #                 "order_type": 0,
        #                 "price": "14000",
        #                 "size": "6",
        #                 "filled_qty": "0",
        #                 "price_avg": "0",
        #                 "fee": "0",
        #                 "state": 0,
        #                 "leverage": "20",
        #                 "turnover": "0",
        #                 "has_stop": 0,
        #                 "insert_time": 1668134664828,
        #                 "time_stamp": 1668134664828
        #             },
        #             ...
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit)

    def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-all-orders-including-history-orders
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#historyorder
        :param str|None symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchOrders', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotOrderHistory',
            'margin': 'privateSpotGetMarginOrderHistory',
            'swap': 'privateSwapGetTradeHistoryOrders',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchOrders', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginOrderHistory'
            marketType = 'margin'
        request = {}
        if marketType == 'swap':
            if since is not None:
                request['start_timestamp'] = since
        else:
            request['market'] = marketType
            if since is not None:
                request['start_time'] = int(since / 1000)  # default 3 days from now, max 30 days
        if market is not None:
            marketIdRequest = 'instrument_id' if (marketType == 'swap') else 'symbol'
            request[marketIdRequest] = market['id']
        if limit is not None:
            request['limit'] = limit
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "order_id": "1590136768156405760",
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "margin_mode": "crossed",
        #                 "contract_val": "0.001",
        #                 "type": 1,
        #                 "order_type": 8,
        #                 "price": "18660.2",
        #                 "size": "1",
        #                 "filled_qty": "1",
        #                 "price_avg": "18514.5",
        #                 "fee": "0.00925725",
        #                 "state": 2,
        #                 "leverage": "20",
        #                 "turnover": "18.5145",
        #                 "has_stop": 0,
        #                 "insert_time": 1667953123526,
        #                 "time_stamp": 1667953123596
        #             },
        #             ...
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit)

    def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-order-status
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#orderinfo
        :param str id: order id
        :param str|None symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchOrder', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotOrder',
            'margin': 'privateSpotGetMarginOrder',
            'swap': 'privateSwapGetTradeOrderInfo',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchOrder', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginOrder'
            marketType = 'margin'
        request = {
            'order_id': id,
        }
        if marketType == 'swap':
            if market is not None:
                request['instrument_id'] = market['id']
        else:
            request['market'] = marketType
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "symbol": "BTC_USDT",
        #                 "order_id": "dd3164b333a4afa9d5730bb87f6db8b3",
        #                 "created_date": 1562303547,
        #                 "finished_date": 0,
        #                 "price": 0.1,
        #                 "amount": 1,
        #                 "cash_amount": 1,
        #                 "executed_amount": 0,
        #                 "avg_price": 0,
        #                 "status": 1,
        #                 "type": "buy",
        #                 "kind": "margin"
        #             }
        #         ]
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "order_id": "1590923061186531328",
        #             "instrument_id": "ETHUSDTPERP",
        #             "margin_mode": "crossed",
        #             "contract_val": "0.01",
        #             "type": 1,
        #             "order_type": 0,
        #             "price": "900",
        #             "size": "6",
        #             "filled_qty": "0",
        #             "price_avg": "0",
        #             "fee": "0",
        #             "state": 0,
        #             "leverage": "20",
        #             "turnover": "0",
        #             "has_stop": 0,
        #             "insert_time": 1668140590372,
        #             "time_stamp": 1668140590372
        #         }
        #     }
        #
        data = self.safe_value(response, 'data')
        order = data if (marketType == 'swap') else self.safe_value(data, 0)
        if order is None:
            raise OrderNotFound(self.id + ' fetchOrder() order ' + str(id) + ' not found')
        return self.parse_order(order, market)

    def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#customer-39-s-trades
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#historytrade
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        self.load_markets()
        market = None
        request = {}
        if symbol is not None:
            market = self.market(symbol)
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchMyTrades', market, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotMytrades',
            'margin': 'privateSpotGetMarginMytrades',
            'swap': 'privateSwapGetTradeHistoryTrades',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchMyTrades', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginMytrades'
            marketType = 'margin'
        if marketType == 'swap':
            if since is not None:
                request['start_timestamp'] = since
        else:
            request['market'] = marketType
            if since is not None:
                request['start_time'] = int(since / 1000)  # default 3 days from now, max 30 days
        marketIdRequest = 'instrument_id' if (marketType == 'swap') else 'symbol'
        if symbol is not None:
            request[marketIdRequest] = market['id']
        if limit is not None:
            request['limit'] = limit
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #      {
        #          "list":[
        #              {
        #                  "timestamp":1639506068,
        #                  "is_maker":false,
        #                  "id":"8975951332",
        #                  "amount":31.83,
        #                  "side":"sell_market",
        #                  "symbol":"DOGE_USDT",
        #                  "fee_currency":"USDT",
        #                  "fee":0.01163774826
        #                  ,"order_id":"32b169792f4a7a19e5907dc29fc123d4",
        #                  "price":0.182811
        #                }
        #             ],
        #           "code": 0
        #      }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "trade_id": "1590136768424841218",
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "order_id": "1590136768156405760",
        #                 "type": 1,
        #                 "order_type": 8,
        #                 "price": "18514.5",
        #                 "size": "1",
        #                 "fee": "0.00925725",
        #                 "close_profit": "0",
        #                 "leverage": "20",
        #                 "trade_type": 0,
        #                 "match_role": 1,
        #                 "trade_time": 1667953123562
        #             },
        #             ...
        #         ]
        #     }
        #
        responseRequest = 'data' if (marketType == 'swap') else 'list'
        data = self.safe_value(response, responseRequest, [])
        return self.parse_trades(data, market, since, limit)

    def parse_ledger_entry_type(self, type):
        types = {}
        return self.safe_string(types, type, type)

    def parse_ledger_entry(self, item, currency=None):
        #
        # spot and margin
        #
        #     {
        #         "currency_mark": "BTC",
        #         "type": 100234,
        #         "num": -10,
        #         "balance": 0.1,
        #         "time": 1546272000
        #     }
        #
        # swap
        #
        #     {
        #         "currency": "USDT",
        #         "finance_type": 17,
        #         "change": "-3.01",
        #         "timestamp": 1650809432000
        #     }
        #
        type = self.parse_ledger_entry_type(self.safe_string_2(item, 'type', 'finance_type'))
        code = self.safe_currency_code(self.safe_string_2(item, 'currency_mark', 'currency'), currency)
        amount = self.safe_number_2(item, 'num', 'change')
        after = self.safe_number(item, 'balance')
        timestamp = self.safe_timestamp(item, 'time')
        if timestamp is None:
            timestamp = self.safe_integer(item, 'timestamp')
        return {
            'info': item,
            'id': None,
            'direction': None,
            'account': None,
            'referenceId': None,
            'referenceAccount': None,
            'type': type,
            'currency': code,
            'amount': amount,
            'before': None,
            'after': after,
            'status': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'fee': None,
        }

    def fetch_ledger(self, code=None, since=None, limit=None, params={}):
        """
        fetch the history of changes, actions done by the user or operations that altered balance of the user
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#spot-margin-otc-financial-logs
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#bills
        :param str|None code: unified currency code, default is None
        :param int|None since: timestamp in ms of the earliest ledger entry, default is None
        :param int|None limit: max number of ledger entrys to return, default is None
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `ledger structure <https://docs.ccxt.com/en/latest/manual.html#ledger-structure>`
        """
        self.load_markets()
        request = {}
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchLedger', None, params)
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetSpotFinancelog',
            'margin': 'privateSpotGetMarginFinancelog',
            'swap': 'privateSwapGetAccountFinanceRecord',
        })
        marginMode, query = self.handle_margin_mode_and_params('fetchLedger', params)
        if marginMode is not None:
            method = 'privateSpotGetMarginFinancelog'
            marketType = 'margin'
        if marketType == 'swap':
            if since is not None:
                request['start_timestamp'] = since
        else:
            request['market'] = marketType
            if since is not None:
                request['start_time'] = int(since / 1000)  # default 3 days from now, max 30 days
        currencyIdRequest = 'currency' if (marketType == 'swap') else 'currency_mark'
        currency = None
        if code is not None:
            currency = self.currency(code)
            request[currencyIdRequest] = currency['id']
        if limit is not None:
            request['limit'] = limit
        response = getattr(self, method)(self.extend(request, query))
        #
        # spot and margin
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "total": 521,
        #             "finance": [
        #                 {
        #                     "currency_mark": "BTC",
        #                     "type": 100234,
        #                     "num": 28457,
        #                     "balance": 0.1,
        #                     "time": 1546272000
        #                 }
        #             ]
        #         }
        #     }
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "currency": "USDT",
        #                 "finance_type": 17,
        #                 "change": "3.01",
        #                 "timestamp": 1650809432000
        #             },
        #         ]
        #     }
        #
        ledger = None
        if marketType == 'swap':
            ledger = self.safe_value(response, 'data', [])
        else:
            data = self.safe_value(response, 'data', {})
            ledger = self.safe_value(data, 'finance', [])
        return self.parse_ledger(ledger, currency, since, limit)

    def parse_deposit_address(self, depositAddress, currency=None):
        #
        #     {
        #         "addressTag":"",
        #         "address":"0xf1104d9f8624f89775a3e9d480fc0e75a8ef4373",
        #         "currency":"USDT",
        #         "chain":"ERC20"
        #     }
        #
        address = self.safe_string(depositAddress, 'address')
        tag = self.safe_string(depositAddress, 'addressTag')
        currencyId = self.safe_string_upper(depositAddress, 'currency')
        code = self.safe_currency_code(currencyId)
        return {
            'info': depositAddress,
            'currency': code,
            'address': address,
            'tag': tag,
            'network': None,
        }

    def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
        }
        response = self.privateSpotGetDepositAddress(self.extend(request, params))
        #
        #     {
        #         "data":[
        #             {
        #                 "addressTag":"",
        #                 "address":"0xf1104d9f8624f89775a3e9d480fc0e75a8ef4373",
        #                 "currency":"USDT",
        #                 "chain":"ERC20"
        #             }
        #         ],
        #         "code":200
        #     }
        #
        data = self.safe_value(response, 'data', [])
        addresses = self.parse_deposit_addresses(data)
        address = self.safe_value(addresses, code)
        if address is None:
            raise InvalidAddress(self.id + ' fetchDepositAddress() did not return an address for ' + code + ' - create the deposit address in the user settings on the exchange website first.')
        return address

    def fetch_transactions_by_type(self, type, code=None, since=None, limit=None, params={}):
        self.load_markets()
        currency = None
        request = {
            # 'currency': currency['id'],
            # 'from': 'fromId',  # When direct is' prev ', from is 1, returning from old to new ascending, when direct is' next ', from is the ID of the most recent record, returned from the old descending order
            # 'size': 100,  # default 100, max 500
            # 'direct': 'prev',  # "prev" ascending, "next" descending
        }
        if code is not None:
            currency = self.currency(code)
            request['currency'] = currency['id']
        if limit is not None:
            request['size'] = min(500, limit)
        method = 'privateSpotGetDepositHistory' if (type == 'deposit') else 'privateSpotGetWithdrawHistory'
        response = getattr(self, method)(self.extend(request, params))
        #
        #     {
        #         "code": 200,
        #         "data": [
        #             {
        #                 "id": 1171,
        #                 "currency": "xrp",
        #                 "hash": "ed03094b84eafbe4bc16e7ef766ee959885ee5bcb265872baaa9c64e1cf86c2b",
        #                 "chain": "",
        #                 "amount": 7.457467,
        #                 "address": "rae93V8d2mdoUQHwBDBdM4NHCMehRJAsbm",
        #                 "memo": "100040",
        #                 "fee": 0,
        #                 "state": "safe",
        #                 "created_date": "2020-04-20 11:23:00",
        #                 "finished_date": "2020-04-20 13:23:00"
        #             },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_transactions(data, currency, since, limit, {'type': type})

    def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return self.fetch_transactions_by_type('deposit', code, since, limit, params)

    def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        return self.fetch_transactions_by_type('withdrawal', code, since, limit, params)

    def parse_transaction_status(self, status):
        # deposit state includes: 1(in deposit), 2(to be confirmed), 3(successfully deposited), 4(stopped)
        # withdrawal state includes: 1(application in progress), 2(to be confirmed), 3(completed), 4(rejected)
        statuses = {
            '1': 'pending',  # in Progress
            '2': 'pending',  # to be confirmed
            '3': 'ok',  # Completed
            '4': 'failed',  # Rejected
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, currency=None):
        #
        # withdraw
        #
        #     {
        #         "code": 200,
        #         "withdraw_id": 700
        #     }
        #
        # fetchDeposits, fetchWithdrawals
        #
        #     {
        #         "id": 1171,
        #         "currency": "xrp",
        #         "hash": "ed03094b84eafbe4bc16e7ef766ee959885ee5bcb265872baaa9c64e1cf86c2b",
        #         "chain": "",
        #         "amount": 7.457467,
        #         "address": "rae93V8d2mdoUQHwBDBdM4NHCMehRJAsbm",
        #         "memo": "100040",
        #         "fee": 0,
        #         "state": "safe",
        #         "created_date": "2020-04-20 11:23:00",
        #         "finished_date": "2020-04-20 13:23:00"
        #     }
        #
        id = self.safe_string_2(transaction, 'id', 'withdraw_id')
        address = self.safe_string(transaction, 'address')
        tag = self.safe_string(transaction, 'memo')
        txid = self.safe_string(transaction, 'hash')
        currencyId = self.safe_string_upper(transaction, 'currency')
        code = self.safe_currency_code(currencyId, currency)
        timestamp = self.parse8601(self.safe_string(transaction, 'created_date'))
        updated = self.parse8601(self.safe_string(transaction, 'finished_date'))
        status = self.parse_transaction_status(self.safe_string(transaction, 'state'))
        amount = self.safe_number(transaction, 'amount')
        feeCost = self.safe_number(transaction, 'fee')
        fee = None
        if feeCost is not None:
            fee = {'currency': code, 'cost': feeCost}
        network = self.safe_string(transaction, 'chain')
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'network': network,
            'address': address,
            'addressTo': address,
            'addressFrom': None,
            'tag': tag,
            'tagTo': tag,
            'tagFrom': None,
            'type': None,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': updated,
            'fee': fee,
        }

    def parse_transfer_status(self, status):
        statuses = {
            '0': 'ok',
        }
        return self.safe_string(statuses, status, status)

    def parse_transfer(self, transfer, currency=None):
        #
        # transfer
        #
        #     {
        #         "code": 0
        #     }
        #
        # fetchTransfers
        #
        #     {
        #         "transfer_id": 130524,
        #         "type": 1,
        #         "currency": "USDT",
        #         "amount": "24",
        #         "timestamp": 1666505659000
        #     }
        #
        fromAccount = None
        toAccount = None
        type = self.safe_integer(transfer, 'type')
        if type == 1:
            fromAccount = 'spot'
            toAccount = 'swap'
        elif type == 2:
            fromAccount = 'swap'
            toAccount = 'spot'
        timestamp = self.safe_integer(transfer, 'timestamp')
        return {
            'info': transfer,
            'id': self.safe_string(transfer, 'transfer_id'),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'currency': self.safe_currency_code(self.safe_string(transfer, 'currency'), currency),
            'amount': self.safe_number(transfer, 'amount'),
            'fromAccount': fromAccount,
            'toAccount': toAccount,
            'status': self.parse_transfer_status(self.safe_string(transfer, 'code')),
        }

    def transfer(self, code, amount, fromAccount, toAccount, params={}):
        """
        transfer currency internally between wallets on the same account
        :param str code: unified currency code
        :param float amount: amount to transfer
        :param str fromAccount: account to transfer from
        :param str toAccount: account to transfer to
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `transfer structure <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        accountsByType = self.safe_value(self.options, 'accountsByType', {})
        fromId = self.safe_string(accountsByType, fromAccount, fromAccount)
        toId = self.safe_string(accountsByType, toAccount, toAccount)
        request = {
            'currency_mark': currency['id'],
            'num': self.currency_to_precision(code, amount),
            'from': fromId,  # 1 = SPOT, 2 = MARGIN, 3 = OTC
            'to': toId,  # 1 = SPOT, 2 = MARGIN, 3 = OTC
        }
        response = self.privateSpotPostTransfer(self.extend(request, params))
        #
        #     {
        #         "code": 0
        #     }
        #
        transfer = self.parse_transfer(response, currency)
        return self.extend(transfer, {
            'amount': amount,
            'currency': code,
            'fromAccount': fromAccount,
            'toAccount': toAccount,
        })

    def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        self.check_address(address)
        self.load_markets()
        currency = self.currency(code)
        request = {
            # 'chain': 'ERC20', 'OMNI', 'TRC20',  # required for USDT
            'address': address,
            'amount': self.currency_to_precision(code, amount),
            'currency': currency['id'],
        }
        if tag is not None:
            request['memo'] = tag
        response = self.privateSpotPostWithdrawNew(self.extend(request, params))
        #
        #     {
        #         "code": 200,
        #         "withdraw_id": 700
        #     }
        #
        return self.parse_transaction(response, currency)

    def fetch_borrow_interest(self, code=None, symbol=None, since=None, limit=None, params={}):
        self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        response = self.privateSpotGetMarginPositions(self.extend(request, params))
        #
        #     {
        #         "margin": "45.71246418952618",
        #         "code": 0,
        #         "margin_rate": "7.141978570340037",
        #         "positions": [
        #             {
        #                 "amount": 0.0006103,
        #                 "side": "go_long",
        #                 "entry_price": 31428.72,
        #                 "liquidation_rate": 0.3,
        #                 "liquidation_price": 10225.335481159,
        #                 "unrealized_roe": -0.0076885829266987,
        #                 "symbol": "BTC_USDT",
        #                 "unrealized_pnl": -0.049158102631999,
        #                 "leverage_ratio": 3
        #             }
        #         ],
        #         "unrealized_pnl": "-0.049158102631998504"
        #     }
        #
        rows = self.safe_value(response, 'positions')
        interest = self.parse_borrow_interests(rows, market)
        return self.filter_by_currency_since_limit(interest, code, since, limit)

    def parse_borrow_interest(self, info, market):
        #
        #     {
        #         "amount": 0.0006103,
        #         "side": "go_long",
        #         "entry_price": 31428.72,
        #         "liquidation_rate": 0.3,
        #         "liquidation_price": 10225.335481159,
        #         "unrealized_roe": -0.0076885829266987,
        #         "symbol": "BTC_USDT",
        #         "unrealized_pnl": -0.049158102631999,
        #         "leverage_ratio": 3
        #     }
        #
        symbol = self.safe_string(info, 'symbol')
        amountString = self.safe_string(info, 'amount')
        leverageString = self.safe_string(info, 'leverage_ratio')
        amountInvested = Precise.string_div(amountString, leverageString)
        amountBorrowed = Precise.string_sub(amountString, amountInvested)
        currency = None if (market is None) else market['base']
        return {
            'account': self.safe_symbol(symbol, market),
            'currency': currency,
            'interest': None,
            'interestRate': 0.001,  # all interest rates on digifinex are 0.1%
            'amountBorrowed': self.parse_number(amountBorrowed),
            'timestamp': None,
            'datetime': None,
            'info': info,
        }

    def fetch_borrow_rate(self, code, params={}):
        self.load_markets()
        request = {}
        response = self.privateSpotGetMarginAssets(self.extend(request, params))
        #
        #     {
        #         "list": [
        #             {
        #                 "valuation_rate": 1,
        #                 "total": 1.92012186174,
        #                 "free": 1.92012186174,
        #                 "currency": "USDT"
        #             },
        #         ],
        #         "total": 45.133305540922,
        #         "code": 0,
        #         "unrealized_pnl": 0,
        #         "free": 45.133305540922,
        #         "equity": 45.133305540922
        #     }
        #
        data = self.safe_value(response, 'list', [])
        result = []
        for i in range(0, len(data)):
            entry = data[i]
            if self.safe_string(entry, 'currency') == code:
                result = entry
        currency = self.safe_string(result, 'currency')
        return self.parse_borrow_rate(result, currency)

    def fetch_borrow_rates(self, params={}):
        """
        fetch the borrow interest rates of all currencies
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a list of `borrow rate structures <https://docs.ccxt.com/en/latest/manual.html#borrow-rate-structure>`
        """
        self.load_markets()
        response = self.privateSpotGetMarginAssets(params)
        #
        #     {
        #         "list": [
        #             {
        #                 "valuation_rate": 1,
        #                 "total": 1.92012186174,
        #                 "free": 1.92012186174,
        #                 "currency": "USDT"
        #             },
        #         ],
        #         "total": 45.133305540922,
        #         "code": 0,
        #         "unrealized_pnl": 0,
        #         "free": 45.133305540922,
        #         "equity": 45.133305540922
        #     }
        #
        result = self.safe_value(response, 'list')
        return self.parse_borrow_rates(result, 'currency')

    def parse_borrow_rate(self, info, currency=None):
        #
        #     {
        #         "valuation_rate": 1,
        #         "total": 1.92012186174,
        #         "free": 1.92012186174,
        #         "currency": "USDT"
        #     }
        #
        timestamp = self.milliseconds()
        currencyId = self.safe_string(info, 'currency')
        return {
            'currency': self.safe_currency_code(currencyId, currency),
            'rate': 0.001,  # all interest rates on digifinex are 0.1%
            'period': 86400000,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'info': info,
        }

    def parse_borrow_rates(self, info, codeKey):
        #
        #     {
        #         "valuation_rate": 1,
        #         "total": 1.92012186174,
        #         "free": 1.92012186174,
        #         "currency": "USDT"
        #     },
        #
        result = {}
        for i in range(0, len(info)):
            item = info[i]
            currency = self.safe_string(item, codeKey)
            code = self.safe_currency_code(currency)
            borrowRate = self.parse_borrow_rate(item, currency)
            result[code] = borrowRate
        return result

    def fetch_funding_rate(self, symbol, params={}):
        """
        fetch the current funding rate
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#currentfundingrate
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `funding rate structure <https://docs.ccxt.com/en/latest/manual.html#funding-rate-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadSymbol(self.id + ' fetchFundingRate() supports swap contracts only')
        request = {
            'instrument_id': market['id'],
        }
        response = self.publicSwapGetPublicFundingRate(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "funding_rate": "-0.00012",
        #             "funding_time": 1662710400000,
        #             "next_funding_rate": "0.0001049907085171607",
        #             "next_funding_time": 1662739200000
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_funding_rate(data, market)

    def parse_funding_rate(self, contract, market=None):
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "funding_rate": "-0.00012",
        #         "funding_time": 1662710400000,
        #         "next_funding_rate": "0.0001049907085171607",
        #         "next_funding_time": 1662739200000
        #     }
        #
        marketId = self.safe_string(contract, 'instrument_id')
        timestamp = self.safe_integer(contract, 'funding_time')
        nextTimestamp = self.safe_integer(contract, 'next_funding_time')
        return {
            'info': contract,
            'symbol': self.safe_symbol(marketId, market),
            'markPrice': None,
            'indexPrice': None,
            'interestRate': None,
            'estimatedSettlePrice': None,
            'timestamp': None,
            'datetime': None,
            'fundingRate': self.safe_string(contract, 'funding_rate'),
            'fundingTimestamp': timestamp,
            'fundingDatetime': self.iso8601(timestamp),
            'nextFundingRate': self.safe_string(contract, 'next_funding_rate'),
            'nextFundingTimestamp': nextTimestamp,
            'nextFundingDatetime': self.iso8601(nextTimestamp),
            'previousFundingRate': None,
            'previousFundingTimestamp': None,
            'previousFundingDatetime': None,
        }

    def fetch_funding_rate_history(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches historical funding rate prices
        :param str|None symbol: unified symbol of the market to fetch the funding rate history for
        :param int|None since: timestamp in ms of the earliest funding rate to fetch
        :param int|None limit: the maximum amount of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>` to fetch
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `funding rate structures <https://docs.ccxt.com/en/latest/manual.html?#funding-rate-history-structure>`
        """
        self.check_required_symbol('fetchFundingRateHistory', symbol)
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadSymbol(self.id + ' fetchFundingRateHistory() supports swap contracts only')
        request = {
            'instrument_id': market['id'],
        }
        if since is not None:
            request['start_timestamp'] = since
        if limit is not None:
            request['limit'] = limit
        response = self.publicSwapGetPublicFundingRateHistory(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "funding_rates": [
        #                 {
        #                     "rate": "-0.00375",
        #                     "time": 1607673600000
        #                 },
        #                 ...
        #             ]
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        result = self.safe_value(data, 'funding_rates', [])
        rates = []
        for i in range(0, len(result)):
            entry = result[i]
            marketId = self.safe_string(data, 'instrument_id')
            symbol = self.safe_symbol(marketId)
            timestamp = self.safe_integer(entry, 'time')
            rates.append({
                'info': entry,
                'symbol': symbol,
                'fundingRate': self.safe_string(entry, 'rate'),
                'timestamp': timestamp,
                'datetime': self.iso8601(timestamp),
            })
        sorted = self.sort_by(rates, 'timestamp')
        return self.filter_by_symbol_since_limit(sorted, symbol, since, limit)

    def fetch_trading_fee(self, symbol, params={}):
        """
        fetch the trading fees for a market
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#tradingfee
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `fee structure <https://docs.ccxt.com/en/latest/manual.html#fee-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadRequest(self.id + ' fetchTradingFee() supports swap markets only')
        request = {
            'instrument_id': market['id'],
        }
        response = self.privateSwapGetAccountTradingFeeRate(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "taker_fee_rate": "0.0005",
        #             "maker_fee_rate": "0.0003"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_trading_fee(data, market)

    def parse_trading_fee(self, fee, market=None):
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "taker_fee_rate": "0.0005",
        #         "maker_fee_rate": "0.0003"
        #     }
        #
        marketId = self.safe_string(fee, 'instrument_id')
        symbol = self.safe_symbol(marketId, market)
        return {
            'info': fee,
            'symbol': symbol,
            'maker': self.safe_number(fee, 'maker_fee_rate'),
            'taker': self.safe_number(fee, 'taker_fee_rate'),
        }

    def fetch_positions(self, symbols=None, params={}):
        """
        fetch all open positions
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#margin-positions
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#positions
        :param [str]|None symbols: list of unified market symbols
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `position structures <https://docs.ccxt.com/en/latest/manual.html#position-structure>`
        """
        self.load_markets()
        symbols = self.market_symbols(symbols)
        request = {}
        market = None
        marketType = None
        if symbols is not None:
            symbol = None
            if isinstance(symbols, list):
                symbolsLength = len(symbols)
                if symbolsLength > 1:
                    raise BadRequest(self.id + ' fetchPositions() symbols argument cannot contain more than 1 symbol')
                symbol = symbols[0]
            else:
                symbol = symbols
            market = self.market(symbol)
        marketType, params = self.handle_market_type_and_params('fetchPositions', market, params)
        marginMode, query = self.handle_margin_mode_and_params('fetchPositions', params)
        if marginMode is not None:
            marketType = 'margin'
        if market is not None:
            marketIdRequest = 'instrument_id' if (marketType == 'swap') else 'symbol'
            request[marketIdRequest] = market['id']
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetMarginPositions',
            'margin': 'privateSpotGetMarginPositions',
            'swap': 'privateSwapGetAccountPositions',
        })
        response = getattr(self, method)(self.extend(request, query))
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "margin_mode": "crossed",
        #                 "avail_position": "1",
        #                 "avg_cost": "18369.3",
        #                 "last": "18404.7",
        #                 "leverage": "20",
        #                 "liquidation_price": "451.12820512820264",
        #                 "maint_margin_ratio": "0.005",
        #                 "margin": "0.918465",
        #                 "position": "1",
        #                 "realized_pnl": "0",
        #                 "unrealized_pnl": "0.03410000000000224",
        #                 "unrealized_pnl_rate": "0.03712716325608732",
        #                 "side": "long",
        #                 "open_outstanding": "0",
        #                 "risk_score": "0.495049504950495",
        #                 "margin_ratio": "0.4029464788983229",
        #                 "timestamp": 1667960497145
        #             },
        #             ...
        #         ]
        #     }
        #
        # margin
        #
        #     {
        #         "margin": "77.71534772983289",
        #         "code": 0,
        #         "margin_rate": "10.284503769497306",
        #         "positions": [
        #             {
        #                 "amount": 0.0010605,
        #                 "side": "go_long",
        #                 "entry_price": 18321.39,
        #                 "liquidation_rate": 0.3,
        #                 "liquidation_price": -52754.371758471,
        #                 "unrealized_roe": -0.002784390267332,
        #                 "symbol": "BTC_USDT",
        #                 "unrealized_pnl": -0.010820048189999,
        #                 "leverage_ratio": 5
        #             },
        #             ...
        #         ],
        #         "unrealized_pnl": "-0.10681600018999979"
        #     }
        #
        positionRequest = 'data' if (marketType == 'swap') else 'positions'
        positions = self.safe_value(response, positionRequest, [])
        result = []
        for i in range(0, len(positions)):
            result.append(self.parse_position(positions[i], market))
        return self.filter_by_array(result, 'symbol', symbols, False)

    def fetch_position(self, symbol, params={}):
        """
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#margin-positions
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#positions
        fetch data on a single open contract trade position
        :param str symbol: unified market symbol of the market the position is held in
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `position structure <https://docs.ccxt.com/en/latest/manual.html#position-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {}
        marketType = None
        marketType, params = self.handle_market_type_and_params('fetchPosition', market, params)
        marginMode, query = self.handle_margin_mode_and_params('fetchPosition', params)
        if marginMode is not None:
            marketType = 'margin'
        method = self.get_supported_mapping(marketType, {
            'spot': 'privateSpotGetMarginPositions',
            'margin': 'privateSpotGetMarginPositions',
            'swap': 'privateSwapGetAccountPositions',
        })
        marketIdRequest = 'instrument_id' if (marketType == 'swap') else 'symbol'
        request[marketIdRequest] = market['id']
        response = getattr(self, method)(self.extend(request, query))
        #
        # swap
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "margin_mode": "crossed",
        #                 "avail_position": "1",
        #                 "avg_cost": "18369.3",
        #                 "last": "18388.9",
        #                 "leverage": "20",
        #                 "liquidation_price": "383.38712921065553",
        #                 "maint_margin_ratio": "0.005",
        #                 "margin": "0.918465",
        #                 "position": "1",
        #                 "realized_pnl": "0",
        #                 "unrealized_pnl": "0.021100000000004115",
        #                 "unrealized_pnl_rate": "0.02297311274790451",
        #                 "side": "long",
        #                 "open_outstanding": "0",
        #                 "risk_score": "0.4901960784313725",
        #                 "margin_ratio": "0.40486964045976204",
        #                 "timestamp": 1667960241758
        #             }
        #         ]
        #     }
        #
        # margin
        #
        #     {
        #         "margin": "77.71534772983289",
        #         "code": 0,
        #         "margin_rate": "10.284503769497306",
        #         "positions": [
        #             {
        #                 "amount": 0.0010605,
        #                 "side": "go_long",
        #                 "entry_price": 18321.39,
        #                 "liquidation_rate": 0.3,
        #                 "liquidation_price": -52754.371758471,
        #                 "unrealized_roe": -0.002784390267332,
        #                 "symbol": "BTC_USDT",
        #                 "unrealized_pnl": -0.010820048189999,
        #                 "leverage_ratio": 5
        #             }
        #         ],
        #         "unrealized_pnl": "-0.10681600018999979"
        #     }
        #
        dataRequest = 'data' if (marketType == 'swap') else 'positions'
        data = self.safe_value(response, dataRequest, [])
        position = self.parse_position(data[0], market)
        if marketType == 'swap':
            return position
        else:
            return self.extend(position, {
                'collateral': self.safe_number(response, 'margin'),
                'marginRatio': self.safe_number(response, 'margin_rate'),
            })

    def parse_position(self, position, market=None):
        #
        # swap
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "margin_mode": "crossed",
        #         "avail_position": "1",
        #         "avg_cost": "18369.3",
        #         "last": "18388.9",
        #         "leverage": "20",
        #         "liquidation_price": "383.38712921065553",
        #         "maint_margin_ratio": "0.005",
        #         "margin": "0.918465",
        #         "position": "1",
        #         "realized_pnl": "0",
        #         "unrealized_pnl": "0.021100000000004115",
        #         "unrealized_pnl_rate": "0.02297311274790451",
        #         "side": "long",
        #         "open_outstanding": "0",
        #         "risk_score": "0.4901960784313725",
        #         "margin_ratio": "0.40486964045976204",
        #         "timestamp": 1667960241758
        #     }
        #
        # margin
        #
        #     {
        #         "amount": 0.0010605,
        #         "side": "go_long",
        #         "entry_price": 18321.39,
        #         "liquidation_rate": 0.3,
        #         "liquidation_price": -52754.371758471,
        #         "unrealized_roe": -0.002784390267332,
        #         "symbol": "BTC_USDT",
        #         "unrealized_pnl": -0.010820048189999,
        #         "leverage_ratio": 5
        #     }
        #
        marketId = self.safe_string_2(position, 'instrument_id', 'symbol')
        market = self.safe_market(marketId, market)
        symbol = market['symbol']
        marginMode = self.safe_string(position, 'margin_mode')
        if marginMode is not None:
            marginMode = 'cross' if (marginMode == 'crossed') else 'isolated'
        else:
            marginMode = 'crossed'
        timestamp = self.safe_integer(position, 'timestamp')
        side = self.safe_string(position, 'side')
        if side == 'go_long':
            side = 'long'
        elif side == 'go_short':
            side = 'short'
        return {
            'info': position,
            'id': None,
            'symbol': symbol,
            'notional': self.safe_number(position, 'amount'),
            'marginMode': marginMode,
            'liquidationPrice': self.safe_number(position, 'liquidation_price'),
            'entryPrice': self.safe_number_2(position, 'avg_cost', 'entry_price'),
            'unrealizedPnl': self.safe_number(position, 'unrealized_pnl'),
            'contracts': self.safe_number(position, 'avail_position'),
            'contractSize': self.safe_number(market, 'contractSize'),
            'markPrice': self.safe_number(position, 'last'),
            'side': side,
            'hedged': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'maintenanceMargin': self.safe_number(position, 'margin'),
            'maintenanceMarginPercentage': self.safe_number(position, 'maint_margin_ratio'),
            'collateral': None,
            'initialMargin': None,
            'initialMarginPercentage': None,
            'leverage': self.safe_number_2(position, 'leverage', 'leverage_ratio'),
            'marginRatio': self.safe_number(position, 'margin_ratio'),
            'percentage': None,
        }

    def set_leverage(self, leverage, symbol=None, params={}):
        """
        set the level of leverage for a market
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#setleverage
        :param float leverage: the rate of leverage
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :param str|None params['marginMode']: either 'cross' or 'isolated', default is cross
        :param str|None params['side']: either 'long' or 'short', required for isolated markets only
        :returns dict: response from the exchange
        """
        self.load_markets()
        self.check_required_symbol('setLeverage', symbol)
        market = self.market(symbol)
        if market['type'] != 'swap':
            raise BadSymbol(self.id + ' setLeverage() supports swap contracts only')
        if (leverage < 1) or (leverage > 100):
            raise BadRequest(self.id + ' leverage should be between 1 and 100')
        request = {
            'instrument_id': market['id'],
            'leverage': leverage,
        }
        defaultMarginMode = self.safe_string_2(self.options, 'marginMode', 'defaultMarginMode')
        marginMode = self.safe_string_lower_2(params, 'marginMode', 'defaultMarginMode', defaultMarginMode)
        if marginMode is not None:
            marginMode = 'crossed' if (marginMode == 'cross') else 'isolated'
            request['margin_mode'] = marginMode
            params = self.omit(params, ['marginMode', 'defaultMarginMode'])
        if marginMode == 'isolated':
            side = self.safe_string(params, 'side')
            if side is not None:
                request['side'] = side
                params = self.omit(params, 'side')
            else:
                self.check_required_argument('setLeverage', side, 'side', ['long', 'short'])
        return self.privateSwapPostAccountLeverage(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "leverage": 30,
        #             "margin_mode": "crossed",
        #             "side": "both"
        #         }
        #     }
        #

    def fetch_transfers(self, code=None, since=None, limit=None, params={}):
        """
        fetch the transfer history, only transfers between spot and swap accounts are supported
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#transferrecord
        :param str|None code: unified currency code of the currency transferred
        :param int|None since: the earliest time in ms to fetch transfers for
        :param int|None limit: the maximum number of  transfers to retrieve
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns [dict]: a list of `transfer structures <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        self.load_markets()
        currency = None
        request = {}
        if code is not None:
            currency = self.safe_currency_code(code)
            request['currency'] = currency['id']
        if since is not None:
            request['start_timestamp'] = since
        if limit is not None:
            request['limit'] = limit  # default 20 max 100
        response = self.privateSwapGetAccountTransferRecord(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "transfer_id": 130524,
        #                 "type": 1,
        #                 "currency": "USDT",
        #                 "amount": "24",
        #                 "timestamp": 1666505659000
        #             },
        #             ...
        #         ]
        #     }
        #
        transfers = self.safe_value(response, 'data', [])
        return self.parse_transfers(transfers, currency, since, limit)

    def fetch_leverage_tiers(self, symbols=None, params={}):
        """
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#instruments
        retrieve information on the maximum leverage, for different trade sizes
        :param [str]|None symbols: a list of unified market symbols
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a dictionary of `leverage tiers structures <https://docs.ccxt.com/en/latest/manual.html#leverage-tiers-structure>`, indexed by market symbols
        """
        self.load_markets()
        response = self.publicSwapGetPublicInstruments(params)
        #
        #     {
        #         "code": 0,
        #         "data": [
        #             {
        #                 "instrument_id": "BTCUSDTPERP",
        #                 "type": "REAL",
        #                 "contract_type": "PERPETUAL",
        #                 "base_currency": "BTC",
        #                 "quote_currency": "USDT",
        #                 "clear_currency": "USDT",
        #                 "contract_value": "0.001",
        #                 "contract_value_currency": "BTC",
        #                 "is_inverse": False,
        #                 "is_trading": True,
        #                 "status": "ONLINE",
        #                 "price_precision": 1,
        #                 "tick_size": "0.1",
        #                 "min_order_amount": 1,
        #                 "open_max_limits": [
        #                     {
        #                         "leverage": "50",
        #                         "max_limit": "1000000"
        #                     },
        #                 ]
        #             },
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        symbols = self.market_symbols(symbols)
        return self.parse_leverage_tiers(data, symbols, 'symbol')

    def parse_leverage_tiers(self, response, symbols=None, marketIdKey=None):
        #
        #     [
        #         {
        #             "instrument_id": "BTCUSDTPERP",
        #             "type": "REAL",
        #             "contract_type": "PERPETUAL",
        #             "base_currency": "BTC",
        #             "quote_currency": "USDT",
        #             "clear_currency": "USDT",
        #             "contract_value": "0.001",
        #             "contract_value_currency": "BTC",
        #             "is_inverse": False,
        #             "is_trading": True,
        #             "status": "ONLINE",
        #             "price_precision": 1,
        #             "tick_size": "0.1",
        #             "min_order_amount": 1,
        #             "open_max_limits": [
        #                 {
        #                     "leverage": "50",
        #                     "max_limit": "1000000"
        #                 }
        #             ]
        #         },
        #     ]
        #
        tiers = {}
        result = {}
        for i in range(0, len(response)):
            entry = response[i]
            marketId = self.safe_string(entry, 'instrument_id')
            market = self.safe_market(marketId)
            symbol = self.safe_symbol(marketId, market)
            symbolsLength = 0
            tiers[symbol] = self.parse_market_leverage_tiers(response[i], market)
            if symbols is not None:
                symbolsLength = len(symbols)
                if self.in_array(symbol, symbols):
                    result[symbol] = self.parse_market_leverage_tiers(response[i], market)
            if symbol is not None and (symbolsLength == 0 or self.in_array(symbols, symbol)):
                result[symbol] = self.parse_market_leverage_tiers(response[i], market)
        return result

    def fetch_market_leverage_tiers(self, symbol, params={}):
        """
        see https://docs.digifinex.com/en-ww/swap/v2/rest.html#instrument
        retrieve information on the maximum leverage, for different trade sizes for a single market
        :param str symbol: unified market symbol
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a `leverage tiers structure <https://docs.ccxt.com/en/latest/manual.html#leverage-tiers-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        if not market['swap']:
            raise BadRequest(self.id + ' fetchMarketLeverageTiers() supports swap markets only')
        request = {
            'instrument_id': market['id'],
        }
        response = self.publicSwapGetPublicInstrument(self.extend(request, params))
        #
        #     {
        #         "code": 0,
        #         "data": {
        #             "instrument_id": "BTCUSDTPERP",
        #             "type": "REAL",
        #             "contract_type": "PERPETUAL",
        #             "base_currency": "BTC",
        #             "quote_currency": "USDT",
        #             "clear_currency": "USDT",
        #             "contract_value": "0.001",
        #             "contract_value_currency": "BTC",
        #             "is_inverse": False,
        #             "is_trading": True,
        #             "status": "ONLINE",
        #             "price_precision": 1,
        #             "tick_size": "0.1",
        #             "min_order_amount": 1,
        #             "open_max_limits": [
        #                 {
        #                     "leverage": "50",
        #                     "max_limit": "1000000"
        #                 }
        #             ]
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.parse_market_leverage_tiers(data, market)

    def parse_market_leverage_tiers(self, info, market):
        #
        #     {
        #         "instrument_id": "BTCUSDTPERP",
        #         "type": "REAL",
        #         "contract_type": "PERPETUAL",
        #         "base_currency": "BTC",
        #         "quote_currency": "USDT",
        #         "clear_currency": "USDT",
        #         "contract_value": "0.001",
        #         "contract_value_currency": "BTC",
        #         "is_inverse": False,
        #         "is_trading": True,
        #         "status": "ONLINE",
        #         "price_precision": 1,
        #         "tick_size": "0.1",
        #         "min_order_amount": 1,
        #         "open_max_limits": [
        #             {
        #                 "leverage": "50",
        #                 "max_limit": "1000000"
        #             }
        #         ]
        #     }
        #
        tiers = []
        brackets = self.safe_value(info, 'open_max_limits', {})
        for i in range(0, len(brackets)):
            tier = brackets[i]
            marketId = self.safe_string(info, 'instrument_id')
            market = self.safe_market(marketId)
            tiers.append({
                'tier': self.sum(i, 1),
                'currency': market['settle'],
                'minNotional': None,
                'maxNotional': self.safe_number(tier, 'max_limit'),
                'maintenanceMarginRate': None,
                'maxLeverage': self.safe_number(tier, 'leverage'),
                'info': tier,
            })
        return tiers

    def handle_margin_mode_and_params(self, methodName, params={}, defaultValue=None):
        """
         * @ignore
        marginMode specified by params["marginMode"], self.options["marginMode"], self.options["defaultMarginMode"], params["margin"] = True or self.options["defaultType"] = 'margin'
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [str|None, dict]: the marginMode in lowercase
        """
        defaultType = self.safe_string(self.options, 'defaultType')
        isMargin = self.safe_value(params, 'margin', False)
        marginMode = None
        marginMode, params = super(digifinex, self).handle_margin_mode_and_params(methodName, params, defaultValue)
        if marginMode is not None:
            if marginMode != 'cross':
                raise NotSupported(self.id + ' only cross margin is supported')
        else:
            if (defaultType == 'margin') or (isMargin is True):
                marginMode = 'cross'
        return [marginMode, params]

    def fetch_deposit_withdraw_fees(self, codes=None, params={}):
        """
        fetch deposit and withdraw fees
        see https://docs.digifinex.com/en-ww/spot/v3/rest.html#get-currency-deposit-and-withdrawal-information
        :param [str]|None codes: not used by fetchDepositWithdrawFees()
        :param dict params: extra parameters specific to the digifinex api endpoint
        :returns dict: a list of `fee structures <https://docs.ccxt.com/en/latest/manual.html#fee-structure>`
        """
        self.load_markets()
        response = self.publicSpotGetCurrencies(params)
        #
        #   {
        #       "data": [
        #           {
        #               "deposit_status": 0,
        #               "min_withdraw_fee": 5,
        #               "withdraw_fee_currency": "USDT",
        #               "chain": "OMNI",
        #               "withdraw_fee_rate": 0,
        #               "min_withdraw_amount": 10,
        #               "currency": "USDT",
        #               "withdraw_status": 0,
        #               "min_deposit_amount": 10
        #           },
        #           {
        #               "deposit_status": 1,
        #               "min_withdraw_fee": 5,
        #               "withdraw_fee_currency": "USDT",
        #               "chain": "ERC20",
        #               "withdraw_fee_rate": 0,
        #               "min_withdraw_amount": 10,
        #               "currency": "USDT",
        #               "withdraw_status": 1,
        #               "min_deposit_amount": 10
        #           },
        #       ],
        #       "code": 200,
        #   }
        #
        data = self.safe_value(response, 'data')
        return self.parse_deposit_withdraw_fees(data, codes)

    def parse_deposit_withdraw_fees(self, response, codes=None, currencyIdKey=None):
        #
        #     [
        #         {
        #             "deposit_status": 0,
        #             "min_withdraw_fee": 5,
        #             "withdraw_fee_currency": "USDT",
        #             "chain": "OMNI",
        #             "withdraw_fee_rate": 0,
        #             "min_withdraw_amount": 10,
        #             "currency": "USDT",
        #             "withdraw_status": 0,
        #             "min_deposit_amount": 10
        #         },
        #         {
        #             "deposit_status": 1,
        #             "min_withdraw_fee": 5,
        #             "withdraw_fee_currency": "USDT",
        #             "chain": "ERC20",
        #             "withdraw_fee_rate": 0,
        #             "min_withdraw_amount": 10,
        #             "currency": "USDT",
        #             "withdraw_status": 1,
        #             "min_deposit_amount": 10
        #         },
        #     ]
        #
        depositWithdrawFees = {}
        codes = self.market_codes(codes)
        for i in range(0, len(response)):
            entry = response[i]
            currencyId = self.safe_string(entry, 'currency')
            code = self.safe_currency_code(currencyId)
            if (codes is None) or (self.in_array(code, codes)):
                depositWithdrawFee = self.safe_value(depositWithdrawFees, code)
                if depositWithdrawFee is None:
                    depositWithdrawFees[code] = self.deposit_withdraw_fee({})
                    depositWithdrawFees[code]['info'] = []
                depositWithdrawFees[code]['info'].append(entry)
                networkId = self.safe_string(entry, 'chain')
                withdrawFee = self.safe_value(entry, 'min_withdraw_fee')
                withdrawResult = {
                    'fee': withdrawFee,
                    'percentage': False if (withdrawFee is not None) else None,
                }
                depositResult = {
                    'fee': None,
                    'percentage': None,
                }
                if networkId is not None:
                    networkCode = self.network_id_to_code(networkId)
                    depositWithdrawFees[code]['networks'][networkCode] = {
                        'withdraw': withdrawResult,
                        'deposit': depositResult,
                    }
                else:
                    depositWithdrawFees[code]['withdraw'] = withdrawResult
                    depositWithdrawFees[code]['deposit'] = depositResult
        depositWithdrawCodes = list(depositWithdrawFees.keys())
        for i in range(0, len(depositWithdrawCodes)):
            code = depositWithdrawCodes[i]
            currency = self.currency(code)
            depositWithdrawFees[code] = self.assign_default_deposit_withdraw_fees(depositWithdrawFees[code], currency)
        return depositWithdrawFees

    def sign(self, path, api=[], method='GET', params={}, headers=None, body=None):
        signed = api[0] == 'private'
        endpoint = api[1]
        pathPart = '/v3' if (endpoint == 'spot') else '/swap/v2'
        request = '/' + self.implode_params(path, params)
        payload = pathPart + request
        url = self.urls['api']['rest'] + payload
        query = self.omit(params, self.extract_params(path))
        urlencoded = self.urlencode(self.keysort(query))
        if signed:
            auth = None
            nonce = None
            if pathPart == '/swap/v2':
                nonce = str(self.milliseconds())
                auth = nonce + method + payload
                if method == 'GET':
                    if urlencoded:
                        auth += '?' + urlencoded
                elif method == 'POST':
                    swapPostParams = json.dumps(params)
                    urlencoded = swapPostParams
                    auth += swapPostParams
            else:
                nonce = str(self.nonce())
                auth = urlencoded
            signature = self.hmac(self.encode(auth), self.encode(self.secret))
            if method == 'GET':
                if urlencoded:
                    url += '?' + urlencoded
            elif method == 'POST':
                headers = {
                    'Content-Type': 'application/x-www-form-urlencoded',
                }
                if urlencoded:
                    body = urlencoded
            headers = {
                'ACCESS-KEY': self.apiKey,
                'ACCESS-SIGN': signature,
                'ACCESS-TIMESTAMP': nonce,
            }
        else:
            if urlencoded:
                url += '?' + urlencoded
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, statusCode, statusText, url, method, responseHeaders, responseBody, response, requestHeaders, requestBody):
        if not response:
            return  # fall back to default error handler
        code = self.safe_string(response, 'code')
        if (code == '0') or (code == '200'):
            return  # no error
        feedback = self.id + ' ' + responseBody
        if code is None:
            raise BadResponse(feedback)
        unknownError = [ExchangeError, feedback]
        ExceptionClass, message = self.safe_value(self.exceptions['exact'], code, unknownError)
        raise ExceptionClass(message)
